/* 
 *  Art Studio Picture Viewer (c) Copyright, Kevin Thacker 2003
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#define WIN32_LEAN_AND_MEAN
#define WIN32_EXTRA_LEAN
#include <windows.h>
#include <commctrl.h>
#include <commdlg.h>
#include "image.h"
#include "artstud.h"
#include "types.h"
#include <stdio.h>
#include <stdlib.h>
#include "resource.h"

#define APPLICATION_CLASS_NAME "PictureViewerClass"

#ifndef abs
#define abs(_x_) ((_x_<0) ? -_x_ : _x_)
#endif

struct Image
{
	BITMAPINFO	*pBitmapInfo;
	char *pPixelData;
	unsigned long Pitch;
};

static struct Image *image;

void	Image_SetPaletteEntry(struct Image *image, const unsigned long Index, const char Red, const char Green, const char Blue)
{
	if (image->pBitmapInfo->bmiHeader.biBitCount<=8)
	{
		unsigned long MaxIndex;

		MaxIndex = (1<<image->pBitmapInfo->bmiHeader.biBitCount);

		if (Index<MaxIndex)
		{
			RGBQUAD *pRGBQuad;

			pRGBQuad = &image->pBitmapInfo->bmiColors[Index];

			pRGBQuad->rgbRed = Red;
			pRGBQuad->rgbGreen = Green;
			pRGBQuad->rgbBlue = Blue;
			pRGBQuad->rgbReserved = 0;
		}
	}
}

char	*Image_GetLineStart(struct Image *image, const unsigned long LineIndex)
{
	unsigned long ImageHeight;
	unsigned long Line;

	/* ensure line is within range */
	ImageHeight = abs(image->pBitmapInfo->bmiHeader.biHeight);

	if (LineIndex>ImageHeight)
		Line = ImageHeight-1;
	else
		Line = LineIndex;
		
	return (char *)image->pPixelData + (image->Pitch*(ImageHeight-Line-1));
}

void Image_Free(struct Image *image)
{
	if (image->pBitmapInfo!=NULL)
	{
		free(image->pBitmapInfo);
		image->pBitmapInfo = NULL;
	}

	if (image->pPixelData!=NULL)
	{
		free(image->pPixelData);
		image->pPixelData = NULL;
	}
}

int Image_Alloc(struct Image *image, const unsigned long Width, const unsigned long Height, const unsigned long BPP)
{
	unsigned long Pitch;
	unsigned long AllocLength;

	image->pBitmapInfo = NULL;
	image->pPixelData = NULL;

	/* size of the default structure */
	AllocLength = sizeof(BITMAPINFO);
	if (BPP<16)
	{
		/* size of additional colour table, excluding the single colour entry
		defined in the default structure */
		AllocLength = (sizeof(RGBQUAD)*(1<<BPP))-1;
	}

	image->pBitmapInfo = (BITMAPINFO *)malloc(AllocLength);

	if (image->pBitmapInfo!=NULL)
	{
		image->pBitmapInfo->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
		image->pBitmapInfo->bmiHeader.biWidth = Width;
		image->pBitmapInfo->bmiHeader.biHeight = Height; 
		image->pBitmapInfo->bmiHeader.biPlanes = 1;
		image->pBitmapInfo->bmiHeader.biBitCount = BPP;
		image->pBitmapInfo->bmiHeader.biCompression = BI_RGB;
		image->pBitmapInfo->bmiHeader.biSizeImage = 0;
		image->pBitmapInfo->bmiHeader.biXPelsPerMeter= 0;
		image->pBitmapInfo->bmiHeader.biYPelsPerMeter= 0;
		image->pBitmapInfo->bmiHeader.biClrImportant = 16;
		image->pBitmapInfo->bmiHeader.biClrUsed = 16;

		Pitch = (Width*image->pBitmapInfo->bmiHeader.biBitCount+7)>>3;

		/* ensure the length of the line in pixels is exactly divisible by 4 */
		Pitch = Pitch + ((4-(Pitch & 0x03))&0x03);

		image->Pitch = Pitch;
	
		image->pPixelData = (char *)malloc(Pitch*Height);

		if (image->pPixelData!=NULL)
		{
			return TRUE;
		}
	}

	return FALSE;
}

void	LoadImageFile(const char *pFilename)
{
	if (ArtStudioViewer_SetScreen(pFilename))
	{
		int Width, Height;

		ArtStudioViewer_GetDimensions(&Width, &Height);

		image = malloc(sizeof(struct Image));

		if (image!=NULL)
		{
			if (Image_Alloc(image, Width, Height, 8))
			{
				ArtStudioViewer_PutToImage(image);
			}
		}
	}
}

void Invalidate(HWND hwnd)
{
	RECT ClientRect;

	GetClientRect(hwnd, &ClientRect);

	InvalidateRect(hwnd, &ClientRect, FALSE);
}


LRESULT CALLBACK WindowProc( HWND hwnd,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_ERASEBKGND:
			return 1;

		case WM_COMMAND:
		{
			WORD nID = LOWORD(wParam);

			switch (nID)
			{
				case ID_FILE_OPEN:
				{
					char FilenameBuffer[MAX_PATH+1];
					OPENFILENAME OpenFilename;

					OpenFilename.lStructSize = sizeof(OPENFILENAME);
					OpenFilename.Flags = OFN_EXPLORER;
					OpenFilename.hwndOwner = hwnd;
					OpenFilename.hInstance = NULL;
					OpenFilename.lpstrFilter = "Art Studio Screen File (*.scr)\0*.scr\0All files (*.*)\0*.*\0\0";
					OpenFilename.lpstrCustomFilter = NULL;
					OpenFilename.nMaxCustFilter = 0;
					OpenFilename.nFilterIndex = 1;
					OpenFilename.lpstrFile = FilenameBuffer;
					OpenFilename.nMaxFile = sizeof(FilenameBuffer);
					OpenFilename.lpstrFileTitle = NULL;
					OpenFilename.nMaxFileTitle = 0;
					OpenFilename.lpstrInitialDir = NULL;
					OpenFilename.lpstrTitle = "Open";
					OpenFilename.nFileOffset = 0;
					OpenFilename.nFileExtension = 0;
					OpenFilename.lpstrDefExt = NULL;
					OpenFilename.lCustData = 0;
					OpenFilename.lpfnHook = NULL;
					OpenFilename.lpTemplateName = NULL;

					FilenameBuffer[0] = '\0';

					if (GetOpenFileName(&OpenFilename))
					{
						LoadImageFile(OpenFilename.lpstrFile);

						Invalidate(hwnd);

					}
				}
				break;

				case ID_VIEW_MODE_MODE0:
				{
					ArtStudioViewer_SetMode(0);
					ArtStudioViewer_PutToImage(image);

					Invalidate(hwnd);
				}
				break;

				case ID_VIEW_MODE_MODE1:
				{
					ArtStudioViewer_SetMode(1);
					ArtStudioViewer_PutToImage(image);

					Invalidate(hwnd);				
				}
				break;

				case ID_VIEW_MODE_MODE2:
				{
					ArtStudioViewer_SetMode(2);
					ArtStudioViewer_PutToImage(image);

					Invalidate(hwnd);
				}
				break;

				case ID_FILE_EXIT:
				{
					/* close the window */
					DestroyWindow(hwnd);
				}
				break;

				default:
					break;
			}
		}
		break;


		case WM_PAINT:
		{
			PAINTSTRUCT PaintStruct;
			HDC hDC;
			RECT ClientRect;

			GetClientRect(hwnd,&ClientRect);

			hDC = BeginPaint(hwnd, &PaintStruct);

			if (hDC!=0)
			{
				if (image!=NULL)
				{
					// stretch the image into the window
					StretchDIBits(hDC,
						ClientRect.left,
						ClientRect.top, 
						ClientRect.right-ClientRect.left,
						ClientRect.bottom-ClientRect.top,
						0,
						0,
						image->pBitmapInfo->bmiHeader.biWidth,
						abs(image->pBitmapInfo->bmiHeader.biHeight),
						image->pPixelData,
						image->pBitmapInfo,
						DIB_RGB_COLORS,
						SRCCOPY);
				}
				else
				{
					HBRUSH hBlackBrush = CreateSolidBrush(RGB(0,0,0));
					FillRect(hDC, &ClientRect, hBlackBrush);
				}

				EndPaint(hwnd, &PaintStruct);
			}
		}
		break;

		case WM_DESTROY:
		{
			/* quit */
			PostQuitMessage(0);
		}
		break;

	}

	return DefWindowProc(hwnd, uMsg, wParam, lParam);
}
 
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nShowCmd)
{
	MSG Msg;
	BOOL bQuit;
	int nResult;
	WNDCLASS WindowClass;

	ArtStudioViewer_Init();

	image = NULL;

	InitCommonControls();

	WindowClass.cbClsExtra = 0;
	WindowClass.cbWndExtra = 0;
	WindowClass.hbrBackground = (HBRUSH)NULL;	/* no background */
	WindowClass.hCursor = (HCURSOR)LoadCursor(NULL,IDC_ARROW); /* standard pointer cursor */
	WindowClass.hIcon = (HICON)NULL; /* no icon */
	WindowClass.hInstance = hInstance;
	WindowClass.lpszClassName = APPLICATION_CLASS_NAME;
	WindowClass.lpszMenuName = NULL;
	WindowClass.style = CS_HREDRAW|CS_VREDRAW|CS_CLASSDC;
	WindowClass.lpfnWndProc = WindowProc;

	if (RegisterClass(&WindowClass))
	{

		HMENU hMenu;
		HWND hWnd;

		hMenu = LoadMenu(hInstance, MAKEINTRESOURCE(IDR_MENU));

		hWnd = CreateWindow(WindowClass.lpszClassName, 
			"Advanced Art Studio Picture Viewer",
			WS_OVERLAPPEDWINDOW|WS_VISIBLE,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			NULL,
			hMenu,
			hInstance,
			NULL,
			);

		if (hWnd)
		{
			ShowWindow(hWnd, SW_SHOW);


			/* process messages */
			nResult = 0;
			bQuit = FALSE;
			while (!bQuit)
			{
				/* check for any messages */
				if (PeekMessage(&Msg, hWnd, 0,0, PM_NOREMOVE))
				{
					/* one or messages are available */
					/* get the message */
					do
					{
						/* get and process the message */
						nResult = GetMessage(&Msg, hWnd, 0,0);

						/* if result is 0, then quit message was received. If result
						is -ve then an error has occured */
						if (nResult<=0)
							bQuit = TRUE;

						DispatchMessage(&Msg);
					}
					while ((nResult!=0) && (bQuit==FALSE));
				}
			}
		}
	}

	if (image)
	{
		Image_Free(image);
		free(image);
	}

	ArtStudioViewer_Free();

	return nResult;

}