/*
 *  (c) Kevin Thacker, 2002
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "aiff.h"

SampleHandler AIFF_SampleHandler=
{
	"Audio Interchange File Format (AIFF)",
	AIFF_Open,
	AIFF_Close,
	AIFF_ReadSample,
	AIFF_IsEOF,
};


unsigned long AIFF_ConvertFloat(unsigned char * buffer)
{
   unsigned long mantissa;
   unsigned long last = 0;
   unsigned char exp;
   unsigned long man;
   
   man = ((unsigned long *)(buffer+2))[0];

#ifdef LSB_FIRST
   mantissa = SwapEndianLong(man);
#else
   mantissa = man;
#endif
	exp = 30 - *(buffer+1);
   while (exp--)
   {
     last = mantissa;
     mantissa >>= 1;
   }
   if (last & 0x00000001) mantissa++;
   return(mantissa);
}

static CHUNK_FILE_READ ChunkFileRead;
static unsigned long EndOfFile;


unsigned long AIFF_IsEOF(SampleDescription *desc)
{
	return EndOfFile;
}


unsigned char AIFF_ReadSample(SampleDescription *desc)
{
	unsigned char Data[8];
	unsigned long BytesPerSample = GetBytesPerSample(desc);

	if (ChunkFile_ReadData(&ChunkFileRead, Data, GetBytesPerSample(desc))!=BytesPerSample)
	{
		EndOfFile = 1;
		return 0x080;
	}

	return ConvertData(Data, desc->Format);
}



void	AIFF_Close(SampleDescription *desc)
{
	ChunkFile_Close(&ChunkFileRead);
}


int AIFF_Open(const char *pFilename,SampleDescription *desc)
{
	int Status = SAMPLE_FILE_NOT_RECOGNISED;
	int bGotFormat = 0;
	unsigned long BitsPerSample;
	unsigned long NumChannels;

//	*ppBuffer = NULL;
//	*pLength = 0;

	if (ChunkFile_Open(&ChunkFileRead, pFilename))
	{
		/* FORM chunk? */
		if (ChunkFile_GetParentChunkID(&ChunkFileRead)== CHUNK_ID_FORM)
		{
			/* big endian */
			unsigned char Type[4];

			/* big endian */
			ChunkFile_SetEndianness(&ChunkFileRead, ENDIANNESS_BIG_ENDIAN);
		
			if (ChunkFile_ReadData(&ChunkFileRead, &Type[0], 4)==4)
			{
				if (
					(Type[0]=='A') ||
					(Type[1]=='I') ||
					(Type[2]=='F') ||
					(Type[3]=='F')
					)
				{
					if (ChunkFile_FindChunk(&ChunkFileRead, CHUNK_ID_COMM))
					{
						unsigned long ChunkLength;

						ChunkFile_OpenChunk(&ChunkFileRead);

						ChunkLength = ChunkFile_GetParentChunkLength(&ChunkFileRead);

						{
							COMM_CHUNK Comm;

							if (ChunkFile_ReadData(&ChunkFileRead, (unsigned char *)&Comm,sizeof(COMM_CHUNK))==sizeof(COMM_CHUNK))
							{
								desc->Rate = AIFF_ConvertFloat((unsigned char *)&Comm.SampleRate);

#ifdef LSB_FIRST
								BitsPerSample = SwapEndianWord(Comm.BitsPerSample);
#else
								BitsPerSample = Comm.BitsPerSample;
#endif
#ifdef LSB_FIRST
								NumChannels = SwapEndianWord(Comm.NumberOfChannels);
#else
								NumChannels = Comm.NumberOfChannels;
#endif
								switch (BitsPerSample)
								{
									case 8:
									{
										if (NumChannels==1)
										{
											desc->Format = SAMPLE_FORMAT_8MS;
										}
										else
										{
											desc->Format = SAMPLE_FORMAT_8SS;
										}
									}
									break;

									case 16:
									{
										if (NumChannels==1)
										{
											desc->Format = SAMPLE_FORMAT_16MS_BE;
										}
										else
										{
											desc->Format = SAMPLE_FORMAT_16SS_BE;
										}
									}
									break;
								}

								bGotFormat = 1;
							}
						}

						ChunkFile_CloseChunk(&ChunkFileRead);
					}

					if (bGotFormat)
					{
						if (ChunkFile_FindChunk(&ChunkFileRead, CHUNK_ID_SSND))
						{
//							unsigned char *pBuffer;
							unsigned long ChunkLength;
							
							ChunkFile_OpenChunk(&ChunkFileRead);
							
							ChunkLength = ChunkFile_GetParentChunkLength(&ChunkFileRead);

							desc->LengthInSamples = (ChunkLength<<3)/(BitsPerSample*NumChannels);

							SampleDescription_InitDuration(desc);

							EndOfFile = 0;

							if (desc->LengthInSamples==0)
							{
								Status = SAMPLE_FILE_NO_DATA;
							}
							else
							{
								Status = SAMPLE_FILE_OK;
							}
						}
					}
				}
			}
		}

		if (Status!=SAMPLE_FILE_OK)
		{
			ChunkFile_Close(&ChunkFileRead);
		}
	}

	return Status;
}

