/*
 *  (c) Kevin Thacker, 2002
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "bufffile.h"
#include <memory.h>

void	BufferedFile_Init(BufferedFile *pBufferedFile, size_t BufferLength)
{
	pBufferedFile->pBuffer = (unsigned char *)malloc(BufferLength);
	if (pBufferedFile->pBuffer!=NULL)
	{
		pBufferedFile->Capacity = BufferLength;
	}
	else
	{
		pBufferedFile->Capacity= 0;
	}
	pBufferedFile->LengthUsed = 0;
	pBufferedFile->ReadPosition = 0;
}

void	BufferedFile_Free(BufferedFile *pBufferedFile)
{
	if (pBufferedFile->pBuffer!=NULL)
	{
		free(pBufferedFile->pBuffer);
	}
	
	pBufferedFile->pBuffer = NULL;
	
	if (pBufferedFile->fh!=NULL)
	{
		fclose(pBufferedFile->fh);
		pBufferedFile->fh=NULL;
	}

	pBufferedFile->ReadPosition = 0;
}

int		BufferedFile_Open(BufferedFile *pBufferedFile, const char *pFilename)
{
	pBufferedFile->fh = fopen(pFilename,"rb");

	return (pBufferedFile->fh!=NULL);
}

size_t BufferedFile_GetPos(BufferedFile *pBufferedFile)
{
	return ftell(pBufferedFile->fh)-(pBufferedFile->LengthUsed-pBufferedFile->ReadPosition);
}


static void BufferedFile_BufferData(BufferedFile *pBufferedFile)
{
	if ((pBufferedFile->fh!=NULL) && (pBufferedFile->pBuffer!=NULL))
	{
		pBufferedFile->LengthUsed = fread(pBufferedFile->pBuffer, 1, pBufferedFile->Capacity, pBufferedFile->fh);
	}

	/* reset read position */
	pBufferedFile->ReadPosition = 0;
}

void BufferedFile_Seek(BufferedFile *pBufferedFile, size_t Offset, size_t Relative)
{
	if (pBufferedFile->fh!=NULL)
	{
		switch (Relative)
		{
			case SEEK_END:
			case SEEK_SET:
			{
				/* do seek */
				fseek(pBufferedFile->fh, Offset, Relative);		
				/* force buffer to be re-filled */
				pBufferedFile->ReadPosition = 0;
				pBufferedFile->LengthUsed = 0;
			}
			break;
		
			case SEEK_CUR:
			{

				if (Offset==0)
					return;

				if (Offset>0)
				{
					size_t BytesRemaining;

					/* number of bytes left to read before buffer must be filled again */
					/* this is also the number of bytes up to the current file position */
					BytesRemaining = pBufferedFile->LengthUsed-pBufferedFile->ReadPosition;
				
					if (Offset<BytesRemaining)
					{
						/* offset within read buffer; just update read position */
						pBufferedFile->ReadPosition+=Offset;
					}
					else
					{
						/* offset outside buffer, take into account number of bytes remaining in buffer */
						fseek(pBufferedFile->fh, Offset-BytesRemaining, SEEK_CUR);
						/* force buffer to be re-filled */
						pBufferedFile->ReadPosition = 0;
						pBufferedFile->LengthUsed = 0;
					}
				}
				else
				{
					Offset=-Offset;

					if (Offset<=pBufferedFile->ReadPosition)
					{
						/* offset within buffer */
						/* just update read position */
						pBufferedFile->ReadPosition-=Offset;
					}
					else
					{
						/* offset outside buffer, take into account number of bytes read so far */
						fseek(pBufferedFile->fh, -Offset-pBufferedFile->LengthUsed, SEEK_CUR);
						/* force buffer to be re-filled */
						pBufferedFile->ReadPosition = 0;
						pBufferedFile->LengthUsed = 0;
					}
				}
				
			}
			break;
		}
	}
}

// read some data from the buffer, if none left then buffer data from the file
//
// returns number of bytes read
// pData is pointer to input buffer, DataLength is length of buffer
size_t BufferedFile_ReadData(BufferedFile *pBufferedFile, void *pData, size_t DataLength)
{
	size_t DataCountRemaining;
	size_t DataCountRead;
	unsigned char *pDataPtr = (unsigned char *)pData;

	// setup size ready to transfer
	DataCountRemaining = DataLength;
	// initialise size read
	DataCountRead = 0;
	
	do
	{
		size_t CountRemaining;

		// calculate size of data remaining 
		CountRemaining = pBufferedFile->LengthUsed - pBufferedFile->ReadPosition;
	
		// no more data remaining?
		if (CountRemaining==0)
		{
			// attempt to buffer data from file
			BufferedFile_BufferData(pBufferedFile);
		
			// was any data read?
			if (pBufferedFile->LengthUsed==0)
			{
				// no data was read, force exit
				DataCountRemaining = 0;
			}
		}
		else
		{
			size_t MaxTransferSize;
			
			// is the number of bytes remaining to be read
			// less or equal to number of bytes in buffer?
			if (DataCountRemaining<=CountRemaining)
			{
				// limit to the size of data remaining
				MaxTransferSize = DataCountRemaining;
			}
			else
			{
				// limit to the amount of data remaining in buffer
				MaxTransferSize = CountRemaining;
			}

			// copy
			memcpy(pDataPtr, &pBufferedFile->pBuffer[pBufferedFile->ReadPosition],MaxTransferSize);

			// update read position in buffer
			pBufferedFile->ReadPosition += MaxTransferSize;
			// update count transfered
			DataCountRead+=MaxTransferSize;
			// update count remaining
			DataCountRemaining -= MaxTransferSize;
			// update pointer in output buffer
			pDataPtr+=MaxTransferSize;
		}
	}		
	while (DataCountRemaining!=0);

	// return actual size read
	return DataCountRead;
}