/*
 *  (c) Kevin Thacker, 2002
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "chunk.h"
#include <memory.h>

/**********************************************************************/

/* get length from chunk when length has been written with big endian */
const unsigned long Chunk_GetLengthBigEndian(CHUNK_HEADER *pChunk)
{
#ifdef LSB_FIRST
	return SwapEndianLong(pChunk->Length);
#else
	return pChunk->Length;
#endif
}

/**********************************************************************/

/* get length from chunk when length has been written with little endian */
const unsigned long Chunk_GetLengthLittleEndian(CHUNK_HEADER *pChunk)
{
#ifdef MSB_FIRST
	return SwapEndianLong(pChunk->Length);
#else
	return pChunk->Length;
#endif
}

/**********************************************************************/

const unsigned long Chunk_GetID(CHUNK_HEADER *pChunk)
{
#ifdef LSB_FIRST
	return SwapEndianLong(pChunk->ID);
#else
	return pChunk->ID;
#endif
}

/**********************************************************************/

const unsigned long ChunkFile_ReadData(CHUNK_FILE_READ *pChunkFileRead, unsigned char *pBuffer, const unsigned long Length)
{
	unsigned long BytesRemaining;
	unsigned long ReadLength;
	unsigned long BytesRead;
	unsigned long ReadOffset;

	ReadOffset = ChunkFile_GetParentChunk(pChunkFileRead)->ReadOffset;
	BytesRemaining = ChunkFile_GetParentChunkLength(pChunkFileRead)- ReadOffset;

	if (BytesRemaining==0)
		return 0;

	/* clip length to amount left in current chunk */
	ReadLength = Length;
	if (BytesRemaining<ReadLength)
	{
		ReadLength = BytesRemaining;
	}

//	BytesRead = fread(pBuffer, 1, ReadLength, pChunkFileRead->fh);

	BytesRead = BufferedFile_ReadData(&pChunkFileRead->bufferedFile, pBuffer, ReadLength);

	ReadOffset+=BytesRead;

	ChunkFile_GetParentChunk(pChunkFileRead)->ReadOffset = ReadOffset;

	return BytesRead;
}

/**********************************************************************/

void ChunkFile_SkipChunk(CHUNK_FILE_READ *pChunkFileRead)
{
	unsigned long ChunkLength;

	if (pChunkFileRead->Endianness==ENDIANNESS_LITTLE_ENDIAN)
	{	
		ChunkLength = Chunk_GetLengthLittleEndian(&pChunkFileRead->Current.Header);
	} 
	else
	{
		ChunkLength = Chunk_GetLengthBigEndian(&pChunkFileRead->Current.Header);
	}

	/*fseek(pChunkFileRead->fh, ChunkLength, SEEK_CUR); */
	BufferedFile_Seek(&pChunkFileRead->bufferedFile, ChunkLength, SEEK_CUR);

}

/**********************************************************************/
/* read a chunk header */
static int ChunkFile_ReadHeader(CHUNK_FILE_READ *pChunkFileRead, CHUNK_HEADER *pHeader)
{
	return (ChunkFile_ReadData(pChunkFileRead, (unsigned char *)pHeader, sizeof(CHUNK_HEADER))==sizeof(CHUNK_HEADER));
}

/**********************************************************************/

/* set the endianness that the chunk file was created with; this determines the endianness of the chunk length
in the chunk header */
void ChunkFile_SetEndianness(CHUNK_FILE_READ *pChunkFileRead, const int Endianness)
{
	/* set expected endianness */
	pChunkFileRead->Endianness = Endianness;
}

/**********************************************************************/
const unsigned long ChunkFile_GetCurrentChunkID(CHUNK_FILE_READ *pChunkFileRead)
{
	return Chunk_GetID(&pChunkFileRead->Current.Header);
}

/**********************************************************************/
const unsigned long ChunkFile_GetCurrentChunkLength(CHUNK_FILE_READ *pChunkFileRead)
{
	if (pChunkFileRead->Endianness==ENDIANNESS_LITTLE_ENDIAN)
	{
		return Chunk_GetLengthLittleEndian(&pChunkFileRead->Current.Header);
	}

	return Chunk_GetLengthBigEndian(&pChunkFileRead->Current.Header);
}

/**********************************************************************/
CHUNK_READ_DATA *ChunkFile_GetParentChunk(CHUNK_FILE_READ *pChunkFileRead)
{
	return &pChunkFileRead->Stack[pChunkFileRead->NumStackItems-1];
}

/**********************************************************************/
const unsigned long ChunkFile_GetParentChunkID(CHUNK_FILE_READ *pChunkFileRead)
{
	return Chunk_GetID(&ChunkFile_GetParentChunk(pChunkFileRead)->Header);
}

/**********************************************************************/
const unsigned long ChunkFile_GetParentChunkLength(CHUNK_FILE_READ *pChunkFileRead)
{
	if (pChunkFileRead->Endianness==ENDIANNESS_LITTLE_ENDIAN)
	{
		return Chunk_GetLengthLittleEndian(&ChunkFile_GetParentChunk(pChunkFileRead)->Header);
	}

	return Chunk_GetLengthBigEndian(&ChunkFile_GetParentChunk(pChunkFileRead)->Header);
}

/**********************************************************************/


/* open a chunk file e.g. WAV, IFF, AIFF */
int	ChunkFile_Open(CHUNK_FILE_READ *pChunkFileRead, const char *pFilename)
{
	int bSuccess = 1;

	/* initialise chunk stack */
	pChunkFileRead->NumStackItems = 0;

	/* attempt to open chunk file */
//	pChunkFileRead->fh = fopen(pFilename,"rb");

	BufferedFile_Init(&pChunkFileRead->bufferedFile,8192);

//	if (pChunkFileRead->fh!=NULL)
	if (BufferedFile_Open(&pChunkFileRead->bufferedFile, pFilename))
	{
		size_t FileSize;
		size_t CurrentPosition;

//		CurrentPosition = ftell(pChunkFileRead->fh);
//		fseek(pChunkFileRead->fh,0, SEEK_END);
//		FileSize = ftell(pChunkFileRead->fh);
//		fseek(pChunkFileRead->fh,CurrentPosition, SEEK_SET);
		CurrentPosition = BufferedFile_GetPos(&pChunkFileRead->bufferedFile);
		BufferedFile_Seek(&pChunkFileRead->bufferedFile,0, SEEK_END);
		FileSize = BufferedFile_GetPos(&pChunkFileRead->bufferedFile);
		BufferedFile_Seek(&pChunkFileRead->bufferedFile,CurrentPosition, SEEK_SET);

		pChunkFileRead->Current.ReadOffset = 0;
		pChunkFileRead->Current.Header.Length = FileSize;
		ChunkFile_OpenChunk(pChunkFileRead);

		/* attempt to read first chunk header */
		if (!ChunkFile_ReadHeader(pChunkFileRead, &pChunkFileRead->Current.Header))
		{
			/* failed, so close chunk file */
			ChunkFile_Close(pChunkFileRead);
			bSuccess = 0;
		}
		else
		{
			/* push onto stack */
			ChunkFile_OpenChunk(pChunkFileRead);
		}
	}

	/* return status of opened file */
	return bSuccess;
}

void ChunkFile_Close(CHUNK_FILE_READ *pChunkFileRead)
{
	BufferedFile_Free(&pChunkFileRead->bufferedFile);

//	if (pChunkFileRead->fh!=NULL)
//	{
//		/* close chunk file */
//		fclose(pChunkFileRead->fh);
//		pChunkFileRead->fh = NULL;
//	}
}

/* push a chunk onto the stack */
static void ChunkFile_PushChunk(CHUNK_FILE_READ *pChunkFileRead)
{
	if (pChunkFileRead->NumStackItems!=CHUNK_MAX_STACK_ITEMS)
	{
		memcpy(&pChunkFileRead->Stack[pChunkFileRead->NumStackItems], &pChunkFileRead->Current, sizeof(CHUNK_READ_DATA));
		pChunkFileRead->NumStackItems++;
	}
}

/* pop a chunk from the stack */
static void ChunkFile_PopChunk(CHUNK_FILE_READ *pChunkFileRead)
{
	if (pChunkFileRead->NumStackItems!=0)
	{
		pChunkFileRead->NumStackItems--;
		memcpy(&pChunkFileRead->Current, &pChunkFileRead->Stack[pChunkFileRead->NumStackItems], sizeof(CHUNK_READ_DATA));
	}
}


void ChunkFile_OpenChunk(CHUNK_FILE_READ *pChunkFileRead)
{
	ChunkFile_PushChunk(pChunkFileRead);
}

void ChunkFile_CloseChunk(CHUNK_FILE_READ *pChunkFileRead)
{
	unsigned long ChunkLength;

	ChunkLength = ChunkFile_GetParentChunkLength(pChunkFileRead);

	/* read to end of chunk? */
	if (ChunkFile_GetParentChunk(pChunkFileRead)->ReadOffset!=ChunkLength)
	{
		/* no */
		unsigned long SeekAmount;

		/* amount to end of chunk */
		SeekAmount = ChunkLength-ChunkFile_GetParentChunk(pChunkFileRead)->ReadOffset;

		/* seek forwards to end of chunk */
//		fseek(pChunkFileRead->fh, SeekAmount, SEEK_CUR);
		BufferedFile_Seek(&pChunkFileRead->bufferedFile, SeekAmount, SEEK_CUR);
	}
	
	/* pop old chunk from stack */
	ChunkFile_PopChunk(pChunkFileRead);
	/* update read position in this chunk */
	ChunkFile_GetParentChunk(pChunkFileRead)->ReadOffset+=ChunkLength;
}


int ChunkFile_FindChunk(CHUNK_FILE_READ *pChunkFileRead, unsigned long ChunkID)
{
	do
	{
		if (ChunkFile_ReadHeader(pChunkFileRead, &pChunkFileRead->Current.Header))
		{
			/* id matches? */
			if (Chunk_GetID(&pChunkFileRead->Current.Header)==ChunkID)
			{
				/* yes, found chunk */
				return 1;
			}

			/* no, skip chunk */
			ChunkFile_SkipChunk(pChunkFileRead);
		}
	}
	while (ChunkFile_GetParentChunkLength(pChunkFileRead)<ChunkFile_GetParentChunk(pChunkFileRead)->ReadOffset);

	return 0;
}
