/*
 *  (c) Kevin Thacker, 2002
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "csw.h"
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <string.h>
#include "bufffile.h"

SampleHandler CSW_SampleHandler=
{
	"Compressed Square Wave (CSW)",
	CSW_Open,
	CSW_Close,
	CSW_ReadSample,
	CSW_IsEOF,
};


static BufferedFile bufferedFile;
//static FILE *fh;

typedef struct CSW_Decoder
{
	unsigned long CurrentLevel; 
	unsigned long NumSamples;
	int bEndOfFile;
} CSW_Decoder;

static CSW_Decoder decoder;


static void CSW_UpdateDecoder(void)
{
	if (decoder.bEndOfFile)
	{
		decoder.NumSamples = 0;
	}
	else
	{
		unsigned char CSW_bytecode;

		if (BufferedFile_ReadData(&bufferedFile, &CSW_bytecode, sizeof(unsigned char))==sizeof(unsigned char))
		{
			if (CSW_bytecode!=0)
			{
				decoder.NumSamples = CSW_bytecode&0x0ff;
			}
			else
			{
				unsigned char CSW_longcode[4];

				/* CSW_bytecode==0, read next 4 bytes */
				if (BufferedFile_ReadData(&bufferedFile,&CSW_longcode[0], sizeof(unsigned char))==sizeof(unsigned char))
				{
					if (BufferedFile_ReadData(&bufferedFile,&CSW_longcode[1], sizeof(unsigned char))==sizeof(unsigned char))
					{
						if (BufferedFile_ReadData(&bufferedFile,&CSW_longcode[2],sizeof(unsigned char))==sizeof(unsigned char))
						{
							if (BufferedFile_ReadData(&bufferedFile,&CSW_longcode[3], sizeof(unsigned char))==sizeof(unsigned char))
							{
								unsigned long Count;

								Count =
									(
									(CSW_longcode[0]&0x0ff) |
									((CSW_longcode[1]&0x0ff)<<8) |
									((CSW_longcode[2]&0x0ff)<<16) |
									((CSW_longcode[3]&0x0ff)<<24) 
									);														

								decoder.NumSamples = Count;
							}
							else
							{
								decoder.bEndOfFile = 1;
							}
						}
						else
						{
							decoder.bEndOfFile = 1;
						}

					}
					else
					{
						decoder.bEndOfFile = 1;
					}
				}
				else
				{
					decoder.bEndOfFile = 1;
				}
			}
		}
		else
		{
			decoder.bEndOfFile = 1;
		}
	}
}

unsigned char CSW_ReadSample(SampleDescription *desc)
{
	if (decoder.NumSamples==0)
	{
		/* swap level */
		if (decoder.CurrentLevel == SAMPLE_VOL_MAX)
			decoder.CurrentLevel = SAMPLE_VOL_MIN;
		else
			decoder.CurrentLevel = SAMPLE_VOL_MAX;

		CSW_UpdateDecoder();
	}
	else
	{
		decoder.NumSamples--;
	}

	return (unsigned char)decoder.CurrentLevel;
}

unsigned long CSW_IsEOF(SampleDescription *desc)
{
	return decoder.bEndOfFile;
}

int		CSW_Open(const char *pFilename, SampleDescription *desc)
{
	int Status = SAMPLE_FILE_NOT_RECOGNISED;

	/* open file */

	BufferedFile_Init(&bufferedFile, 8192);

	if (BufferedFile_Open(&bufferedFile, pFilename))
	{
		CSW_FILE_HEADER	Header;

		/* attempt to read CSW header */
		if (BufferedFile_ReadData(&bufferedFile, &Header, sizeof(CSW_FILE_HEADER))==sizeof(CSW_FILE_HEADER))
		{
			/* test for signature */
			if (memcmp(&Header.Ident[0], CSW_IDENT,0x017)==0)
			{
				if (Header.MajorFormatRevision!=1)
				{
					/* unsupported version */
					Status = SAMPLE_FILE_VERSION_UNSUPPORTED;
				}
				else
				if ((Header.MinorFormatRevision!=0) && (Header.MinorFormatRevision!=1))
				{
					Status = SAMPLE_FILE_VERSION_UNSUPPORTED;
				}
				else
				{
					if (Header.CompressionType!=1)
					{
						/* unsupported data format */
						Status = SAMPLE_FILE_DATA_FORMAT_UNSUPPORTED;
					}
					else
					{
						unsigned long LengthInSamples = 0;
						size_t SampleDataOffset;
						unsigned char CSW_bytecode;
						int bEndOfFile = 0;
//						unsigned char *pBufferPtr;
//						unsigned char *pBuffer;

						/* setup sample description */
						desc->Rate = (Header.SampleRateHzL & 0x0ff) | ((Header.SampleRateHzH & 0x0ff)<<8);

						/* get the current position in the file */
						SampleDataOffset = BufferedFile_GetPos(&bufferedFile);

						/* calculate length in samples */

						bEndOfFile = 0;
						while (!bEndOfFile)
						{
							if (BufferedFile_ReadData(&bufferedFile, &CSW_bytecode, sizeof(unsigned char))==sizeof(unsigned char))
							{
								if (CSW_bytecode!=0)
								{
									LengthInSamples+=(CSW_bytecode&0x0ff);
								}
								else
								{
									unsigned char CSW_longcode[4];

									/* CSW_bytecode==0, read next 4 bytes */
									if (BufferedFile_ReadData(&bufferedFile, &CSW_longcode[0], sizeof(unsigned char))==sizeof(unsigned char))
									{
										if (BufferedFile_ReadData(&bufferedFile, &CSW_longcode[1], sizeof(unsigned char))==sizeof(unsigned char))
										{
											if (BufferedFile_ReadData(&bufferedFile, &CSW_longcode[2], sizeof(unsigned char))==sizeof(unsigned char))
											{
												if (BufferedFile_ReadData(&bufferedFile, &CSW_longcode[3], sizeof(unsigned char))==sizeof(unsigned char))
												{
													LengthInSamples += 
														(
														(CSW_longcode[0]&0x0ff) |
														((CSW_longcode[1]&0x0ff)<<8) |
														((CSW_longcode[2]&0x0ff)<<16) |
														((CSW_longcode[3]&0x0ff)<<24) 
														);														
												}
												else
												{
													bEndOfFile = 1;
												}
											}
											else
											{
												bEndOfFile = 1;
											}

										}
										else
										{
											bEndOfFile = 1;
										}
									}
									else
									{
										bEndOfFile = 1;
									}
								}
							}
							else
							{
								bEndOfFile = 1;
							}
						}

						/* seek back to start of data */
						BufferedFile_Seek(&bufferedFile, SampleDataOffset, SEEK_SET);

						desc->LengthInSamples = LengthInSamples;

						/* length of sample in samples now known */
						SampleDescription_InitDuration(desc);

						if (LengthInSamples==0)
						{
							Status = SAMPLE_FILE_NO_DATA;
						}
						else
						{

							if (Header.MinorFormatRevision!=0)
							{
								/* set initial level */
								if (Header.Flags & CSW_POLARITY_MASK)
								{
									decoder.CurrentLevel= (unsigned char)SAMPLE_VOL_MAX;
								}
								else
								{
									decoder.CurrentLevel = (unsigned char)SAMPLE_VOL_MIN;
								}
							}
							else
							{
								decoder.CurrentLevel = (unsigned char)SAMPLE_VOL_MIN;
							}

							decoder.bEndOfFile = 0;

							CSW_UpdateDecoder();

							Status = SAMPLE_FILE_OK;
						}
					}
				}
			}
		}

		if (Status!=SAMPLE_FILE_OK)
		{
			/* close file */
			BufferedFile_Free(&bufferedFile);

//				fclose(fh);
//				fh = NULL;
		}
	}

	return Status;
}

void CSW_Close(SampleDescription *desc)
{
	BufferedFile_Free(&bufferedFile);

//	if (fh!=NULL)
//	{
//		fclose(fh);
//		fh = NULL;
//	}
}



