/*
 *  (c) Kevin Thacker, 2002
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "sample.h"
#include <math.h>

#define SECONDS_PER_MINUTE 60
#define MINUTES_PER_HOUR 60

#include "csw.h"
#include "wav.h"
#include "voc.h"
#include "iff.h"
#include "aiff.h"

SampleHandler *supportedFileFormats[]=
{
	&CSW_SampleHandler,
	&VOC_SampleHandler,
	&WAV_SampleHandler,
	&IFF_SampleHandler,
	&AIFF_SampleHandler,
	NULL,
};


int SampleFile_Open(const char *pFilename,SampleDescription *desc)
{
	int Status;
	SampleHandler **current;

	current = supportedFileFormats;

	while ((*current)!=NULL)
	{
		if ((*current)->pOpen)
		{
			Status = (*current)->pOpen(pFilename, desc);

			desc->handler = (*(*current));

			if (Status!=SAMPLE_FILE_NOT_RECOGNISED)
				return Status;
		}
		current++;
	}

	return SAMPLE_FILE_NOT_RECOGNISED;
}

void	SampleFile_Close(SampleDescription *desc)
{
	if (desc->handler.pClose)
		desc->handler.pClose(desc);
}


void	SampleDescription_InitDuration(SampleDescription *desc)
{
	/* the rate is the number of samples per second */
	/* e.g. 22050hz is 22050 samples per second */

	/* calculate number of seconds per sample */
	float	SecondsPerSample = 1.0f/(float)desc->Rate;
	/* total length of sample in seconds */
	float	LengthInSeconds = SecondsPerSample*desc->LengthInSamples;
	float	SecondsRemaining = LengthInSeconds;

	desc->duration.Hours = (unsigned long)floor(SecondsRemaining/(float)(SECONDS_PER_MINUTE*MINUTES_PER_HOUR));
	SecondsRemaining -= desc->duration.Hours*(SECONDS_PER_MINUTE*MINUTES_PER_HOUR);
	desc->duration.Minutes = (unsigned long)floor(SecondsRemaining/(float)(SECONDS_PER_MINUTE));
	SecondsRemaining -= desc->duration.Minutes*(SECONDS_PER_MINUTE);
	desc->duration.Seconds = (unsigned long)floor(SecondsRemaining);
	desc->duration.FractionalSeconds = SecondsRemaining*100;
}

int GetBytesPerSample(SampleDescription *desc)
{
	switch (desc->Format)
	{
		case SAMPLE_FORMAT_8MU:	/* 8 bit mono unsigned */
		case SAMPLE_FORMAT_8MS:	/* 8 bit mono signed */
			return 1;

		case SAMPLE_FORMAT_8SU:	/* 8 bit stereo unsigned */
		case SAMPLE_FORMAT_8SS:	/* 8 bit stereo signed */
		case SAMPLE_FORMAT_16MU_LE:	/* 16-bit mono unsigned, little endian */
		case SAMPLE_FORMAT_16MS_LE:	/* 16-bit mono signed, little endian */
		case SAMPLE_FORMAT_16MU_BE:	/* 16-bit mono unsigned, big endian */
		case SAMPLE_FORMAT_16MS_BE:	/* 16-bit mono signed, big endian */
			return 2;

		case SAMPLE_FORMAT_16SU_LE:	/* 16-bit stereo unsigned, little endian */
		case SAMPLE_FORMAT_16SS_LE:	/* 16-bit stereo signed, little endian */
		case SAMPLE_FORMAT_16SU_BE:	/* 16-bit stereo unsigned, big endian*/
		case SAMPLE_FORMAT_16SS_BE:	/* 16-bit stereo signed, big endian */
			return 4;
	}

	return 0;
}




unsigned char ConvertData(unsigned char *pBufferPtr, unsigned long FormatType)
{
//	unsigned long i;

	switch (FormatType)
	{
		case SAMPLE_FORMAT_8MU:
			return (pBufferPtr[0]^0x080);

		case SAMPLE_FORMAT_8MS:
			return pBufferPtr[0];

		case SAMPLE_FORMAT_8SU:
		{
			unsigned long result;

			result = (pBufferPtr[0]+ pBufferPtr[1])>>1;
			
			return (result^0x080);
		}
		break;

		case SAMPLE_FORMAT_8SS:
		{
			unsigned long result;

			result = (pBufferPtr[0]+ pBufferPtr[1])>>1;
			
			return result;
		}
		break;

		/* 16-bit mono signed, little endian */
		case SAMPLE_FORMAT_16MS_LE:
		{
//			signed short result;
//
//			result = (pBufferPtr[0]&0x0ff)|((pBufferPtr[1]&0x0ff)<<8);
//			return (result>>8);
			return pBufferPtr[1];
		}
		break;

		/* 16-bit stereo signed, little endian */
		case SAMPLE_FORMAT_16SS_LE:
		{
			signed short result1,result2,result;

			result1 = (pBufferPtr[0]&0x0ff)|((pBufferPtr[1]&0x0ff)<<8);
			result2 = (pBufferPtr[2]&0x0ff)|((pBufferPtr[3]&0x0ff)<<8);
			result = (result1+result2)>>1;
			return (result>>8);
		}
		break;

		/* 16-bit mono signed, big endian */
		case SAMPLE_FORMAT_16MS_BE:
		{
//			signed short result;
//
//			result = (pBufferPtr[1]&0x0ff)|((pBufferPtr[0]&0x0ff)<<8);
//			return (result>>8);

			return pBufferPtr[0];
		}
		break;

		/* 16-bit stereo signed, big endian */
		case SAMPLE_FORMAT_16SS_BE:
		{
			signed short result1,result2,result;

			result1 = (pBufferPtr[1]&0x0ff)|((pBufferPtr[0]&0x0ff)<<8);
			result2 = (pBufferPtr[3]&0x0ff)|((pBufferPtr[2]&0x0ff)<<8);
			result = (result1+result2)>>1;
			return (result>>8);
		}
		break;

		/* 16-bit mono unsigned, little endian */
		case SAMPLE_FORMAT_16MU_LE:
		{
//			signed short result;
//
//			result = (pBufferPtr[0]&0x0ff)|((pBufferPtr[1]&0x0ff)<<8);
//			return ((result>>8)^0x080);

			return pBufferPtr[1]^0x080;
		}
		break;

		/* 16-bit stereo unsigned, little endian */
		case SAMPLE_FORMAT_16SU_LE:
		{
			signed short result1,result2,result;

			result1 = (pBufferPtr[0]&0x0ff)|((pBufferPtr[1]&0x0ff)<<8);
			result2 = (pBufferPtr[2]&0x0ff)|((pBufferPtr[3]&0x0ff)<<8);
			result = (result1+result2)>>1;
			return ((result>>8)^0x080);
		}
		break;

		/* 16-bit mono unsigned, big endian */
		case SAMPLE_FORMAT_16MU_BE:
		{
//			signed short result;
//
//			result = (pBufferPtr[1]&0x0ff)|((pBufferPtr[0]&0x0ff)<<8);
//			return ((result>>8)^0x080);

			return pBufferPtr[0]^0x080;
		}
		break;

		/* 16-bit stereo unsigned, big endian */
		case SAMPLE_FORMAT_16SU_BE:
		{
			signed short result1,result2,result;

			result1 = (pBufferPtr[1]&0x0ff)|((pBufferPtr[0]&0x0ff)<<8);
			result2 = (pBufferPtr[3]&0x0ff)|((pBufferPtr[2]&0x0ff)<<8);
			result = (result1+result2)>>1;
			return ((result>>8)^0x080);
		}
		break;

		default:
			break;
	}

	return 0;
}