/*
 *  (c) Kevin Thacker, 2002
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "voc.h"
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <string.h>
#include "bufffile.h"

SampleHandler VOC_SampleHandler=
{
	"Creative Voice File (VOC)",
	VOC_Open,
	VOC_Close,
	VOC_ReadSample,
	VOC_IsEOF,
};



typedef struct VOC_CURRENT_STATE
{
	unsigned long EndOfFile;
//	FILE *fh;
	BufferedFile bufferedFile;
	unsigned long OffsetInBlock;
	unsigned long BlockLength;
	unsigned long MasterRate;
	unsigned long SampleRate;
	unsigned long BitsPerSample;
	unsigned long NumChannels;
	unsigned long Format;
} VOC_CURRENT_STATE;

typedef struct VOC_REPEAT_DATA
{
	/* offset of block following repeat */
	size_t Offset;
	/* count remaining */
	unsigned short CountRemaining;
} VOC_REPEAT_DATA;

/* maximum of 32 nested loops */
#define MAX_ENTRIES 32

typedef struct VOC_REPEAT_STACK
{
	int NumEntries;
	VOC_REPEAT_DATA Stack[MAX_ENTRIES];
} VOC_REPEAT_STACK;

/*-----------------------------------------------------------------------------*/

static VOC_REPEAT_STACK RepeatStack;
static VOC_REPEAT_DATA CurrentRepeat;
static VOC_CURRENT_STATE CurrentState;


/*-----------------------------------------------------------------------------*/
/* init stack */
void VocRepeatStack_Init(VOC_REPEAT_STACK *pStack)
{
	pStack->NumEntries = 0;
}

/*-----------------------------------------------------------------------------*/
/* push a item onto the stack */
int VocRepeatStack_Push(VOC_REPEAT_STACK *pStack, VOC_REPEAT_DATA *pRepeatData)
{
	if (pStack->NumEntries>=MAX_ENTRIES)
	{
		/* failed to push onto stack */
		return 0;
	}

	/* push onto stack */
	memcpy(&pStack->Stack[pStack->NumEntries], pRepeatData, sizeof(VOC_REPEAT_DATA));
	pStack->NumEntries++;

	/* succeeded to push onto stack */
	return 1;
}

/*-----------------------------------------------------------------------------*/
/* pop a item from the stack */
int VocRepeatStack_Pop(VOC_REPEAT_STACK *pStack, VOC_REPEAT_DATA *pRepeatData)
{
	if (pStack->NumEntries==0)
		return 0;

	pStack->NumEntries--;
	memcpy(pRepeatData, &pStack->Stack[pStack->NumEntries], sizeof(VOC_REPEAT_DATA));

	return 1;
}

/*-----------------------------------------------------------------------------*/

/* encountered a 'end repeat' */
static int UpdateRepeat(BufferedFile *bufferedFile, VOC_REPEAT_DATA *pCurrentRepeat, VOC_REPEAT_STACK *pRepeatStack)
{
	/* if count remaining is 0, then there is an error, as there is no repeat defined */
	if (pCurrentRepeat->CountRemaining==0)
		return 0;

	/* decrement count */
	pCurrentRepeat->CountRemaining--;

	/* count over? */
	if (pCurrentRepeat->CountRemaining!=0)
	{
		/* no, seek back to block after repeat */
		BufferedFile_Seek(bufferedFile, pCurrentRepeat->Offset, SEEK_SET);
	}
	else
	{
		/* yes, pop the next repeat from the stack; allows nested repeats */
		if (!VocRepeatStack_Pop(pRepeatStack, pCurrentRepeat))
			return 0;
	}

	return 1;
}
/*-----------------------------------------------------------------------------*/
static int SetupRepeat(BufferedFile *bufferedFile, VOC_REPEAT_DATA *pRepeatData, VOC_REPEAT_STACK *pRepeatStack, unsigned long Count)
{
	if (Count!=0)
	{
		if (VocRepeatStack_Push(pRepeatStack, pRepeatData))
		{
			/* store count */
			pRepeatData->CountRemaining = Count;
			/* store offset */
			pRepeatData->Offset = BufferedFile_GetPos(bufferedFile);

			return 1;
		}

		return 0;
	}

	return 1;
}

static unsigned long GetBlockLength(VOC_BLOCK_HEADER *pBlockHeader)
{
	return
				(
				(pBlockHeader->LengthLow&0x0ff) |
				((pBlockHeader->LengthMid&0x0ff)<<8) |
				((pBlockHeader->LengthHigh&0x0ff)<<16) 
				);
}

/*-----------------------------------------------------------------------------*/
static int HandleBlock(BufferedFile *bufferedFile, VOC_BLOCK_HEADER *pBlockHeader)
{
	switch (pBlockHeader->ID)
	{
		/* marker */
		case 0x04:
		{
			VOC_BLOCK4	 Block4;

			return (BufferedFile_ReadData(bufferedFile,&Block4, sizeof(VOC_BLOCK4))==sizeof(VOC_BLOCK4));
		}
		break;

		/* ascii */
		case 0x05:
		{
			unsigned long BlockLength;

			/* get block length */
			BlockLength = GetBlockLength(pBlockHeader);

			/* skip */
			BufferedFile_Seek(bufferedFile, BlockLength, SEEK_CUR);
		}
		break;


		/* Number of repetitions + 1
           
			 Count# may be 1 to FFFE for 0 - FFFD repetitions
             or FFFF for endless repetitions
		*/

		/* repeat */
		case 0x06:
		{	
			VOC_BLOCK6	 Block6;

			if (BufferedFile_ReadData(bufferedFile,&Block6, sizeof(VOC_BLOCK6))==sizeof(VOC_BLOCK6))
			{
				unsigned long RepeatCount;

				RepeatCount = (
						(Block6.RepeatCount[0]&0x0ff)|
						((Block6.RepeatCount[1]&0x0ff)<<8));

				return SetupRepeat(bufferedFile, &CurrentRepeat, &RepeatStack, RepeatCount);
			}
			else
				return 0;
		}
		break;

		/* end repeat */
		case 0x07:
		{
			return UpdateRepeat(bufferedFile, &CurrentRepeat, &RepeatStack);
		}
		break;
	}

	return 1;
}

static unsigned long UnpackFrequency(unsigned char PackedFrequency)
{
	return (1000000L/(256-(PackedFrequency & 0x0ff)));
}

/*-----------------------------------------------------------------------------*/

static void VOC_SetupFormat(void)
{
	switch (CurrentState.BitsPerSample)
	{
		case 8:
		{
			if (CurrentState.NumChannels==1)
			{
				CurrentState.Format = SAMPLE_FORMAT_8MS;
			}
			else
			{
				CurrentState.Format = SAMPLE_FORMAT_8SS;
			}
		}
		break;

		case 16:
		{
			if (CurrentState.NumChannels==1)
			{
				CurrentState.Format = SAMPLE_FORMAT_16MU_LE;
			}
			else
			{
				CurrentState.Format = SAMPLE_FORMAT_16SU_LE;
			}
		}
		break;
	}
}

static unsigned long CalculateLengthInSamples(BufferedFile *bufferedFile)
{
	VOC_BLOCK_HEADER Header;

	unsigned long NumSamples = 0;

	do
	{
		/* read header */
		if (BufferedFile_ReadData(bufferedFile,&Header, sizeof(VOC_BLOCK_HEADER))==sizeof(VOC_BLOCK_HEADER))
		{
			unsigned long BlockLength;

			/* get block length */
			BlockLength = GetBlockLength(&Header);

			switch (Header.ID)
			{
				/* terminator */
				case 0:
					break;

				/* sound data */
				case 1:
				{
					VOC_BLOCK1 Block1;

					if (BufferedFile_ReadData(bufferedFile,&Block1, sizeof(VOC_BLOCK1))==sizeof(VOC_BLOCK1))
					{
						unsigned long DataLength;

						CurrentState.SampleRate = UnpackFrequency(Block1.SampleRate); 						

						if (CurrentState.MasterRate==-1)
						{
							CurrentState.MasterRate = CurrentState.SampleRate;
						}

						if (Block1.CompressionType==0)
						{
							CurrentState.BitsPerSample = 8;
							CurrentState.NumChannels = 1;
						}


						DataLength = (BlockLength-sizeof(VOC_BLOCK1));

						NumSamples += (DataLength<<3)/(CurrentState.BitsPerSample*CurrentState.NumChannels);

						/* skip */
						BufferedFile_Seek(bufferedFile, DataLength, SEEK_CUR);
					}
				}
				break;

				/* continuation */
				case 2:
				{
					NumSamples += (BlockLength<<3)/(CurrentState.BitsPerSample*CurrentState.NumChannels);

					/* skip */
					BufferedFile_Seek(bufferedFile, BlockLength, SEEK_CUR);
				}
				break;

				/* silence */
				case 3:
				{
					VOC_BLOCK3 Block3;

					if (BufferedFile_ReadData(&CurrentState.bufferedFile,&Block3, sizeof(VOC_BLOCK3))==sizeof(VOC_BLOCK3))
					{



					}

				}
				break;

				/* control */
				case 4:
				case 5:
				case 6:
				case 7:
					HandleBlock(bufferedFile, &Header);
					break;

				/* extended */
				case 8:
				{


				}
				break;

				case 9:
				{
					VOC_BLOCK9 Block9;

					if (BufferedFile_ReadData(&CurrentState.bufferedFile,&Block9, sizeof(VOC_BLOCK9))==sizeof(VOC_BLOCK9))
					{
						unsigned long DataLength;

						CurrentState.SampleRate = (
							(Block9.SampleFrequency[0]&0x0ff) |
							((Block9.SampleFrequency[1]&0x0ff)<<8) |
							((Block9.SampleFrequency[2]&0x0ff)<<16) |
							((Block9.SampleFrequency[3]&0x0ff)<<24)
							);

						if (CurrentState.MasterRate==-1)
						{
							CurrentState.MasterRate = CurrentState.SampleRate;
						}

						CurrentState.NumChannels = Block9.NoOfChannels;
						CurrentState.BitsPerSample = Block9.BitsPerSample;

						DataLength = (BlockLength-sizeof(VOC_BLOCK9));

						NumSamples += (DataLength<<3)/(CurrentState.BitsPerSample*CurrentState.NumChannels);

						/* skip */
						BufferedFile_Seek(bufferedFile, DataLength, SEEK_CUR);
					}
				}
				break;
			}
		}
	}
	while (Header.ID!=0);

	return NumSamples;
}


static void VOC_GetNextDataBlock()
{
	VOC_BLOCK_HEADER Header;
	int bControlBlock = 0;

	do
	{
		/* read header */
		if (BufferedFile_ReadData(&CurrentState.bufferedFile,&Header, sizeof(VOC_BLOCK_HEADER))==sizeof(VOC_BLOCK_HEADER))
		{
			unsigned long BlockLength;

			/* get block length */
			BlockLength = GetBlockLength(&Header);

			switch (Header.ID)
			{
				/* terminator */
				case 0:
				{
					CurrentState.BlockLength = 0;
					break;
				}

				/* sound data */
				case 1:
				{
					VOC_BLOCK1 Block1;

					if (BufferedFile_ReadData(&CurrentState.bufferedFile,&Block1, sizeof(VOC_BLOCK1))==sizeof(VOC_BLOCK1))
					{
						unsigned long DataLength;

						/* voc2tzx reads all up to 246 */

						CurrentState.SampleRate = UnpackFrequency(Block1.SampleRate); 						

						if (CurrentState.MasterRate==-1)
						{
							CurrentState.MasterRate = CurrentState.SampleRate;
						}

						/* 
						Compression Type:
						8-bits    = 0
                         4-bits    = 1
                         2.6-bits  = 2
                         2-bits    = 3
                         Multi DAC = 3+(# of channels) [interesting--
                                       this isn't in the developer's manual]
						*/
						if (Block1.CompressionType==0)
						{
							CurrentState.BitsPerSample = 8;
							CurrentState.NumChannels = 1;
						}
						
						VOC_SetupFormat();

						DataLength = (BlockLength-sizeof(VOC_BLOCK1));

						CurrentState.BlockLength = DataLength;
					}
				}
				break;

				/* continuation */
				case 2:
				{
					CurrentState.BlockLength = BlockLength;
				}
				break;

				/* silence */
				case 3:
				{
					VOC_BLOCK3 Block3;

					if (BufferedFile_ReadData(&CurrentState.bufferedFile,&Block3, sizeof(VOC_BLOCK3))==sizeof(VOC_BLOCK3))
					{



					}

				}
				break;

				/* control */
				case 4:
				case 5:
				case 6:
				case 7:
				{
					bControlBlock = 1;
					HandleBlock(&CurrentState.bufferedFile, &Header);
				}
				break;

				/* extended */
				case 8:
				{


				}
				break;

				case 9:
				{
					VOC_BLOCK9 Block9;

					if (BufferedFile_ReadData(&CurrentState.bufferedFile,&Block9, sizeof(VOC_BLOCK9))==sizeof(VOC_BLOCK9))
					{
						unsigned long DataLength;

						CurrentState.SampleRate = (
							(Block9.SampleFrequency[0]&0x0ff) |
							((Block9.SampleFrequency[1]&0x0ff)<<8)
							);

						if (CurrentState.MasterRate==-1)
						{
							CurrentState.MasterRate = CurrentState.SampleRate;
						}

						CurrentState.NumChannels = Block9.NoOfChannels;
						CurrentState.BitsPerSample = Block9.BitsPerSample;

						DataLength = (BlockLength-sizeof(VOC_BLOCK9));
	
						VOC_SetupFormat();

						CurrentState.BlockLength = DataLength;
					}
				}
				break;
			}
		}
		else
		{
			CurrentState.BlockLength = 0;
		}
	}
	while (bControlBlock);
}

int VOC_Open(const char *filename, SampleDescription *desc)
{
	int Status = SAMPLE_FILE_NOT_RECOGNISED;

	BufferedFile_Init(&CurrentState.bufferedFile, 8192);

	/* open file */
	if (BufferedFile_Open(&CurrentState.bufferedFile, filename))
	{
		VOC_MAIN_HEADER Header;

		if (BufferedFile_ReadData(&CurrentState.bufferedFile, &Header, sizeof(VOC_MAIN_HEADER))==sizeof(VOC_MAIN_HEADER))
		{
			/* header text matches?? */
			if (memcmp(Header.IdentText, VOC_MAIN_HEADER_IDENT,0x014)==0)
			{
				/* yes */
				
				unsigned long Version;
				unsigned long EncodedVersion;

				/* get unencoded version */

				/* get version major */
				Version = Header.UnencodedVersionMajor<<8;
				/* get version minor */
				Version |= Header.UnencodedVersionMinor;

				/* get encoded version */
				EncodedVersion = Header.EncodedVersionMajor<<8;
				EncodedVersion |= Header.EncodedVersionMinor;

				EncodedVersion -= 0x01234;
				EncodedVersion ^= 0x0ffff;

				if ((EncodedVersion & 0x0ffff) == (Version & 0x0ffff))
				{
					unsigned long OffsetOfFirstDataBlock;
					unsigned long DataToSkip;
					unsigned long FirstBlockOffset;

					OffsetOfFirstDataBlock = Header.FirstBlockOffsetLow | 
							(Header.FirstBlockOffsetHigh<<8);
					
					DataToSkip = OffsetOfFirstDataBlock - sizeof(VOC_MAIN_HEADER);

					if (DataToSkip!=0)
					{
						/* seek forwards */
						BufferedFile_Seek(&CurrentState.bufferedFile, DataToSkip, SEEK_CUR);
					}

					FirstBlockOffset = BufferedFile_GetPos(&CurrentState.bufferedFile);
					
					
					CurrentState.MasterRate = -1;
					desc->LengthInSamples = CalculateLengthInSamples(&CurrentState.bufferedFile);
					desc->Rate = CurrentState.MasterRate;

					BufferedFile_Seek(&CurrentState.bufferedFile, FirstBlockOffset, SEEK_SET);

					SampleDescription_InitDuration(desc);

					CurrentState.OffsetInBlock = 0;
					CurrentState.BlockLength = 0;

					CurrentState.EndOfFile = 0;

					Status = SAMPLE_FILE_OK;
				}
				else
				{
					Status = SAMPLE_FILE_ERROR;
				}
			}
		}

		if (Status!=SAMPLE_FILE_OK)
		{
			BufferedFile_Free(&CurrentState.bufferedFile);
/*			fclose(CurrentState.fh);
			CurrentState.fh = NULL; */
		}
	}

	return Status;
}

void VOC_Close(SampleDescription *desc)
{
	BufferedFile_Free(&CurrentState.bufferedFile);

//	if (CurrentState.fh!=NULL)
//	{
//		fclose(CurrentState.fh);
//		CurrentState.fh = NULL;
//	}
}

unsigned char VOC_ReadSample(SampleDescription *desc)
{
	if (CurrentState.OffsetInBlock==CurrentState.BlockLength)
	{
		CurrentState.OffsetInBlock = 0;

		VOC_GetNextDataBlock();

		desc->Format = CurrentState.Format;

		/* end of file */
		if (CurrentState.BlockLength==0)
		{
			CurrentState.EndOfFile = 1;
			return 0;
		}
	}

	{
		unsigned char Data[8];
		unsigned long BytesPerSample;

		BytesPerSample = GetBytesPerSample(desc);
		CurrentState.OffsetInBlock+=BytesPerSample;

		BufferedFile_ReadData(&CurrentState.bufferedFile,Data, BytesPerSample);

		return ConvertData(Data, desc->Format);
	}
}

unsigned long VOC_IsEOF(SampleDescription *desc)
{
	return CurrentState.EndOfFile;
}

