/*
 *  (c) Kevin Thacker, 2002
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "wav.h"
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <string.h>

SampleHandler WAV_SampleHandler=
{
	"Wave Audio File (WAV)",
	WAV_Open,
	WAV_Close,
	WAV_ReadSample,
	WAV_IsEOF,
};


static unsigned long EndOfFile;
static CHUNK_FILE_READ ChunkFileRead;


int WAV_Open(const char *pFilename,SampleDescription *desc)
{
	int Status = SAMPLE_FILE_NOT_RECOGNISED;
	int bGotFormat = 0;
	unsigned long BitsPerSample;
	unsigned long NumChannels;

	if (ChunkFile_Open(&ChunkFileRead, pFilename))
	{
		int bOk = 0;

		if (ChunkFile_GetParentChunkID(&ChunkFileRead)== CHUNK_ID_RIFF)
		{
			ChunkFile_SetEndianness(&ChunkFileRead, ENDIANNESS_LITTLE_ENDIAN);
			bOk = 1;
		}
		else if (ChunkFile_GetParentChunkID(&ChunkFileRead)==CHUNK_ID_RIFX)
		{
			ChunkFile_SetEndianness(&ChunkFileRead, ENDIANNESS_BIG_ENDIAN);
			bOk = 1;
		}

		if (bOk)
		{
			int bGotFormat = 0;
			unsigned char Type[4];

			if (ChunkFile_ReadData(&ChunkFileRead, &Type[0], 4)==4)
			{
				if (
					(Type[0]=='W') ||
					(Type[1]=='A') ||
					(Type[2]=='V') ||
					(Type[3]=='E')
					)
				{
					if (ChunkFile_FindChunk(&ChunkFileRead, CHUNK_ID_fmt))
					{
						unsigned long ChunkLength;

						ChunkFile_OpenChunk(&ChunkFileRead);

						ChunkLength = ChunkFile_GetParentChunkLength(&ChunkFileRead);

						/* the 'fmt ' chunk must have a size of at least
						WAV_WAVEFORMAT otherwise there is an error */
						if (ChunkLength < sizeof(WAV_WAVEFORMAT))
						{
							Status = SAMPLE_FILE_ERROR;
						}
						else
						{
							/* the chunk has a size which is equal or greater than
							WAV_WAVEFORMAT */
							unsigned long ReadLength;
							WAV_WAVEFORMATEX WaveFormat;

							/* read a maximum of WAV_WAVEFORMATEX */
							if (ChunkLength>=sizeof(WAV_WAVEFORMATEX))
								ReadLength = sizeof(WAV_WAVEFORMATEX);
							else
								ReadLength = sizeof(WAV_WAVEFORMAT);
					
							if (ChunkFile_ReadData(&ChunkFileRead, (unsigned char *)&WaveFormat,ReadLength)==ReadLength)
							{
								unsigned short Format;

								/* get format details */
#ifdef MSB_FIRST
								Format = SwapEndianWord(WaveFormat.FormatTag);
#else
								Format = WaveFormat.FormatTag;
#endif
								if (Format!=1)
									Status = SAMPLE_FILE_DATA_FORMAT_UNSUPPORTED;
#ifdef MSB_FIRST
								desc->Rate = SwapEndianLong(WaveFormat.SamplesPerSecond);
#else
								desc->Rate = WaveFormat.SamplesPerSecond;
#endif
								BitsPerSample = WaveFormat.BitsPerSample;

								NumChannels = WaveFormat.NoOfChannels;

								switch (BitsPerSample)
								{
									case 8:
									{
										if (NumChannels==1)
										{
											desc->Format = SAMPLE_FORMAT_8MU;
										}
										else
										{
											desc->Format = SAMPLE_FORMAT_8SU;
										}
									}
									break;

									case 16:
									{
										if (NumChannels==1)
										{
											desc->Format = SAMPLE_FORMAT_16MS_LE;
										}
										else
										{
											desc->Format = SAMPLE_FORMAT_16SS_LE;
										}
									}
									break;
								}


								bGotFormat = 1;
							}
						}

						ChunkFile_CloseChunk(&ChunkFileRead);
					}


					if (bGotFormat)
					{
						if (ChunkFile_FindChunk(&ChunkFileRead, CHUNK_ID_data))
						{
							unsigned long ChunkLength;
							
							ChunkFile_OpenChunk(&ChunkFileRead);
							
							ChunkLength = ChunkFile_GetParentChunkLength(&ChunkFileRead);

							desc->LengthInSamples = (ChunkLength<<3)/(BitsPerSample*NumChannels);

							EndOfFile =0;
							SampleDescription_InitDuration(desc);

							if (desc->LengthInSamples==0)
							{
								Status = SAMPLE_FILE_NO_DATA;
							}
							else
							{
								Status = SAMPLE_FILE_OK;

							}
						}
					}
				}
			}
		}

		if (Status!=SAMPLE_FILE_OK)
		{
			ChunkFile_Close(&ChunkFileRead);
		}
	}

	return Status;
}

void	WAV_Close(SampleDescription *desc)
{
	ChunkFile_Close(&ChunkFileRead);
}
	
unsigned long WAV_IsEOF(SampleDescription *desc)
{
	return EndOfFile;
}


unsigned char WAV_ReadSample(SampleDescription *desc)
{
	unsigned char Data[8];
	unsigned long BytesPerSample = GetBytesPerSample(desc);

	if (ChunkFile_ReadData(&ChunkFileRead, Data, GetBytesPerSample(desc))!=BytesPerSample)
	{
		EndOfFile = 1;
		return 0x080;
	}

	return ConvertData(Data, desc->Format);
}
