/* Convert a PC bitmap file to a MODE 0 CPC screen file

   There are several restrictions on what type of bitmap file can be used:

   * Both the width and height must be divisible by 8 - this is due to the
     nature of the CPC's CRTC settings
   * It must have a colour depth of 4 bits (i.e. 16 colours), which is also
     the colour depth used by the CPC in MODE 0
   * It must not be compressed
   * It must fit within the standard CPC screen size of 0x4000 bytes. If it is
     too large, the width and/or height of the bitmap screen must be reduced
     
   This routine does not allow for direct loading of the CPC screen to screen
   memory. This is to reduce the size of the converted screen (direct loading
   to screen memory would require that the file is padded out to 0x4000 bytes).
   Instead, it will be necessary to write your own Z80 assembler routine to
   copy the screen in the appropriate manner
   
   Program written by Nicholas Campbell
   Started 7th May 2005
   Finished 15th May 2005
   
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>

#define AMSDOS_HEADER_SIZE 128
#define CPC_SCREEN_START_ADDR 0x4000

unsigned char bitmap_screen_mem[0x8000];
   /* Memory reserved for the bitmap screen */
unsigned long int cpc_screen_size;   /* Size of the CPC screen */

/* Define a structure to store information about a bitmap file */

typedef struct bitmap {
  unsigned long int file_size;
  unsigned long int width;
  unsigned long int height;
  unsigned short int bit_depth;
  unsigned long int compression;
  unsigned char *data;
} BITMAP;

/* ---------
   FUNCTIONS
   --------- */

/* load_bitmap_file

   Load a bitmap file into memory

   Check if there is an AMSDOS header at the start of a file
  
   An AMSDOS header is 128 bytes long. A 16-bit checksum is stored at bytes 67
   and 68 of the file, and it is calculated by adding bytes 0-66 together. If
   the calculated checksum does not match the value stored at bytes 67 and 68,
   it is assumed that there is no AMSDOS header at the start of the file
   
   Parameters:
   filename - a string containing the name of the file to check
   
   Returns:
   0 if the file does not contain an AMSDOS header, or 1 if it does
   
*/

void load_bitmap_file(unsigned char *filename, BITMAP *bitmap_info) {

  FILE *file_in;  /* Stream for reading the bitmap file */
  unsigned long int bitmap_offset;   /* Offset from the beginning of the bitmap
    file to the beginning of the bitmap data */
  unsigned long int bitmap_size;   /* Size of the bitmap data */
  unsigned char error_flag = 0;
  int i;  /* Variable used in loops */

  /* Open the bitmap screen file for reading */

  file_in = fopen(filename,"rb");  
  
  /* Check if the bitmap screen file has the correct header before reading it
  
     The first two bytes of a bitmap file are the ASCII characters 'B' and 'M'
     respectively. If this is not the case, print an error message */
  
  if ((fgetc(file_in) != 'B') || (fgetc(file_in) != 'M')) {
	fprintf(stderr,"ERROR: %s is not a bitmap file!\n",filename);
    exit(1);
  }

  /* Read the size of the bitmap file */
  
  fread(&bitmap_info->file_size,sizeof(unsigned long int),1,file_in);

  /* Skip the next four bytes; these should all be 0, although this is not
     checked */
  
  for (i = 0; i < 4; i++) { fgetc(file_in); }
  
  /* Read the offset from the beginning of the file to the beginning of the
     pixel data */

  fread(&bitmap_offset,sizeof(unsigned long int),1,file_in);

  /* Skip the next four bytes; these contain the size of the 'core header' */
  
  for (i = 0; i < 4; i++) { fgetc(file_in); }
      
  /* Read the width and height of the bitmap
  
     In order for the bitmap to fit neatly on a CPC screen, both the height and
     width of the bitmap must be divisible by 8; if this is not the case, print
     an error message later on */
  
  fread(&bitmap_info->width,sizeof(unsigned long int),1,file_in);
  fread(&bitmap_info->height,sizeof(unsigned long int),1,file_in);

  if (bitmap_info->width%8 != 0) {
	fprintf(stderr,"ERROR: Width of bitmap screen in pixels must be divisible by 8!\n");
	error_flag++;
  }
  if (bitmap_info->width%8 != 0) {
	fprintf(stderr,"ERROR: Height of bitmap screen in pixels must be divisible by 8!\n");
	error_flag++;
  }
  
  /* Skip the next two bytes; these contain the number of planes in the bitmap
     (which should be 1) */
    
  for (i = 0; i < 2; i++) { fgetc(file_in); }

  /* The bitmap conversion routine will only read bitmaps with a 4-bit colour
     depth (i.e. 16 colours), as this is the maximum number of colours that can
     be displayed normally on a CPC screen. If any other colour depth is
     encountered, print an error message later on */
    
  fread(&bitmap_info->bit_depth,sizeof(unsigned short int),1,file_in);

  if (bitmap_info->bit_depth != 4) {
	fprintf(stderr,"ERROR: Bitmap screen must have a colour depth of 4 bits (i.e. 16 colours)!\n");
	error_flag++;
  }

  /* Read the compression method used in the bitmap (if any)
  
     Only non-compressed bitmaps can be read, so if a compressed bitmap is
     encountered, print an error message */
  
  fread(&bitmap_info->compression,sizeof(unsigned long int),1,file_in);

  if (bitmap_info->compression != 0) {
	fprintf(stderr,"ERROR: Bitmap screen must not be compressed!\n");
	error_flag++;
  }
    
  /* Print any error messages relating to the bitmap headers */
  
  if (error_flag) { exit(1); }
  
  /* 34 bytes have already been read, so skip the rest of the bytes in the
     headers, as we do not need the information from them */
  
  for (i = 0; i < bitmap_offset-34; i++) { fgetc(file_in); }

  /* Calculate the size of the CPC screen that will result; this must be less
     than or equal to 0x4000 bytes, otherwise there will be problems displaying
     it correctly on a CPC (assuming that overscan is not being used, which it
     won't be)
     
     On a CPC, each byte can store two MODE 0 pixels, which equates to four
     pixels in the bitmap screen, so the formula for calculating the CPC screen
     size is:
     
     (width * height) / 4 */

  cpc_screen_size = (bitmap_info->width * bitmap_info->height)/4;
  if (cpc_screen_size > 0x4000) {
	fprintf(stderr,"ERROR: Bitmap is too large to fit on a standard CPC screen!\n");
	fprintf(stderr," Maximum size of standard CPC screen = 0x4000 bytes\n");
	fprintf(stderr," Size of CPC screen file from %s = 0x%lx bytes\n",filename,cpc_screen_size); 
    exit(1);
  }
       
  /* Allocate memory for the bitmap data
  
     With 4-bit colour depth, each pixel requires half a byte, so one byte can
     store two pixels, and the number of bytes required for one line will be
     width/2 */

  bitmap_size = (bitmap_info->width/2)*bitmap_info->height;

  bitmap_info->data = (char *)malloc(bitmap_size);
  if (bitmap_info->data == NULL) {
	fprintf(stderr,"ERROR: Unable to allocate memory for bitmap data!\n");
    exit(1);
  }
  
  /* Read the bitmap data */
  
  fread(bitmap_info->data,1,bitmap_size,file_in);

  /* Close the bitmap file */
  
  fclose(file_in);

}


/* convert_pixels_to_screen_byte

   Convert two pixels (left and right) to the corresponding byte that represents
   these pixels on a CPC screen in MODE 0
   
   The screen display on a CPC in MODE 0 is rather strange. If the bits of the
   left pixel are labelled L3 L2 L1 L0, and the bits of the right pixel are
   labelled R3 R2 R1 R0, the corresponding screen byte is
   L0 R0 L2 R2 L1 R1 L3 R3
   
   Parameters:
   left_pixel - the ink number of the left pixel (0-15)
   right_pixel - the ink number of the right pixel (0-15)
   
   Returns:
   screen_byte - the byte representing the left and right pixels on a CPC screen
   in MODE 0
   
*/

unsigned char convert_pixels_to_screen_byte(unsigned char left_pixel,
  unsigned char right_pixel) {
	  
  unsigned char screen_byte = 0;  /* Reset the screen byte */

  /* Firstly, read bits 3-0 of the left pixel one at a time in that order by
     resetting all the other bits, move the bit to its correct location in the
     screen byte, then OR it with the current screen byte for all the other
     bits that have been read so far */
  
  screen_byte = (left_pixel & 0x08)>>2;
  screen_byte |= (left_pixel & 0x04)<<3;
  screen_byte |= (left_pixel & 0x02)<<2;
  screen_byte |= (left_pixel & 0x01)<<7;

  /* Do the same for bits 3-0 of the right pixel */
  
  screen_byte |= (right_pixel & 0x08)>>3;
  screen_byte |= (right_pixel & 0x04)<<2;
  screen_byte |= (right_pixel & 0x02)<<1;
  screen_byte |= (right_pixel & 0x01)<<6;  

  /* Return the corresponding CPC screen byte */
    
  return(screen_byte);
  		
}


/* check_amsdos_character

   Check if the current character is valid for use in an AMSDOS filename
   
   As well as A-Z, a-z and 0-9, the following symbols are valid characters for
   use in an AMSDOS filename:
   
   ! " # $ & ' + - @ ^ ` { }
   
   Parameters:
   character - the character to check the validity of

   Returns:
   0 if the character is not valid for use in an AMSDOS filename, or 1 if it is

*/

int check_amsdos_character (char character) {

  /* First, check if the character matches any alphanumeric characters (both
     upper case and lower case) */
	
  if (!( ((character >= 'A') && (character <= 'Z'))
	|| ((character >= 'a') && (character <= 'z'))
    || ((character >= '0') && (character <= '9')) )) {

    switch (character) {
      case '!' : break; case '"' : break; case '#' : break; case '$' : break;
      case '&' : break; case '\'' : break; case '+' : break; case '-' : break;
      case '@' : break; case '^' : break; case '`' : break; case '{' : break;
      case '}' : break;

      /* The character has been tested against all of the valid AMSDOS
         characters and it does not match any of them, so it is not valid */
      
      default:
        return(0);

    }

  }

  /* The character matches one of the valid AMSDOS characters */
  
  return(1);
  		
}


/* setup_amsdos_header

   Set up an AMSDOS header to be written to a file for use on a CPC

   Parameters:
   filename - the filename to write to the AMSDOS header; this must conform to
    the MS-DOS style 8.3 convention (e.g. filename.ext)
   file_type - the file type (0 = BASIC, 2 = binary)
   file_start_addr - the default start address that the file is loaded to
   file_length - the length of the file
   file_entry_addr - the address to CALL or JumP to once the file is loaded

   Returns:
   amsdos_header - a pointer to the AMSDOS header
      
*/

unsigned char * setup_amsdos_header(unsigned char *filename,
  unsigned char file_type, unsigned short int file_start_addr,
  unsigned short int file_length, unsigned short int file_entry_addr) {

  /* Declare variables */

  unsigned char *amsdos_header;   /* Area of memory reserved for AMSDOS header */
  unsigned int amsdos_header_index = 0;   /* Current position in AMSDOS header */
  short unsigned int checksum = 0;  /* Checksum of AMSDOS header */
  unsigned int index = 0;
  unsigned int filename_length = 0;   /* Length of the filename before the
    full stop separator (if there is one) */
  unsigned int extension_length = 0;   /* Length of the extension after the
    full stop separator (if there is one) */
  int i;  /* Variable used in loops */

  /* Allocate memory for the AMSDOS header */

  amsdos_header = (unsigned char *)malloc(AMSDOS_HEADER_SIZE);
  amsdos_header_index = 0;  /* Current position in AMSDOS header */
  
  for (i = 0; i < AMSDOS_HEADER_SIZE; i++) { amsdos_header[i] = 0; }
  amsdos_header_index++;

  /* Examine each character in the filename until a full stop separator or the
     end of the string is found */
  
  while ((filename[index] != '.') && (filename[index] != 0)) {

    if (!(check_amsdos_character(filename[index]))) {
	  fprintf(stderr,"ERROR: AMSDOS filename contains an invalid character (%c)!\n",filename[index]);
      exit(1);
    }

	filename_length++; index++;
	  	  
  }
  
  /* If the part of the filename before the separator is 0 or more than 8
     characters long, print an error message */

  if ((filename_length == 0) || (filename_length > 8)) {
	fprintf(stderr,"ERROR: AMSDOS filename does not conform to 8.3 rule (e.g. FILENAME.EXT)!\n");
    exit(1);
  }
  
  /* Otherwise, copy the characters before the separator into the AMSDOS
     header and convert any letters into upper case */
  
  else {
	for (i = 0; i < index; i++) {
	  amsdos_header[amsdos_header_index] = toupper(filename[i]);
	  amsdos_header_index++;
    }

    /* If the filename before the separator is less than 8 characters long, fill
       it with spaces to make it 8 characters long */
        
    i = 0;
    while (i < 8-filename_length) {
	  amsdos_header[amsdos_header_index] = ' ';
	  amsdos_header_index++;
	  i++;
    }   

  }

  /* The character currently being looked at will either be a full stop or \0,
     which means that the end of the string has been reached. If this is the
     case, the filename has no extension and it is not necessary to check its
     length */
     
  if (filename[index] != 0) {
	  
	/* Skip the full stop separator */
	
	index++;

    /* Examine each character in the filename until a full stop separator or the
       end of the string is found */
	
	while (filename[index] != 0) {
      if (!(check_amsdos_character(filename[index]))) {
	    fprintf(stderr,"ERROR: AMSDOS filename contains an invalid character (%c)!\n",filename[index]);
	    exit(1);
      }
      
      /* As well as the invalid AMSDOS characters, the extension must also not
         contain a full stop (which would mean that the filename has at least
         two extensions) */
      
      if (check_amsdos_character(filename[index]) == '.') {
	    fprintf(stderr,"ERROR: AMSDOS filename does not conform to 8.3 rule (e.g. FILENAME.EXT)!\n");
        exit(1);
      }
      
	  extension_length++; index++;

    }

  }
  
  /* If the extension is more than 3 characters long, print an error message */

  if (extension_length > 3) {
	fprintf(stderr,"ERROR: AMSDOS filename does not conform to 8.3 rule (e.g. FILENAME.EXT)!\n");
    exit(1);
  }

  /* Otherwise, copy the characters after the separator into the AMSDOS header
     and convert any letters into upper case */

  else {
	for (i = index-extension_length; i < index; i++) {
	  amsdos_header[amsdos_header_index] = toupper(filename[i]);
	  amsdos_header_index++;
    }

    /* If the extension is less than 3 characters long, fill it with spaces to
       make it 3 characters long */

    i = 0;
    while (i < 3-extension_length) {
	  amsdos_header[amsdos_header_index] = ' ';
	  amsdos_header_index++;
	  i++;
    }

  }
  
  /* Write the file type, start address, length and entry address
  
     These variables are stored in the following bytes in the AMSDOS header:
     
     File type - 18
     Start address - 21-22
     Length - 24-25 and 64-65
     Entry address - 26-27
     
     The start address, length and entry address are stored in little-endian
     format (i.e. the least significant byte is stored first) */
  
  amsdos_header[18] = file_type;

  amsdos_header[21] = file_start_addr%256;
  amsdos_header[22] = file_start_addr/256;

  amsdos_header[24] = file_length%256;
  amsdos_header[25] = file_length/256;

  amsdos_header[26] = file_entry_addr%256;
  amsdos_header[27] = file_entry_addr/256;

  amsdos_header[64] = file_length%256;
  amsdos_header[65] = file_length/256;
  
  /* Calculate the checksum and store it in bytes 67 and 68 of the AMSDOS
     header
     
     The checksum is stored in little-endian format (i.e. the least significant
     byte is stored first) */

  for (i = 0; i <= 66; i++) { checksum += amsdos_header[i]; }
  
  amsdos_header[67] = checksum%256;
  amsdos_header[68] = checksum/256;
  
  return(amsdos_header);
		
}


/* ------------
   MAIN PROGRAM
   ------------ */

int main(int argc, char *argv[]) {
	
  /* Declare variables */
	
  struct stat file_info;  /* Buffer to store file information; the stat
                             structure is defined in sys/types.h */
  unsigned char *bitmap_screen_filename;  /* Name of bitmap screen file */
  BITMAP *bitmap_screen_info;  /* Information about bitmap screen file */

  FILE *file_out;  /* Stream for writing the CPC screen file */
  unsigned char *cpc_screen_filename;  /* Name of CPC screen file */
  unsigned char *cpc_screen_data;  /* Area of memory to allocate to CPC
    screen */
  unsigned char *amsdos_header;  /* Area of memory to allocate to AMSDOS
    header */
  unsigned int cpc_screen_data_index;  /* Current position in CPC screen data */

  unsigned char left_pixel, right_pixel;   /* Ink numbers of left and right
    pixels, used when converting bitmap pixels to CPC screen bytes */
  int x, y, y2;  /* Variable used in loops */
  unsigned short int bitmap_pos;  /* Used to calculate position of pixels in
    bitmap */
  
  /* Check the command line arguments
  
     If there is no filename specified, print an error message */
  
  if (argc < 3) {
    fputs("USAGE: sprtodat bitmap_screen_filename cpc_screen_filename\n",stderr);
    exit(1);
  }
	
  /* Get the filename of the CPC screen from the command line and store it in
     another string */
  
  bitmap_screen_filename = (char *)malloc(strlen(argv[1])+1);
  strcpy(bitmap_screen_filename,argv[1]);
  
  cpc_screen_filename = (unsigned char *)malloc(strlen(argv[2])+1);
  strcpy(cpc_screen_filename,argv[2]);

  /* If the bitmap screen file does not exist, print an error message */
  
  if (stat(bitmap_screen_filename,&file_info)) {
	fprintf(stderr, "ERROR: Unable to open %s!\n",bitmap_screen_filename);
	exit(1);
  }

  /* Initialise the structure */
  
  bitmap_screen_info = (BITMAP *)malloc(sizeof(BITMAP));
  
  /* Read the bitmap file into memory */
  
  load_bitmap_file(bitmap_screen_filename,bitmap_screen_info);
  
  /* Allocate memory for the CPC screen file */

  cpc_screen_data = (unsigned char *)malloc(cpc_screen_size);
  cpc_screen_data_index = 0;

  /* The CPC screen is divided into rows, each consisting of 8 lines. It is
     stored in memory like this:
     
     Row 0, line 0; row 1, line 0; row 2, line 0 ... row h, line 0
     Row 0, line 1; row 1, line 1; row 2, line 1 ... row h, line 1
     ...
     Row 0, line 7; row 1, line 7; row 2, line 7 ... row h, line 7
     
     The routine below is difficult for me to explain! */
     
  for (y = 0; y < 8; y++) {

    /* The number of rows on the screen is equal to the height of the bitmap
       divided by 8 (since there are 8 lines in each row */

	for (y2 = 0; y2 < (bitmap_screen_info->height)/8; y2++) {

	  /* The width of each line in bytes is equal to the width of the bitmap
	     divided by 4 */
		
      for (x = 0; x < (bitmap_screen_info->width)/4; x++) {
	  
        /* Given the row number, line number and position in the line,
           calculate the equivalent position in the bitmap data
           
           The formula for calculating this position is:

           pos = ((height-1) - ((y*80) + (y2 * 0x800))) * (width/2) + x*2
           
           where x = 0-79 (column), y = 0-24 (row), y2 = 0-7 (line in row),
           width = bitmap width in pixels, height = bitmap height in pixels */

	    bitmap_pos = (bitmap_screen_info->height - 1 - (y2*8+y))
	      * ((bitmap_screen_info->width)/2) + x*2;

	    /* Get the ink numbers of the left and right pixels
	    
	       In a 4-bit bitmap, each byte contains two pixels, but each MODE 0
	       pixel corresponds to two bitmap pixels. Therefore, each set of four
	       pixels in the bitmap screen needs to be converted to two pixels in
	       the CPC screen.

	       If the pixels in the bitmap screen are labelled P0 P1 P2 P3, then
	       the first byte stores P0 and P1, and the second byte stores P2 and
	       P3. This routine gets the ink numbers of P0 and P2 and ignores P1 and
	       P3, but is also possible to do the reverse
	       
	       P0 and P2 are both stored in bits 7-4 of their respective bytes. To
	       convert this to an ink number from 0-15, AND it with 0xf0
	       (= 11110000), then shift the bits so they are in bits 3-0 */
	      
	    left_pixel = ((bitmap_screen_info->data[bitmap_pos]) & 0xf0)>>4;
	    right_pixel = ((bitmap_screen_info->data[bitmap_pos+1]) & 0xf0)>>4;

	    /* Convert these pixels to their corresponding CPC screen byte */

        cpc_screen_data[cpc_screen_data_index] =
	      convert_pixels_to_screen_byte(left_pixel,right_pixel);
	      
        cpc_screen_data_index++;

      }
	      
    }
	  
  }

  /* Set up an AMSDOS header for the CPC screen. This will be a binary file, so
     the file type is set to 2 */
  
  amsdos_header = setup_amsdos_header(cpc_screen_filename,2,
    CPC_SCREEN_START_ADDR,cpc_screen_size,0);

  /* Write the CPC screen file - the AMSDOS header first, followed by the
     converted screen data */
  
  file_out = fopen(cpc_screen_filename,"wb");  

  fwrite(amsdos_header,1,AMSDOS_HEADER_SIZE,file_out);
  fwrite(cpc_screen_data,1,cpc_screen_size,file_out);

  /* Close the file and free memory allocated for the bitmap data, AMSDOS
     header and CPC screen data */
    
  fclose(file_out);
  free(bitmap_screen_info->data);
  free(amsdos_header);
  free(cpc_screen_data);

  return(0);
    
}
