;Loader for zblast SD - loads the title screen, and then the game
;
;Written by Nicholas Campbell
;Started 7th May 2005; finished 15th May 2005

org &0200

;The tables containing the addresses of the sprites for each type of baddie and
;the ink data for writing text characters; these are stored at other locations
;initially, but they must be page-aligned
char_ink_table_hb equ &3e

;The mask table, used for drawing sprites transparently; this is 256 bytes long
;and must be page-aligned 
mask_table_hb equ &3f

tmpsprite	equ &a500	;temp. sprite used for drawing text


start:
;Initialise AMSDOS
ld hl,(&be7d)   ;Get the address where the current drive in use is stored
ld a,(hl)   ;Get the drive number (0 = A, 1 = B)
push af   ;Store the drive number

ld de,&40   ;DE = first byte of usable memory
ld hl,&abff   ;HL = last byte of usable memory
ld c,7   ;C = ROM number (AMSDOS is stored in ROM 7)
call &bcce   ;Initialise AMSDOS. This resets the drive to A, which is why it was
             ;necessary to preserve the drive that is currently being used

pop af   ;Get the drive number
ld hl,(&be7d)   ;Get the address where the current drive in use is stored
ld (hl),a   ;Store the drive number

;Disable disc error messages
ld a,&c9   ;&c9 = Z80 RET instruction
ld (&bb5a),a   ;Disable the firmware CALL to print text to the screen
ld a,&ff
ld (&be78),a   ;If any non-zero number is stored at &be78, any 'Retry, Ignore or
               ;Cancel?' messages are replaced with 'Bad command'. Disabling the
               ;firmware CALL to print text prevents any error messages being
               ;displayed at all


;Set the mode and border
xor a
call &bc0e
ld bc,0
call &bc38

;Set all the inks to black
ld hl,inkdata_black
call set_inks

;Change the CRTC settings to those of the standard CPC screen
ld hl,crtc_data
call set_crtc

ld hl,&0802
ld (tmpsprite),hl   ;Set the size of an ASCII character, which is stored in the
                    ;temporary sprite area

;generate zxyaddr table

zxyaddrgen:
ld de,zxyaddr
ld hl,&0008   ;the screen starts at an offset of 8 (i.e. &4008 or &c008)
ld b,200   ;there are 200 lines on the screen
gzxylp1:
push bc   ;Store the number of lines remaining
ld a,l
ld (de),a
inc de
ld a,h
ld (de),a
inc de

add a,8   ;Calculate the screen address for the line below the current one, by
ld h,a    ;adding &800 to it
bit 6,a
call nz,nextline   ;The screen address needs to be adjusted after every eight
                  ;lines, by resetting the high byte and adding &c050
pop bc   ;Get the number of lines remaining

djnz gzxylp1


;Set up the mask table, so that sprites can be displayed transparently, using
;ink 0 as a transparent ink
xor a
ld hl,mask_table_hb*256

setup_mask_table_loop:
push af   ;Store the current byte

set_mask_for_left_pixel:
and &aa   ;Get the value of the left pixel only by resetting the even bits
          ;(&aa = &x10101010)
or a
jr nz,left_pixel_is_not_0
ld b,&aa
jr set_mask_for_right_pixel
left_pixel_is_not_0:
ld b,0

set_mask_for_right_pixel:
pop af   ;Get the current byte
push af   ;Store it again
and &55   ;Get the value of the right pixel only, by resetting the odd bits
          ;(&55 = &x01010101)
or a
jr nz,right_pixel_is_not_0
ld a,&55
jr store_mask_byte_in_table
right_pixel_is_not_0:
xor a

store_mask_byte_in_table:
or b
ld (hl),a

pop af   ;Get the current byte
inc hl   ;Go to the next entry in the mask table
add a,1   ;Go to the next byte; unlike the inc a instruction, this affects the
          ;carry flag, which we need to test for
jr nc,setup_mask_table_loop


;Copy the character ink table to another location so that it is page-aligned
ld hl,char_ink_table_temp
ld de,char_ink_table_hb*256
ld bc,char_ink_table_temp_end-char_ink_table_temp
ldir

;start up double-buffering
ld hl,bankm
ld a,&40   ;set the screen to draw
ld hl,banks
ld a,&10   ;set the screen to display
call flipend	;writes byte and re-enables ints

;Write the text to the introduction screen
call write_intro_text

;Set the inks
ld hl,inkdata_intro
call set_inks

ld a,1   ;On a Spectrum, this is set to bright ink 1 (65)
ld (cyclecol),a

mainloop:
;Print the 'zblast SD' text using colour cycling, by printing it in a different
;style (i.e. colour) in every cycle of the main loop from 1-7
ld hl,cyclecol
ld a,(hl)
inc a
cp 8
jr c,mlskip
ld a,1
mlskip:
ld (hl),a
ld bc,&100a
call ilprint
defb 'z b l a s t  S ','D'+128

call doflip


;Wait for a short while to slow down the rate at which the 'zblast SD' text
;flashes
ld bc,$2300
delay_loop:
dec bc
ld a,b
or c
jr nz,delay_loop

;Check if a key has been pressed, and if not, stay on the intro screen
call &bb1b
jr nc,mainloop


;Ensure that the screen at &c000 is selected, by changing register 13 (&10 =
;&4000, &30 = &c000)
ld bc,&bc0c
out (c),c
ld bc,&bd30
out (c),c


;Load the title screen to &4000
load_title_scr:
ld hl,filename_title_scr
ld de,&4000
call load_file
jr nc,load_title_scr   ;If there was an error loading the file, try again


;Set all the inks to black, so that we cannot see the title screen data being
;copied to the screen
ld hl,inkdata_black
call set_inks

;Change the CRTC settings for the title screen
ld hl,crtc_data_title_scr
call set_crtc


;Copy the screen from &4000 to &c000
ld b,8
ld hl,&4000
ld de,&c000
copy_screen_loop:
push bc
ld bc,52*32   ;BC = the width of the title screen in bytes, multiplied by its
              ;height in rows (not lines)
push de   ;Store the address to write the title screen data to
ldir
pop de   ;Get the address to write the title screen data to
ld a,d
add a,8   ;Go to the next set of lines by adding &800 to the screen address of
ld d,a    ;the first row
pop bc
djnz copy_screen_loop


;Set the inks for the title screen, so that it is now visible
ld hl,inkdata_title_screen
call set_inks

;Wait for a few seconds, but allow the user to press a key to skip the delay
ld bc,$6000
delay_loop_title_scr:
call &bb1b
jr c,load_game
dec bc
ld a,b
or c
jr nz,delay_loop_title_scr


;Load the game
load_game:
ld hl,filename_game
ld de,&8000
call load_file
jr nc,load_game   ;If there was an error loading the file, try again


;Set all the inks to black again
ld hl,inkdata_black
call set_inks

;Hide the screen by setting its height to zero; this prevents the 'Venetian
;blind' effect when the main game clears the screen
ld bc,&bc06
out (c),c
ld bc,&bd00
out (c),c

;Let's start blasting!
jp &8000


;Write the text for the introduction before the title screen. This is written to
;both screens in memory
write_intro_text:
call write_intro_text2
write_intro_text2:
;first do a *full* cls
ld a,(bankm)  ;get current screen address
ld h,a
ld l,0
ld d,h
ld e,l
inc e
ld bc,&3fff
ld (hl),l
ldir

ld bc,&0210
ld a,4
call ilprint
defb 'Spectrum version programmed b','y'+128

ld bc,&0312
ld a,4
call ilprint
defb 'Russell Marks for the 2003 4','K'+128

ld bc,&0c14
ld a,4
call ilprint
defb 'Minigame competitio','n'+128

ld bc,&0a18
ld a,5
call ilprint
defb 'Amstrad CPC conversio','n'+128

ld bc,&011a
ld a,5
call ilprint
defb 'programmed by Nicholas Campbel','l'+128

ld bc,&191c
ld a,5
call ilprint
defb 'in 200','5'+128

ld bc,&0722
ld a,4
call ilprint
defb 'Press any key to continu','e'+128
jp doflip
;falls through

doflip:
;On the Spectrum, frame control is used when flipping screens, in order to slow
;down the game a bit. On the CPC, there is no frame control as it takes several
;frames to draw everything on the screen


di
ld hl,banks
ld a,(hl)
xor &20   ;switches between &10 and &30, which correspond to &40 and &c0
flipend:
ld (hl),a
push af
call memflip   ;change the screen to draw to as well (either &4000 or &c000)
pop af
ld bc,&bc0c
out (c),c  ;select CRTC register 12
inc b
out (c),a  ;output the high byte of the screen address
ei
ret


;flip memory bank *only*, without waiting
;no need to disable ints for this one
memflip:
ld a,(bankm)
xor &80
ld (bankm),a
ret


;Set the inks
;
;Entry conditions - HL = address of ink data
;Exit conditions - A = 16, HL = address immediately following end of ink data
; B, C, DE corrupted
;
;The ink data must use firmware colours (0-26) and not hardware colours

set_inks:
xor a   ;Start at ink 0
set_inks_loop:
ld b,(hl)   ;B = firmware colour for the current ink
ld c,b   ;Making C = B means that the ink does not flash
push af   ;Store the ink number
push hl   ;Store the address containing the firmware colour of the current ink
call &bc32   ;Set the ink
pop hl   ;Get the address containing the firmware colour of the current ink
pop af   ;Get the ink number
inc hl   ;Go to the next entry in the table
inc a   ;Go to the next ink number
cp 16
jr nz,set_inks_loop
call &bd19   ;Wait for two frame flybacks to ensure that all the inks have
jp &bd19     ;been set, since the firmware does not set the inks to their new
             ;colours immediately

inkdata_intro:
defb 0    ;Ink 0 = black
defb 1    ;Ink 1 = dark blue
defb 3    ;Ink 2 = brown
defb 4    ;Ink 3 = purple
defb 9    ;Ink 4 = dark green
defb 10   ;Ink 5 = grey/blue
defb 12   ;Ink 6 = yellow/brown
defb 13   ;Ink 7 = grey
defb 0    ;Ink 8 = black
defb 2    ;Ink 9 = blue
defb 6    ;Ink 10 = red
defb 8    ;Ink 11 = magenta
defb 18   ;Ink 12 = green
defb 20   ;Ink 13 = cyan
defb 24   ;Ink 14 = yellow
defb 26   ;Ink 15 = white

inkdata_title_screen:
defb 0    ;Ink 0 = black
defb 1    ;Ink 1 = dark blue
defb 2    ;Ink 2 = blue
defb 3    ;Ink 3 = brown
defb 4    ;Ink 4 = purple
defb 6    ;Ink 5 = red
defb 8    ;Ink 6 = magenta
defb 9    ;Ink 7 = dark green
defb 10   ;Ink 8 = grey/blue
defb 12   ;Ink 9 = yellow/brown
defb 14   ;Ink 10 = pastel blue
defb 16   ;Ink 11 = pink
defb 18   ;Ink 12 = green
defb 20   ;Ink 13 = cyan
defb 24   ;Ink 14 = yellow
defb 26   ;Ink 15 = white

inkdata_black:
defs 16   ;All inks are black


;Change the CRTC settings
;
;Entry conditions - HL = address of CRTC data
;Exit conditions - B = &bc, HL = address immediately following end of CRTC data
; A, C corrupted
;
;The CRTC data consists of register/value pairs, and it must end with 255

set_crtc:
ld b,&bc
set_crtc_loop:
ld c,(hl)   ;Get the CRTC register to change
inc hl
ld a,c
cp 255   ;255 marks the end of the CRTC data
ret z
ld a,(hl)   ;Get the value to change it to
inc hl
out (c),c   ;Select the CRTC register
inc b
out (c),a   ;Change it to the new value
dec b
jr set_crtc_loop
ret

crtc_data:
defb 1,40     ;Register 1 = screen width
defb 2,46     ;Register 2 = horizontal screen position
defb 6,25     ;Register 6 = screen height
defb 7,30     ;Register 7 = vertical screen position
defb 12,&30   ;Register 12 = high byte of screen address (&30 = &c000)
defb 13,0     ;Register 13 = low byte of screen address
defb 255

crtc_data_title_scr:
defb 1,26,2,39,6,32,7,34,255


;Print a string
;
;Entry conditions - A = ink to write string in (0-7), B = x-coordinate, C =
; y-coordinate; the string must immediately follow the CALL to this routine, and
; the last character in the string must have bit 7 set
;Exit conditions - B = 0, C = 0, DE = screen address immediately after last
; character in the string, HL = address to jump to after exiting from this
; routine
; A corrupted

get_char_data_ink equ get_char_data_inkop+1
ilprint:
rlca   ;Multiply the ink by 4, as each entry in the ink table consists of 4
rlca   ;bytes
ld (get_char_data_ink),a

call calc_screen_addr   ;Calculate the screen address to write the string to
                        ;(stored in HL)
ex de,hl   ;Now DE = screen address

pop hl   ;Get the address of the string immediately after the CALL to this
         ;routine

ilprint_loop:
ld a,(hl)   ;Get the character from the string
and 127   ;Ensure that it is between 0 and 127
cp 32
jr z,ilprint_next_char

push hl   ;Store the address of the current character in the string
push de   ;Store the screen address
call get_char_data   ;Convert the character to a MODE 0 sprite and store it in
                     ;the temporary sprite

pop hl   ;Get the screen address (now stored in HL)
push hl   ;Store it again
ld de,tmpsprite   ;The temporary sprite contains the character to print
call drawsprgen2
pop de   ;Get the screen address (now stored in DE)

pop hl   ;Get the address of the current character in the string

ilprint_next_char:
inc de
inc de   ;Set the screen address for the next character
bit 7,(hl)   ;The last character in the string must have bit 7 set
inc hl   ;Go to the next character in the string
jr z,ilprint_loop   ;Repeat for the remaining characters in the string

jp (hl)   ;HL = address immediately after end of string, so return from this
	      ;routine

	      
;Create a temporary sprite for an ASCII character, to be used when printing the
;character itself
;
;Entry conditions - A = ASCII number of character
;Exit conditions - B = 0, C = 0, DE = byte immediately following end of temporary
; sprite, HL = byte immediately following end of character data
; A corrupted
;
;The ink to use should already be set at get_char_data_inkop; the routine to
; print a string does this

get_char_data:
ld de,char_table-256   ;The character table starts at character 32, but it is
                       ;necessary to start at the address of the non-existent
                       ;character 0
ld h,0
ld l,a
add hl,hl   ;HL = character number * 8 (each entry in the character table uses
add hl,hl   ;8 bytes)
add hl,hl
				  
ex de,hl   ;Now DE = character number * 8, HL = address of entry in character
	   ;table for character 0
add hl,de   ;HL = address of entry in character table

ld de,tmpsprite+2
ld b,8   ;Each character consists of 8 lines

get_char_data_loopy:
ld c,2
ld a,(hl)   ;Get the bit data for the current line of the current character

get_char_data_loopx:
rlca   ;Rotate the bits left twice so that the next set of 2 pixels can be read
rlca
push af   ;Store the bit data

and 3   ;A is now a number between 0 and 3, representing the bit data for 2
        ;pixels
push de   ;Store the temporary sprite address
ld e,a   ;E = bit data for 2 pixels

;Compare this data with values in the ink table

push hl   ;Store the character table address
ld h,char_ink_table_hb   ;Set high byte of ink table
get_char_data_inkop: ld a,0   ;modified (A = ink * 4)
add a,e
ld l,a   ;DE = ink * 4 + bit data for 2 pixels
ld a,(hl)   ;A = ink data corresponding to this style and the 2 pixels
            ;specified

pop hl   ;Get the character table address
pop de   ;Get the temporary sprite address
ld (de),a   ;Write the result to the temporary sprite
inc de   ;Go to the next byte in the temporary sprite
pop af   ;Get the bit data

dec c
jr nz,get_char_data_loopx

inc hl   ;Go to the next line for this character in the character table

djnz get_char_data_loopy   ;Repeat for the remaining lines
ret


;Print a sprite using transparent mode (i.e. without erasing the background)
;
;Entry conditions - B = x-coordinate (0-31), C = y-coordinate (0-23), DE =
; sprite address
;Exit conditions - B = 0, DE = byte immediately following end of sprite, HL =
; screen address of top left of sprite
; A corrupted
;
;This routine requires a mask table of 256 bytes (1 for each combination of 2
; pixels) which must be initialised before calling this routine
;The screen address is calculated using the x- and y-coordinates stored in BC. If
; it is already known, you can call drawsprgen2 instead and skip the screen
; address calculation (this is exploited in the routine to print a string)
;Ink 0 is used as the transparent ink

print_sprite_width equ print_sprite_widthop+1
drawsprgen_screen_addr equ drawsprgen_screen_addrop+1

drawsprgen:
push de   ;Store the sprite address
call calc_screen_addr   ;Calculate the screen address to print the sprite
                        ;(from 0 to &3fff)
pop de   ;Get the sprite address

drawsprgen2:
ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
or h   ;Add this to the screen address to print the sprite
ld h,a   ;Now HL = screen address from &4000 to &7fff or &c000 to &ffff
ld (drawsprgen_screen_addr),hl   ;Store the screen address so that it is
                                 ;returned in HL on exiting

ld a,(de)   ;Get the width of the sprite in bytes
ld (print_sprite_width),a   ;Set the width of the sprite in bytes
inc de
ld a,(de)   ;Get the height of the sprite in lines
ld b,a   ;B = height of sprite in lines
inc de

print_sprite_loopy:
push bc   ;Store the number of lines remaining

print_sprite_widthop: ld b,0   ;B = width of sprite (modified)
push hl   ;Store the screen address of the start of the current line

print_sprite_loopx:
ex de,hl   ;Now DE = screen address, HL = sprite address
ld a,(hl)   ;Get the sprite data

push hl   ;Store the sprite address
ld h,mask_table_hb
ld l,a   ;HL = address of entry in mask table
ld c,(hl)   ;C = entry in mask table corresponding to pixel data
pop hl   ;Get the sprite address

ld a,(de)   ;Get the byte that is currently on the screen
and c   ;Apply the mask, so that any pixel that does not contain ink 0 will be
        ;erased, allowing the sprite pixels to be merged with the screen pixels
or (hl)   ;Merge the sprite pixels with the screen pixels

ld (de),a   ;Write this to the screen
ex de,hl   ;Now DE = sprite address, HL = screen address
inc de   ;Go to the next byte of the sprite
inc hl   ;Go to the next byte on the screen

djnz print_sprite_loopx

pop hl   ;Get the screen address of the start of this line
ld bc,&0800
add hl,bc   ;Set the screen address to the line below
bit 6,h   ;Check if bit 14 of the screen address is reset (which means that the
          ;result of the addition is off the screen and must be re-adjusted)
call z,nextline   ;Calculate the screen address of the next line if necessary

pop bc
djnz print_sprite_loopy
drawsprgen_screen_addrop: ld hl,&0000   ;The screen address of the top left of
                                        ;the sprite - this is modified
ret


;The font data and the character ink table for writing text
include 'font.dat'


;Calculate the screen address of a specified x- and y-coordinate
;Entry conditions - B = x-coordinate, C = y-coordinate
;Exit conditions - A = x-coordinate * 2, HL = screen address
; DE corrupted

;This routine requires a screen address table, which stores the addresses of each
;of the lines on the screen, to have been initialised beforehand

calc_screen_addr:
ld de,zxyaddr   ;DE = address of screen address table
ld a,c
rlca   ;Multiply the y-coordinate by 4 (each y-coordinate is separated by 4
rlca   ;lines)

ld h,0
ld l,a   ;HL = y-coordinate * 4
add hl,hl   ;Multiply HL by 2, since each entry in the screen address table uses
            ;2 bytes
add hl,de   ;Now HL = location of screen address of current line
ex de,hl    ;Now DE = location of screen address of current line

;Read the screen address of the current line from the screen address table
;(stored in DE), and store the address in HL

calc_screen_addr_x:
ld a,(de)
ld l,a
inc de
ld a,(de)
ld h,a
ld d,0
ld e,b   ;DE = x-coordinate
add hl,de   ;Now HL = the screen address with the x-coordinate set
ret


;Go to the first line of the next row of the screen
;Entry conditions - HL = screen address
;Exit conditions - BC = &c050, HL = screen address of line below

nextline:
ld bc,&c050
add hl,bc
ret


;Load a file into memory
;
;Entry conditions - DE = address to load file to, HL = filename address
;Exit conditions - carry set if file was loaded successfully, carry reset if
; there was an error while loading the file
; A, BC, DE, HL corrupted
;
;The filename must be 12 characters long; if it is less than this, use
;additional spaces to bring it to this length (e.g. "file.ext" ->
;"file    .ext")
load_file_address equ load_file_addressop+1

load_file:
ld (load_file_address),de   ;Set the address to load the file to
ld de,&c000   ;DE = address of 2K buffer for reading files (not used when
              ;loading from disc)
ld b,12   ;B = length of filename
call &bc77   ;Open the file for reading
ret nc

load_file_addressop: ld hl,0   ;HL = address to load the file to (modified)
call &bc83   ;Load the file
ret nc
jp &bc7a   ;Close the file and return

filename_title_scr:
defb "zblastsd.scr"

filename_game:
defb "zblastsd.bin"


;high byte of address to draw screen (&4000 or &c000)
bankm defb &40
;high byte of CRTC register 12, which determines the screen to display (&10 =
;&4000; &30 = &c000)
banks defb &10


cyclecol
zxyaddr equ cyclecol+1
