/* xnc100em, an X-based NC100 emulator (based on nc100em).
 * Copyright (C) 1994 Ian Collier. xnc100em changes (C) 1996 Russell Marks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/Intrinsic.h>

#ifdef MITSHM
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#endif

#ifdef HAS_UNAME
#include <sys/utsname.h>
#endif

#include "z80.h"

#define MAX_DISP_LEN 256

#include "xnc100em.icon"

#if SCALE>1
static unsigned long scaleup[256]; /* to hold table of scaled up bytes,
                                 e.g. 00110100B -> 0000111100110000B */
#endif


#define BORDER_WIDTH	5
int rrshm=10,rrnoshm=10,mitshm=1;


/* the memory is 256k ROM, 64k RAM, and 1024k PCMCIA memory card */
unsigned char mem[(256+64+1024)*1024];
unsigned char *dfile;
/* boots with all ROM0 */
unsigned char *memptr[4]={mem,mem,mem,mem};
int kports[8];
unsigned long tstates=0,tsmax=60000;

int memattr[4]={0,0,0,0};	/* all ROM at boot */

int hsize=480*SCALE,vsize=64*SCALE;
volatile int interrupted=0;
int input_wait=0;
int scrn_freq=10;

int card_size=0;		/* size of pcmcia card, or 0 if none */
int card_status=0;		/* as read from port 0xA0 */
int irq_status=0xff;		/* as read from port 0x90 */
int irq_mask=0;			/* as set via port 0x60 */
int scrnaddr=0;
int do_nmi=0;
int count_and=0x7ff;

struct tm *rtc_tm;

#define SCRN_XOFS	80		/* x offset of display */
#define SCRN_YOFS	200		/* y offset */

#ifdef PTY_SERIAL

int pty_fd;
unsigned char seroutbuf[16384];		/* won't need this many - defensive */
int seroutpos=0;

#ifndef SERIAL_MOUSE
#if 0
/* XXX this is a revolting kludge */
#define PTY_NAME	"/dev/ptysf"
#else
/* this is more sensible for the X version! */
#define PTY_NAME	"/dev/tty"
#endif
#else
#define PTY_NAME	"/dev/ttyS0"
#endif
#endif

unsigned char keyports[10]={0,0,0,0,0, 0,0,0,0,0};



void sighandler(a)
int a;
{
interrupted=1;
}


void dontpanic(a)
int a;
{
closedown();
writecard(mem+CARD_START);
exit(1);
}


void dontpanic_nmi()
{
closedown();
writeram(mem+RAM_START);
writecard(mem+CARD_START);
exit(1);
}



main(argc,argv)
int argc;
char **argv;
{
struct sigaction sa;
struct itimerval itv;
int tmp=1000/100;	/* 100 ints/sec */

loadrom(mem);
loadram(mem+RAM_START);
loadcard(mem+CARD_START);
set_card_status();

#ifdef PTY_SERIAL
serial_init();
#endif

startup(&argc,argv);

if(argc==2) scrn_freq=atoi(argv[1])*2;
if(scrn_freq!=0) argc--,argv++;
if(scrn_freq<4) scrn_freq=4;
if(scrn_freq>100) scrn_freq=100;

if(argc>=2)
  {
  FILE *in;
  
  if((in=fopen(argv[1],"rb"))==NULL)
    printf("Couldn't open file.\n");
  else
    {
    if(argc==3)
      fread(mem+RAM_START+atoi(argv[2]),1,49152,in);
    else
      fread(mem+RAM_START+0x100,1,49152,in);
    fclose(in);
    }
  }

sa.sa_handler=dontpanic;
sa.sa_mask=0;
sa.sa_flags=SA_ONESHOT;

sigaction(SIGINT, &sa,NULL);
sigaction(SIGHUP, &sa,NULL);
sigaction(SIGILL, &sa,NULL);
sigaction(SIGTERM,&sa,NULL);
sigaction(SIGQUIT,&sa,NULL);
sigaction(SIGSEGV,&sa,NULL);

#ifndef NO_SPEED_CONTROL
sa.sa_handler=sighandler;
sa.sa_mask=0;
sa.sa_flags=SA_RESTART;

sigaction(SIGALRM,&sa,NULL);

itv.it_value.tv_sec=  0;
itv.it_value.tv_usec= 100000;
itv.it_interval.tv_sec=  tmp/1000;
itv.it_interval.tv_usec=(tmp%1000)*1000;
setitimer(ITIMER_REAL,&itv,NULL);
#endif

mainloop();

/* doesn't ever get here really, but all the same... */
closedown();
writeram(mem+RAM_START);
writecard(mem+CARD_START);
exit(0);
}


/* the card_status flag never changes after this */
set_card_status()
{
card_status=0x33; /* no batteries low, and parallel always ACK and !BUSY */
if(card_size==0) card_status|=0x80;	/* is card present? */
}


loadrom(x)
unsigned char *x;
{
int i;
FILE *in;

if((in=fopen("nc100.rom","rb"))!=NULL)
  {
  fread(x,1024,256,in);
  fclose(in);
  }
else
  {
  printf("Couldn't load ROM.\n");
  exit(1);
  }
}


loadram(x)
unsigned char *x;
{
int i;
FILE *in;

if((in=fopen("nc100.ram","rb"))!=NULL)
  {
  fread(x,1024,64,in);
  fclose(in);
  }
else
  {
  printf("Couldn't load RAM.\n");
  /* not fatal - blank it and carry on */
  memset(x,0xff,64*1024);
  }
}


loadcard(x)
unsigned char *x;
{
int i;
FILE *in;

if((in=fopen("nc100.card","rb"))!=NULL)
  {
  card_size=fread(x,1024,1024,in);
  fclose(in);
  }
else
  {
  printf("Couldn't load PCMCIA memory card.\n");
  /* also not fatal, blank it */
  memset(x,0,1024*1024);
  }
}


writeram(x)
unsigned char *x;
{
FILE *out;

if((out=fopen("nc100.ram","wb"))==NULL)
  printf("Couldn't write RAM.\n");
else
  {
  fwrite(x,1024,64,out);
  fclose(out);
  }
}


writecard(x)
unsigned char *x;
{
FILE *out;

if(card_size>0)
  {
  if((out=fopen("nc100.card","wb"))==NULL)
    printf("Couldn't write PCMCIA memory card.\n");
  else
    {
    fwrite(x,1024,card_size,out);
    fclose(out);
    }
  }
}


/* I/O, quoting from nciospec.doc:
[we ignore some]
D0-DF                   RTC (TC8521)            R/W	(ignore for now)
C0-C1                   UART (uPD71051)         R/W	(ignore for now)
B0-B9                   Key data in             R
A0                      Card Status etc.        R
90                      IRQ request status      R/W
70                      Power on/off control    W
60                      IRQ Mask                W
50-53                   Speaker frequency       W	(ignore)
40                      Parallel port data      W	(ignore)
30                      Baud rate etc.          W	(ignore)
20                      Card wait control       W	(ignore)
10-13                   Memory management       R/W
00                      Display memory start    W
*/


/* simulate reads from paging ports.
 * this is slow and cack but these are almost never read, just written.
 */
int check_paging(page)
int page;
{
int ofs,ret;

if(page<0 || page>3)
  {
  printf("can't happen");
  return(0);
  }

ret=0;	/* default to ROM */
ofs=memptr[page]-mem;
if(ofs>=RAM_START && ofs<CARD_START)
  ret|=0x40,ofs-=RAM_START;
else
  if(ofs>=CARD_START)
    ret|=0x80,ofs-=CARD_START;

ret|=(ofs/16384);

return(ret);
}


/* now this *is* used a bit, but fortunately it's dead quick :-) */
do_paging(page,n)
int page,n;
{
switch(n&0xc0)
  {
  case 0x00:	/* ROM */
    memptr[page]=mem+16384*n;
    memattr[page]=0;
    break;
  case 0x40:	/* RAM */
    memptr[page]=mem+RAM_START+16384*(n&0x0f);
    memattr[page]=1;
    break;
  case 0x80:	/* card ram */
    memptr[page]=mem+CARD_START+16384*(n&0x3f);
    memattr[page]=1;
    /* one way of detecting the size of a card requires
     * that writes which are off-card must fail, so...
     */
    if(16*(n&0x3f)>=card_size) memattr[page]=0;
    break;
  }
}



int read_kybd(n)
int n;
{
return(keyports[n]);
}



unsigned int in(h,l)
int h,l;
{
static int ts=(13<<8);	/* num. t-states for this out, times 256 */

/* h is ignored by the NC100 */
switch(l)
  {
  /* RTC */
  case 0xd0: case 0xd1: case 0xd2: case 0xd3:
  case 0xd4: case 0xd5: case 0xd6: case 0xd7:
  case 0xd8: case 0xd9: case 0xda: case 0xdb:
  case 0xdc: case 0xdd: case 0xde: case 0xdf:
    switch(l-0xd0)
      {
      case 0:	return(ts|('0'+rtc_tm->tm_sec%10));
      case 1:	return(ts|('0'+rtc_tm->tm_sec/10));
      case 2:	return(ts|('0'+rtc_tm->tm_min%10));
      case 3:	return(ts|('0'+rtc_tm->tm_min/10));
      case 4:	return(ts|('0'+rtc_tm->tm_hour%10));
      case 5:	return(ts|('0'+rtc_tm->tm_hour/10));
      
      case 6:	return(ts);
      
      case 7:	return(ts|('0'+rtc_tm->tm_mday%10));
      case 8:	return(ts|('0'+rtc_tm->tm_mday/10));
      case 9:	return(ts|('0'+(rtc_tm->tm_mon+1)%10));
      case 10:	return(ts|('0'+(rtc_tm->tm_mon+1)/10));
      case 11:	return(ts|('0'+(rtc_tm->tm_year-90)%10));
      case 12:	return(ts|('0'+(rtc_tm->tm_year-90)/10));
      
      case 13:
      case 14:
      case 15:
        return(ts);
      
      default:
        printf("can't happen\n");
      }
    return(ts|0);
  
  
#ifndef PTY_SERIAL
  case 0xc0: case 0xc1: return(ts|0);
#else
  /* UART */
  case 0xc0:	/* this reads from the serial port */
    if(serial_input_pending()) return(get_serial_byte());
    return(ts|0);
  
  case 0xc1:    /* returns bit 0=1 if can send a byte */
    /* we assume it's always possible, though (XXX) bad things
     * may happen if there isn't a process reading the pty...?
     */
    return(ts|1);
#endif
    
  /* keyboard */
  case 0xb0: case 0xb1: case 0xb2: case 0xb3: case 0xb4:
  case 0xb5: case 0xb6: case 0xb7: case 0xb8: case 0xb9:
    /* reading 0xb9 also sets bit 3 of irq_status */
    if(l==0xb9) irq_status|=8;
    return(ts|read_kybd(l-0xb0));
  
  case 0xa0:	/* card etc. status */
    return(ts|card_status);
  
  case 0x90:	/* IRQ status */
    return(ts|irq_status);
  
  /* memory paging */
  case 0x10: return(ts|check_paging(0));
  case 0x11: return(ts|check_paging(1));
  case 0x12: return(ts|check_paging(2));
  case 0x13: return(ts|check_paging(3));
  }

/* otherwise... */
#if 0
fprintf(stderr,"in l=%02X\n",l);
#endif
return(ts|255);
}


unsigned int out(h,l,a)
int h,l,a;
{
static int ts=13;	/* num. t-states for this out */
time_t timet;

/* h is ignored by the NC100 */
switch(l)
  {
  /* RTC */
  case 0xdd:
    /* this one locks output (I think) when bit 3 is 1 */
    /* get time */
    timet=time(NULL);
    rtc_tm=localtime(&timet);
    return(ts);
      
  case 0xd0: case 0xd1: case 0xd2: case 0xd3:
  case 0xd4: case 0xd5: case 0xd6: case 0xd7:
  case 0xd8: case 0xd9: case 0xda: case 0xdb:
  case 0xdc:            case 0xde: case 0xdf:
    /* ignored, don't want to let them set the time!
     * (would need to be root anyway)
     */
    return(ts);
  
#ifndef PTY_SERIAL
  case 0xc0: case 0xc1: return(ts);
#else
  /* UART */
  case 0xc0:	/* this writes to the serial port */
    put_serial_byte(a);
    return(ts);
  
  case 0xc1:    /* sets up various serial parms which we ignore */
    return(ts);
#endif
  
  case 0x90:	/* IRQ status */
    /* when a zero is written to a bit, it should re-enable that IRQ line */
    irq_status|=(a^255);
    return(ts);
  
  case 0x70:	/* power on/off */
    /* if bit 0 is 0, turn off */
    if((a&1)==0) dontpanic_nmi();
    return(ts);
  
  case 0x60:	/* set irq mask */
    irq_mask=a;
    return(ts);
  
  case 0x50: case 0x51: case 0x52: case 0x53:
    /* speaker frequency, ignored */
    return(ts);
  
  case 0x40:	/* parallel port data, ignored */
    return(ts);
  
  case 0x30:	/* baud rate etc., ignored */
    return(ts);
  
  case 0x20:	/* card wait control, ignored */
    return(ts);
  
  /* memory paging */
  case 0x10: do_paging(0,a); return(ts);
  case 0x11: do_paging(1,a); return(ts);
  case 0x12: do_paging(2,a); return(ts);
  case 0x13: do_paging(3,a); return(ts);
  
  case 0:	/* set screen addr */
    scrnaddr=(a&0xf0)*256;
    return(ts);
  }

/* otherwise... */
#if 0
fprintf(stderr,"out l=%02X,a=%02X\n",l,a);
#endif
return(ts);
}


fix_tstates()
{
tstates=0;
#ifdef NO_SPEED_CONTROL
interrupted=1;
#else
pause();
#endif
}


do_interrupt()
{
static int count=0;
static int scount=0;

/* only do refresh() every 1/Nth */
count++;
if(count>=scrn_freq)
  count=0,refresh();
check_events(1);

#ifdef PTY_SERIAL
scount++;
if(scount==4) scount=0,serout_flush();
#endif

interrupted=0;
}


#ifdef PTY_SERIAL

serial_init()
{
/* XXX should do this properly... */
/* mind you this is probably more portable than termios :-) */
char buf[1024];

#ifndef SERIAL_MOUSE
sprintf(buf,"stty raw -echo <%s",PTY_NAME);
#else
sprintf(buf,"stty 1200 cs7 raw <%s",PTY_NAME);
#endif
system(buf);

if((pty_fd=open(PTY_NAME,O_RDWR|O_NONBLOCK))<0)
  {
  printf("Couldn't open pty!\n");
  exit(1);
  }
}


int serial_input_pending()
{
struct timeval tv;
fd_set fds;

tv.tv_sec=0; tv.tv_usec=0;
FD_ZERO(&fds); FD_SET(pty_fd,&fds);
select(pty_fd+1,&fds,NULL,NULL,&tv);

return(FD_ISSET(pty_fd,&fds));
}


int serial_output_allowed()
{
#if 1
/* assume it always is. select() takes yonks in our terms. */
return(1);
#else
struct timeval tv;
fd_set fds;

tv.tv_sec=0; tv.tv_usec=0;
FD_ZERO(&fds); FD_SET(pty_fd,&fds);
select(pty_fd+1,NULL,&fds,NULL,&tv);

return(FD_ISSET(pty_fd,&fds));
#endif
}

#include <errno.h>

int get_serial_byte()
{
int res;
unsigned char c=0;

if((res=read(pty_fd,&c,1))<=0)
  {
  if(res<0 && errno==EAGAIN) return(0);
  /* it probably closed it; re-open and try again */
  close(pty_fd);
  if((pty_fd=open(PTY_NAME,O_RDWR|O_NONBLOCK))<0)
    {
    printf("Couldn't open pty!\n");
    dontpanic();
    }
  read(pty_fd,&c,1);
  }

return((int)c);
}


put_serial_byte(n)
int n;
{
unsigned char c=n;

#if 0
if(serial_output_allowed())
#endif
  seroutbuf[seroutpos++]=c;
}

serout_flush()
{
if(seroutpos>0)
  write(pty_fd,seroutbuf,seroutpos);
seroutpos=0;
}

#endif /* PTY_SERIAL */



/* the remainder of xmain.c is based on xz80's xspectrum.c. */

#ifdef SunKludge
char *shmat();
char *getenv();
#endif

int refresh_screen=1;

/* remember, this table is ignoring shifts... */
static struct {unsigned char port,mask;} keytable[]={
/*  SP      !        "        #        $        %        &        '  */
{1,0x08},{2,0x04},{3,0x01},{6,0x10},{4,0x01},{1,0x40},{7,0x02},{7,0x04},
/*  (       )        *        +        ,        -        .        /  */
{9,0x02},{9,0x01},{8,0x01},{7,0x01},{8,0x80},{8,0x02},{9,0x80},{6,0x20},
/*  0       1        2        3        4        5        6        7  */
{9,0x01},{2,0x04},{3,0x02},{3,0x01},{4,0x01},{1,0x40},{6,0x01},{7,0x02},
/*  8       9        :        ;        <        =        >        ?  */
{8,0x01},{9,0x02},{9,0x08},{9,0x08},{8,0x80},{7,0x01},{9,0x80},{6,0x20},
/*  @       A        B        C        D        E        F        G  */
{8,0x10},{4,0x10},{5,0x04},{5,0x80},{3,0x80},{3,0x10},{4,0x80},{5,0x40},
/*  H       I        J        K        L        M        N        O  */
{6,0x40},{8,0x20},{8,0x40},{7,0x80},{9,0x20},{7,0x40},{6,0x80},{9,0x40},
/*  P       Q        R        S        T        U        V        W  */
{9,0x08},{3,0x04},{4,0x40},{3,0x40},{5,0x10},{7,0x20},{5,0x08},{3,0x08},
/*  X       Y        Z        [        \        ]        ^        _  */
{4,0x08},{5,0x20},{4,0x04},{8,0x08},{7,0x04},{8,0x04},{6,0x01},{8,0x02},
/*  `       a        b        c        d        e        f        g  */
{7,0x10},{4,0x10},{5,0x04},{5,0x80},{3,0x80},{3,0x10},{4,0x80},{5,0x40},
/*  h       i        j        k        l        m        n        o  */
{6,0x40},{8,0x20},{8,0x40},{7,0x80},{9,0x20},{7,0x40},{6,0x80},{9,0x40},
/*  p       q        r        s        t        u        v        w  */
{9,0x08},{3,0x04},{4,0x40},{3,0x40},{5,0x10},{7,0x20},{5,0x08},{3,0x08},
/*  x       y        z        {        |        }        ~       DEL */
{4,0x08},{5,0x20},{4,0x04},{8,0x08},{7,0x04},{8,0x04},{6,0x10},{9,0x04}
};

static unsigned short rgbvals[2][3]={
        { 0x0000, 0x0000, 0x0000}, { 0xffff, 0xffff, 0xffff}
};


static Display *display;
static Screen *scrptr;
static int screen;
static Window root;
static Colormap cmap;
static GC maingc;
static GC fggc, bggc;
static Window borderwin, mainwin;
static XImage *ximage;
static unsigned char *image;
static int linelen;
static int black,white;
static int bytecycles;
static int framebytes;
#ifdef MITSHM
static XShmSegmentInfo xshminfo;
#endif
static int invert=0;
static int borderchange=1;

static int is_local_server(name)
char *name;
{
#ifdef HAS_UNAME
   struct utsname un;
#else
   char sysname[MAX_DISP_LEN];
#endif

   if(name[0]==':')return 1;
   if(!strncmp(name,"unix",4))return 1;
   if(!strncmp(name,"localhost",9))return 1;

#ifdef HAS_UNAME
   uname(&un);
   if(!strncmp(name,un.sysname,strlen(un.sysname)))return 1;
   if(!strncmp(name,un.nodename,strlen(un.nodename)))return 1;
#else
   gethostname(sysname,MAX_DISP_LEN);
   if(!strncmp(name,sysname,strlen(sysname)))return 1;
#endif
   return 0;
}


static Display *open_display(argc,argv)
int *argc;
char **argv;
{
   int i;
   char *ptr;
   char *myname="xz80",*myclass="Xz80";
   char dispname[MAX_DISP_LEN];
   Display *display;

   if((ptr=getenv("DISPLAY")))
     strcpy(dispname,ptr);
   else
     strcpy(dispname,":0.0");
   
   if(!(display=XOpenDisplay(dispname))){
      fprintf(stderr,"Unable to open display %s\n",dispname);
      exit(1);
   }

#ifdef MITSHM   
   mitshm=1;
#else
   mitshm=0;
#endif
   
   /* XXX deal with args here */

   if(mitshm && !is_local_server(dispname)){
      fputs("Disabling MIT-SHM on remote X server\n",stderr);
      mitshm=0;
   }
   return display;
}


static int image_init()
{
#ifdef MITSHM
   if(mitshm){
      ximage=XShmCreateImage(display,DefaultVisual(display,screen),
             DefaultDepth(display,screen),ZPixmap,NULL,&xshminfo,
             hsize,vsize);
      if(!ximage){
         fputs("Couldn't create X image\n",stderr);
         return 1;
      }
      xshminfo.shmid=shmget(IPC_PRIVATE,
               ximage->bytes_per_line*(ximage->height+1),IPC_CREAT|0777);
      if(xshminfo.shmid == -1){
         perror("Couldn't perform shmget");
         return 1;
      }
      xshminfo.shmaddr=ximage->data=shmat(xshminfo.shmid,0,0);
      if(!xshminfo.shmaddr){
         perror("Couldn't perform shmat");
         return 1;
      }
      xshminfo.readOnly=0;
      if(!XShmAttach(display,&xshminfo)){
         perror("Couldn't perform XShmAttach");
         return 1;
      }
      scrn_freq=rrshm;
   } else
#endif
   {
      ximage=XCreateImage(display,DefaultVisual(display,screen),
             DefaultDepth(display,screen),ZPixmap,0,NULL,hsize,vsize,
             8,0);
      if(!ximage){
         perror("XCreateImage failed");
         return 1;
      }
      ximage->data=malloc(ximage->bytes_per_line*(ximage->height+1));
      if(!ximage->data){
         perror("Couldn't get memory for XImage data");
         return 1;
      }
      scrn_freq=rrnoshm;
   }
   linelen=ximage->bytes_per_line/SCALE;
   if(linelen!=60 && linelen!=480)
      fprintf(stderr,"Line length=%d; expect strange results!\n",linelen);
#if 0
   if(linelen==60 &&
         (BitmapBitOrder(display)!=MSBFirst || ImageByteOrder(display)!=MSBFirst))
      fprintf(stderr,"BitmapBitOrder=%s and ImageByteOrder=%s.\n",
         BitmapBitOrder(display)==MSBFirst?"MSBFirst":"LSBFirst",
         ImageByteOrder(display)==MSBFirst?"MSBFirst":"LSBFirst"),
      fputs("If the display is mixed up, please mail me these values\n",stderr),
      fputs("and describe the display as accurately as possible.\n",stderr);
#endif
   ximage->data+=linelen;
   image=ximage->data;
   return 0;
}


static void notify(argc,argv)
int *argc;
char **argv;
{
   Pixmap icon;
   XWMHints xwmh;
   XSizeHints xsh;
   XClassHint xch;
   XTextProperty appname, iconname;
   char *apptext;
   char *icontext="xnc100em";

   icon=XCreatePixmapFromBitmapData(display,root,xnc100em_bits,
        xnc100em_width,xnc100em_height,black,white,
        DefaultDepth(display,screen));
   apptext="xnc100em";
   xsh.flags=PSize|PMinSize|PMaxSize;
   xsh.min_width=hsize;
   xsh.min_height=vsize;
   xsh.max_width=hsize+BORDER_WIDTH*2;
   xsh.max_height=vsize+BORDER_WIDTH*2;
   if(!XStringListToTextProperty(&apptext,1,&appname)){
      fputs("Can't create a TextProperty!",stderr);
      return;
   }
   if(!XStringListToTextProperty(&icontext,1,&iconname)){
      fputs("Can't create a TextProperty!",stderr);
      return;
   }
   xwmh.initial_state=NormalState;
   xwmh.input=1;
   xwmh.icon_pixmap=icon;
   xwmh.flags=StateHint|IconPixmapHint|InputHint;
   xch.res_name="xnc100em";
   xch.res_class="Xnc100em";
   XSetWMProperties(display,borderwin,&appname,&iconname,argv,
      *argc,&xsh,&xwmh,&xch);
}


#if SCALE>1
static void scaleup_init(){
   int j, k, l, m;
   unsigned long bigval; /* SCALING must be <= 4! */
   for(l=0;l<256;scaleup[l++]=bigval)
      for(m=l,bigval=j=0;j<8;j++) {
         for(k=0;k<SCALE;k++)
            bigval=(bigval>>1)|((m&1)<<31);
         m>>=1;
      }
}
#endif


startup(argc,argv)
int *argc;
char **argv;
{
   display=open_display(argc,argv);
   if(!display){
      fputs("Failed to open X display\n",stderr);
      exit(1);
   }
   invert=0;
   screen=DefaultScreen(display);
   scrptr=DefaultScreenOfDisplay(display);
   root=DefaultRootWindow(display);
   white=WhitePixel(display,screen);
   black=BlackPixel(display,screen);
   maingc=XCreateGC(display,root,0,NULL);
   XCopyGC(display,DefaultGC(display,screen),~0,maingc);
   XSetGraphicsExposures(display,maingc,0);
   fggc=XCreateGC(display,root,0,NULL);
   XCopyGC(display,DefaultGC(display,screen),~0,fggc);
   XSetGraphicsExposures(display,fggc,0);
   bggc=XCreateGC(display,root,0,NULL);
   XCopyGC(display,DefaultGC(display,screen),~0,bggc);
   XSetGraphicsExposures(display,bggc,0);
   cmap=DefaultColormap(display,screen);
   if(image_init()){
      if(mitshm){
         fputs("Failed to create X image - trying again with mitshm off\n",stderr);
         mitshm=0;
         if(image_init())exit(1);
      }
      else exit(1);
   }

#if SCALE>1
   if(linelen==60) scaleup_init();
#endif

   borderwin=XCreateSimpleWindow(display,root,0,0,
             hsize+BORDER_WIDTH*2,vsize+BORDER_WIDTH*2,0,0,0);
   mainwin=XCreateSimpleWindow(display,borderwin,BORDER_WIDTH,
             BORDER_WIDTH,hsize,vsize,0,0,0);
   notify(argc,argv);
   XSelectInput(display,borderwin,KeyPressMask|KeyReleaseMask|
      ExposureMask|EnterWindowMask|LeaveWindowMask|
      StructureNotifyMask);
   XMapRaised(display,borderwin);
   XMapRaised(display,mainwin);
   XFlush(display);
   refresh_screen=1;
}


static int process_keypress(kev)
XKeyEvent *kev;
{
   char buf[3];
   KeySym ks;

   XLookupString(kev,buf,2,&ks,NULL);

   switch(ks){
      case XK_F5:
        do_nmi=1;
        break;
      case XK_F10:
        dontpanic();
        /* doesn't return */
        break;
      
      case XK_Insert:
              keyports[1]|=0x01; break;

      case XK_Return: 
              keyports[0]|=0x10; break;
      case XK_Control_L:
              keyports[1]|=0x02; break;
      case XK_Shift_L:
              keyports[0]|=0x01; break;
      case XK_Shift_R:
              keyports[0]|=0x02; break;
      case XK_Alt_L: case XK_Alt_R:
      case XK_Meta_L: case XK_Meta_R:
              keyports[2]|=0x02;
              break;
      case XK_BackSpace: case XK_Delete:
              keyports[9]|=0x04; break;
      case XK_Escape:
              keyports[1]|=0x04; break;
      case XK_Tab:
              keyports[2]|=0x08; break;
      case XK_Up:
              keyports[7]|=0x08; break;
      case XK_Down:
              keyports[6]|=0x02; break;
      case XK_Left:
              keyports[0]|=0x08; break;
      case XK_Right:
              keyports[6]|=0x08; break;
      case XK_minus:
              keyports[8]|=0x02; break;
      case XK_underscore:
              keyports[8]|=0x02; break;
      case XK_equal:
              keyports[7]|=0x01; break;
      case XK_plus:
              keyports[7]|=0x01; break;
      case XK_semicolon:
              keyports[9]|=0x10; break;
      case XK_colon:
              keyports[9]|=0x10; break;
      case XK_apostrophe:
              keyports[8]|=0x10; break;
      case XK_quotedbl:
              keyports[3]|=0x02; break;
      case XK_exclam:
              keyports[2]|=0x04; break;
      case XK_at:
              keyports[8]|=0x10; break;
      case XK_numbersign:
              keyports[6]|=0x10; break;
      case XK_dollar:
              keyports[4]|=0x01; break;
      case XK_percent:
              keyports[1]|=0x40; break;
      case XK_asciicircum:
              keyports[6]|=0x01; break;
      case XK_ampersand:
              keyports[7]|=0x02; break;
      case XK_asterisk:
              keyports[8]|=0x01; break;
      case XK_parenleft:
              keyports[9]|=0x02; break;
      case XK_parenright:
              keyports[9]|=0x01; break;
      case XK_comma:
              keyports[8]|=0x80; break;
      case XK_less:
              keyports[8]|=0x80; break;
      case XK_period:
              keyports[9]|=0x80; break;
      case XK_greater:
              keyports[9]|=0x80; break;
      case XK_slash:
              keyports[6]|=0x20; break;
      case XK_question:
              keyports[6]|=0x20; break;
      case XK_backslash: case XK_bar:
              keyports[7]|=0x04; break;
              break;
      default:
              if(ks>='A' && ks<='Z')keyports[0]&=0xfe;
              if((int)(ks-=32)>=0 && ks<128)
                 keyports[keytable[ks].port]|=keytable[ks].mask;
   }
   return 0;
}


#undef dosym
#define dosym (symbolshift || (keyports[7]|=2))

static void process_keyrelease(kev)
XKeyEvent *kev;
{
   char buf[3];
   KeySym ks;

   XLookupString(kev,buf,2,&ks,NULL);

   switch(ks){
      case XK_Insert:
              keyports[1]&=~0x01; break;

      case XK_Return: 
              keyports[0]&=~0x10; break;
      case XK_Control_L:
              keyports[1]&=~0x02; break;
      case XK_Shift_L:
              keyports[0]&=~0x01; break;
      case XK_Shift_R:
              keyports[0]&=~0x02; break;
      case XK_Alt_L: case XK_Alt_R:
      case XK_Meta_L: case XK_Meta_R:
              keyports[2]&=~0x02;
              break;
      case XK_BackSpace: case XK_Delete:
              keyports[9]&=~0x04; break;
      case XK_Escape:
              keyports[1]&=~0x04; break;
      case XK_Tab:
              keyports[2]&=~0x08; break;
      case XK_Up:
              keyports[7]&=~0x08; break;
      case XK_Down:
              keyports[6]&=~0x02; break;
      case XK_Left:
              keyports[0]&=~0x08; break;
      case XK_Right:
              keyports[6]&=~0x08; break;
      case XK_minus:
              keyports[8]&=~0x02; break;
      case XK_underscore:
              keyports[8]&=~0x02; break;
      case XK_equal:
              keyports[7]&=~0x01; break;
      case XK_plus:
              keyports[7]&=~0x01; break;
      case XK_semicolon:
              keyports[9]&=~0x10; break;
      case XK_colon:
              keyports[9]&=~0x10; break;
      case XK_apostrophe:
              keyports[8]&=~0x10; break;
      case XK_quotedbl:
              keyports[3]&=~0x02; break;
      case XK_exclam:
              keyports[2]&=~0x04; break;
      case XK_at:
              keyports[8]&=~0x10; break;
      case XK_numbersign:
              keyports[6]&=~0x10; break;
      case XK_dollar:
              keyports[4]&=~0x01; break;
      case XK_percent:
              keyports[1]&=~0x40; break;
      case XK_asciicircum:
              keyports[6]&=~0x01; break;
      case XK_ampersand:
              keyports[7]&=~0x02; break;
      case XK_asterisk:
              keyports[8]&=~0x01; break;
      case XK_parenleft:
              keyports[9]&=~0x02; break;
      case XK_parenright:
              keyports[9]&=~0x01; break;
      case XK_comma:
              keyports[8]&=~0x80; break;
      case XK_less:
              keyports[8]&=~0x80; break;
      case XK_period:
              keyports[9]&=~0x80; break;
      case XK_greater:
              keyports[9]&=~0x80; break;
      case XK_slash:
              keyports[6]&=~0x20; break;
      case XK_question:
              keyports[6]&=~0x20; break;
      case XK_backslash: case XK_bar:
              keyports[7]&=~0x04; break;
              break;
      default:
              if((int)(ks-=32)>=0 && ks<96)
                 keyports[keytable[ks].port]&=~keytable[ks].mask;
   }
   return;
}


int check_events(flag)
int flag;
{
   static int answer=0;
   int i;
   static XEvent xev;
   XConfigureEvent *conf_ev;
   XCrossingEvent *cev;
   
   while (XEventsQueued(display,QueuedAfterReading)){
      XNextEvent(display,&xev);
      switch(xev.type){
         case Expose: refresh_screen=1;
            break;
         case ConfigureNotify:
            conf_ev=(XConfigureEvent *)&xev;
            XMoveWindow(display,mainwin,(conf_ev->width-hsize)/2,
                        (conf_ev->height-vsize)/2);
            break;
         case MapNotify: case UnmapNotify: case ReparentNotify:
            break;
         case EnterNotify:
            cev=(XCrossingEvent *)&xev;
            if(cev->detail!=NotifyInferior)
               XAutoRepeatOff(display),XFlush(display);
            break;
         case LeaveNotify:
            cev=(XCrossingEvent *)&xev;
            if(cev->detail!=NotifyInferior)
               XAutoRepeatOn(display),XFlush(display);
            break;
         case KeyPress: i=process_keypress((XKeyEvent *)&xev);
            if(!answer)answer=i;
            break;
         case KeyRelease: process_keyrelease((XKeyEvent *)&xev);
            break;
         default:
            fprintf(stderr,"unhandled X event, type %d\n",xev.type);
      }
   }
   if(flag){
      i=answer;
      answer=0;
      return i;
   }
   else return 0;
}


refresh(){
   int h,i,j,k,l,m,n;
   int x,y;
   int attr;
   unsigned char *src,*dst;
   unsigned char val;
   int mask;
   
   if(borderchange>0)
     {
     /* XXX what about expose events? need to set borderchange... */
     XSetWindowBackground(display,borderwin,white);
     XClearWindow(display,borderwin);
     XFlush(display);
     borderchange=0;
     }

   src=mem+RAM_START+scrnaddr;
   dst=image;
   for(y=0;y<64;y++,src+=4)
     if(linelen==60)
       {
/* XXX need to do scaleup for mono */
       /* 1-bit mono */
       for(x=0;x<60;x++,src++)
         *dst++=~(*src);
       }
     else
       {
       /* assume 8-bit */
       for(x=0;x<60;x++,src++)
         for(i=0,val=(*src),mask=128;i<8;i++,mask>>=1)
#if SCALE<2
           /* i.e. actual size */
           *dst++=((val&mask)?black:white);
#else
           {
           m=((val&mask)?black:white);
           for(j=0;j<SCALE;j++)
             for(k=0;k<SCALE;k++)
               dst[j*hsize+k]=m;
           dst+=SCALE;
           }
       dst+=(SCALE-1)*hsize;
#endif
       }
   
#ifdef MITSHM
   if(mitshm)
      XShmPutImage(display,mainwin,maingc,ximage,0,0,0,0,
        hsize,vsize,0);
   else
#endif
/*   if(refresh_screen)*/
      XPutImage(display,mainwin,maingc,ximage,0,0,0,0,
        hsize,vsize);
   XFlush(display);
   refresh_screen=0;
}


closedown(){
   char buf[1024];
#ifdef MITSHM
   if(mitshm){
      XShmDetach(display,&xshminfo);
      XDestroyImage(ximage);
      shmdt(xshminfo.shmaddr);
      shmctl(xshminfo.shmid,IPC_RMID,0);
   }
#endif
   XAutoRepeatOn(display);
   XCloseDisplay(display);
#ifdef PTY_SERIAL
   close(pty_fd);
   sprintf(buf,"stty -raw echo <%s",PTY_NAME);
   system(buf);
#endif
}
