/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Arithmetic/logic unit
 */

#include "alu.h"

/*
 * These routines handle numerical expressions. There are operators for
 * arithmetic, logic, shift and comparison operations.
 * The core of this is ALU_PVR(), it will recursively call itself to find out
 * the second operand when needed. Its parameter is the priority value of the
 * current operator; all operations with higher priorities are done on the new
 * level. When an illegal character is read or when an operator has a priority
 * lower than the given parameter, the routine returns to the old level.
 * The global variable "ffValue" holds some information about the parsed value:
 *  -A label was referenced that had its "unsure"-flag set
 *  -A definite solution was achieved (no "undefined"-label was referenced)
 *  -The value's size (unknown, 1, 2 or 3 bytes)
 *  -An operation was done on the uppermost recursion level (VP_Work)
 *  -There was nothing to parse
 * "unsure" is needed for producing the same addresses in all passes; because
 * in the first pass there will almost for sure be labels that are undefined,
 * you can't simply get the addressing mode from looking at the parameter's
 * value.
 * "defined" shows that the value really could be computed - so if an undefined
 * label was referenced, this flag will be cleared.
 * "VP_Work" is needed to recognize unnecessary parentheses (which imply use of
 * indirect adressing modes).
 */

/*
 * There are four different ways to call the core routine:
 * ALU_GetValue_Strict()   This routine throws a serious error (and
 *   therefore stops assembly) whenever the result's "defined" flag is clear.
 * ALU_GetValue_Empty()    This routine throws a serious error (and
 *   therefore stops assembly) whenever both the flags "defined" and "nothing
 *   to parse" are clear. This means: Empty expressions are valid, but all
 *   other undefined expressions are not.
 * ALU_GetValue_Medium()   The 'normal' call, it parses an expression and
 *   returns the result. It does not act upon the result's flags.
 * ALU_GetValue_Liberal()  This routine is the most sloppy one - it even
 *   allows one "(" too many in the expression (This is only needed for some
 *   indirect addressing modes).
 */

/*
 * This routine outputs a serious error (stopping assembly) if the result's
 * "defined" flag is clear.
 */
static Value ALU_GetValue_Strict() {
  Value v = ALU_GetValue_Liberal(FALSE);
  if(nParenth) Message(pseSyntax, EERROR);
  if((ffValue & MVALUE_DEFINED) == 0) Message(pseNeedValue, ESERIOUS);
  return(v);
}

/*
 * This routine outputs a serious error (stopping assembly) if the
 * result's "defined" flag is clear and the "exists" flag is set.
 */
static Value ALU_GetValue_Empty() {
  Value v = ALU_GetValue_Liberal(FALSE);
  if(nParenth) Message(pseSyntax, EERROR);
  if((ffValue & (MVALUE_EXISTS | MVALUE_DEFINED))==MVALUE_EXISTS)
    Message(pseNeedValue, ESERIOUS);
  return(v);
}

/*
 * This routine outputs an error if there is no value given at all.
 */
static Value ALU_GetValue_Medium() {
  Value v = ALU_GetValue_Liberal(TRUE);
  if(nParenth)              Message(pseSyntax, EERROR);
  if((ffValue & MVALUE_EXISTS)==0) Message(pseNeedValue, EERROR);
  return(v);
}

/*
 * This routine allows for one "(" too many. Needed when parsing indirect
 * addressing modes with internal indices has to be possible. The parameter
 * defines whether "NeedValue" should be called if the value is undefined.
 */
static Value ALU_GetValue_Liberal(int fNeedValue) {
  Value v;
  ffValue = MVALUE_DEFINED;/* sure, defined, not existant */
  VP_Work = FALSE;
  nLevel   = 0;
/* This variable measures the recursion depth, so an error will be produced if
 * the expression uses too many parentheses, for example.*/
  nParenth = 0;
/* This variable counts the parentheses that are still open. After calling the
 * core routine, it should be zero again (or 1 in special cases, mentioned in
 * the header).*/

  v = ALU_PVR(PRIO_SUBVALUE);
  if(nParenth > 1) {
    nParenth = 0;
    Message(pseTooOpen, EERROR);
  }
#ifdef FDEBUG
  printf("ffValue after ALU call: %x", ffValue);
#endif
  /* allow only *one* force bit */
  if(ffValue & MVALUE_FORCE24) ffValue &= ~(MVALUE_FORCE16 | MVALUE_FORCE08);
  if(ffValue & MVALUE_FORCE16) ffValue &= ~MVALUE_FORCE08;
  /* if value is sure, check to set ISBYTE */
  if((ffValue & MVALUE_UNSURE) == 0) {
    if((v<256) && (v>-129)) ffValue |= MVALUE_ISBYTE;
  }
  /* if there was nothing to parse, mark as undefined */
  if((ffValue & MVALUE_EXISTS)==0) ffValue &= ~MVALUE_DEFINED;
  /* if undefined, return zero */
  if((ffValue & MVALUE_DEFINED) == 0) {
    v = 0;/* if undefined, return 0. */
    if(fNeedValue) NeedValue();
  }
#ifdef FDEBUG
  printf(", after checks: %x\n", ffValue);
#endif
  return(v);
}

/*
 * The actual core of the valueparser, it recursively calls itself when needed.
 * Is re-entrant, of course.
 */
static Value ALU_PVR(int Prio_end) {/* GotByte = first char */
  Value v     = 0,    /* Return value            */
        t;            /* Temporary value         */
  int   fWork = FALSE,/* Operation on this level */
        fEnd  = FALSE,/* Stop recursion if nothing happens */
        ffBuffer;/* needed when "<", ">" or "^" reset size bits */

#ifdef FDEBUG
  printf("R");
#endif
/* Processing stops when an illegal character is reached or an operator is
 * read whose priority is lower than the given parameter.*/
  nLevel += 1;/* increase recursion counter */
  if(nLevel > MAXLEVEL) Message(pseTooDeep, ESERIOUS);
  hOp_Now = HOP_NONE;/* no operator buffered */
/* First part:
 * Expect operand or unary operator.*/
  SKIPSPACE;
  switch(GotByte) {

    case '!':
    GetByte();
    v = (~ALU_PVR(PRIO_NOT));
    ffValue &= ~MVALUE_ISBYTE;
    break;

    case '-':
    GetByte();
    v = (-ALU_PVR(PRIO_NEGATE));
    ffValue &= ~MVALUE_ISBYTE;
    break;

    case '<':
    GetByte();
    ffBuffer = ffValue & MVALUE_FORCEBITS;
    v = (ALU_PVR(PRIO_LOWBYTEOF) & 255);
    ffValue &= ~MVALUE_FORCEBITS;
    ffValue |= MVALUE_ISBYTE | ffBuffer;
    break;

    case '>':
    GetByte();
    ffBuffer = ffValue & MVALUE_FORCEBITS;
    v = ((ALU_PVR(PRIO_HIGHBYTEOF) >> 8) & 255);
    ffValue &= ~MVALUE_FORCEBITS;
    ffValue |= MVALUE_ISBYTE | ffBuffer;
    break;

    case '^':
    GetByte();
    ffBuffer = ffValue & MVALUE_FORCEBITS;
    v = ((ALU_PVR(PRIO_BANKBYTEOF) >> 16) & 255);
    ffValue &= ~MVALUE_FORCEBITS;
    ffValue |= MVALUE_ISBYTE | ffBuffer;
    break;

    case '"':
    /*FALLTHROUGH*/

    case '\'':
    v = ALU_ReadCharValue();
    break;

    case '%':
    v = ALU_ReadBinaryValue();
    break;

    case '$':
    v = ALU_ReadHexadecimalValue();
    break;

    case '&':
    v = ALU_ReadOctalValue();
    break;

    case '*':
    v = ALU_ReadProgramCounter();
    break;

    case '.':
    v = ALU_ReadLocalLabel();
    break;

    case '(':
    v = ALU_ReadSublevelValue();
    break;

    case ')':/* this makes "()" throw a syntax error */
    Message(pseSyntax, EERROR);
    break;

    default:/* other characters (of decimal values or global labels) */
#ifdef FDEBUG
    printf("{o}");
#endif
    if((GotByte > 47) && (GotByte < 58)) {
      v = ALU_ReadDecimalValue();
    } else {
      if(pFlagTable[(u_char) GotByte] & MBYTEILLEGAL) {
        v = 0;/* illegal character read - so don't go on */
        fEnd = TRUE;
      } else v = ALU_ReadGlobalOrNOT();
    }
  }
/* Second part:
 * Start reading operators and operands in a loop.*/
  while(!fEnd) {
    fEnd = TRUE;/* if nothing happens, stop this recursion level */
    /* if there is no operator present, try to read one */
    if(hOp_Now == HOP_NONE) hOp_Now = ALU_PVR_GetOp();
    if((hOp_Now == HOP_RIGHTPARENTHESIS) && (Prio_end == PRIO_SUBVALUE)) {
      hOp_Now = HOP_NONE;
    }
    if((hOp_Now != HOP_NONE) && (hOp_Now != HOP_RIGHTPARENTHESIS)) {
      /* if there is an operator present now, act upon it */
      fWork = TRUE;
      switch(hOp_Now) {

        case HOP_POWEROF:
        if(PRIO_POWEROF > Prio_end) {
          fEnd = FALSE;
          t = ALU_PVR(PRIO_POWEROF);
          if(t >= 0) {
            v = (Value) pow((double) v, (double) t);
          } else {
            if(ffValue & MVALUE_DEFINED) Message(pseNegExp, EERROR);
            v = 0;
          }
        }
        break;

        case HOP_MULTIPLY:
        if(PRIO_MULTIPLY > Prio_end) {
          fEnd = FALSE;
          v *= ALU_PVR(PRIO_MULTIPLY);
        }
        break;

        case HOP_DIVIDE:
        if(PRIO_DIVIDE > Prio_end) {
          fEnd = FALSE;
          t = ALU_PVR(PRIO_DIVIDE);
          if(t) {
            v /= t;
          } else {
            if(ffValue & MVALUE_DEFINED) Message(pseDivZero, EERROR);
            v = 0;
          }
        }
        break;

        case HOP_MODULO:
        if(PRIO_MODULO > Prio_end) {
          fEnd = FALSE;
          t = ALU_PVR(PRIO_MODULO);
          if(t) {
            v %= t;
          } else {
            if(ffValue & MVALUE_DEFINED) Message(pseDivZero, EERROR);
            v = 0;
          }
        }
        break;

        case HOP_ADD:
        if(PRIO_ADD > Prio_end) {
          fEnd = FALSE;
          v += ALU_PVR(PRIO_ADD);
        }
        break;

        case HOP_SUBTRACT:
        if(PRIO_SUBTRACT > Prio_end) {
          fEnd = FALSE;
          v -= ALU_PVR(PRIO_SUBTRACT);
        }
        break;

        case HOP_SL:
        if(PRIO_SL > Prio_end) {
          fEnd = FALSE;
          v = (v << ALU_PVR(PRIO_SL));
        }
        break;

        case HOP_LSR:
        if(PRIO_LSR > Prio_end) {
          fEnd = FALSE;
          v = (v >> ALU_PVR(PRIO_LSR));
        }
        break;

        case HOP_LE:
        if(PRIO_LE > Prio_end) {
          fEnd = FALSE;
          v = (v <= ALU_PVR(PRIO_LE));
        }
        break;

        case HOP_LT:
        if(PRIO_LT > Prio_end) {
          fEnd = FALSE;
          v = (v < ALU_PVR(PRIO_LT));
        }
        break;

        case HOP_GE:
        if(PRIO_GE > Prio_end) {
          fEnd = FALSE;
          v = (v >= ALU_PVR(PRIO_GE));
        }
        break;

        case HOP_GT:
        if(PRIO_GT > Prio_end) {
          fEnd = FALSE;
          v = (v > ALU_PVR(PRIO_GT));
        }
        break;

        case HOP_NOTEQUAL:
        if(PRIO_NOTEQUAL > Prio_end) {
          fEnd = FALSE;
          v = (v != ALU_PVR(PRIO_NOTEQUAL));
        }
        break;

        case HOP_EQUALS:
        if(PRIO_EQUALS > Prio_end) {
          fEnd = FALSE;
          v = (v == ALU_PVR(PRIO_EQUALS));
        }
        break;

        case HOP_AND:
        if(PRIO_AND > Prio_end) {
          fEnd = FALSE;
          v &= ALU_PVR(PRIO_AND);
        }
        break;

        case HOP_EXOR:
        if(PRIO_EXOR > Prio_end) {
          fEnd = FALSE;
          v ^= ALU_PVR(PRIO_EXOR);
        }
        break;

        case HOP_OR:
        if(PRIO_OR > Prio_end) {
          fEnd = FALSE;
          v |= ALU_PVR(PRIO_OR);
        }
        break;

        default:
        Message(pseUkOp, EERROR);
        fEnd = TRUE;
      }
      ffValue &= ~MVALUE_ISBYTE;
    }
  }
  nLevel -= 1;/* decrease recursion counter */
  VP_Work = fWork;
#ifdef FDEBUG
  printf("%d_", v);
#endif
  return(v);
}

/*
 * Utility routine for reading an operator
 */
static int ALU_PVR_GetOp() {
  /* no operator buffered, so try to read one */
  SKIPSPACE;
  switch(GotByte) {

    case '^':/* "to the power of" */
    GetByte();
    return(HOP_POWEROF);

    case '*':/* "times" */
    GetByte();
    return(HOP_MULTIPLY);

    case '/':/* "divided by" */
    GetByte();
    return(HOP_DIVIDE);

    case '%':/* "modulo" */
    GetByte();
    return(HOP_MODULO);

    case '+':/* "plus" */
    GetByte();
    return(HOP_ADD);

    case '-':/* "minus" */
    GetByte();
    return(HOP_SUBTRACT);

    case '=':/* "equals" */
    GetByte();
    return(HOP_EQUALS);

    case '&':/* "AND" */
    GetByte();
    return(HOP_AND);

/*
 * This part is commented out because there is no "EXOR" character defined
 *
    case '#':/ * "EXOR" * /
    GetByte();
    return(HOP_EXOR);
*/

    case '|':/* "OR" */
    GetByte();
    return(HOP_OR);

    case '<':/* check for "<", "<=", "<<", "<>" */
    return(ALU_PVR_LT());

    case '>':/* check for ">", ">=", ">>", "><" */
    return(ALU_PVR_HT());

    case '!':/* check for "!=" */
    return(ALU_PVR_EM());

    case ')':/* end of sublevel is handled like an operator*/
    return(ALU_PVR_KZ());

    default:/* check string version of operators */
    if((pFlagTable[(u_char) GotByte] & MBYTEILLEGAL) == 0) {
      int len;
      ListItem *p;
      len = Stream_ReadKeyword(&StringItem.String[0], LSMAX, FALSE);
      /* (now: GotByte = illegal character) */
      /* search for list item */
      p = Struct_Search(&StringItem, HTYPE_OPERATOR, len, 0);
      if(p) return(p->Data.Bytes.Code);
      Message(pseUkOp, EERROR);
    }
    return(HOP_NONE);
  }
}

/*
 * Routine for reading a subvalue (after left parenthesis)
 */
static Value ALU_ReadSublevelValue() {
  nParenth++;
  GetByte();/* start sublevel on next char */
  return(ALU_PVR(PRIO_SUBVALUE));
}

/*
 * Routine for finishing a subvalue (after right parenthesis)
 */
static int ALU_PVR_KZ() {
  GetByte();/* proceed with next char */
  if(nParenth) {
    nParenth--;
  } else {
    Message(pseTooClosed, EERROR);
  }
  return(HOP_RIGHTPARENTHESIS);
}

/*
 * Routine for reading the program counter
 */
static Value ALU_ReadProgramCounter() {
#ifdef FDEBUG
  printf("{PC=%d}", PC_CPU);
#endif
  GetByte();/* proceed with next char */
  if(fPCdefined == FALSE) {
    Message(pseNoPC, EERROR);
    fPCdefined = TRUE;
  }
  ffValue |= MVALUE_EXISTS;
  return(PC_CPU);
}

/*
 * Routine for reading a local label
 */
static Value ALU_ReadLocalLabel() {
  int len = Stream_ReadKeyword(&StringItem.String[2], LSMAX - 2, TRUE);

  /* offsets 0 and 1 are for zone */
  /* (now: GotByte = illegal character) */
  if(len) return(ALU_GetLabelValue(Context[nContext].nZone_Now, len));
  return(0);
}

/*
 * Routine for reading a global label (or "NOT")
 */
static Value ALU_ReadGlobalOrNOT() {
  int len = Stream_ReadKeyword(&StringItem.String[2], LSMAX - 2, FALSE);

  /* offsets 0 and 1 are for zone */
  /* (now: GotByte = illegal character) */
  if(len == 3) {
    /* Check for NOT. Okay, it's hardcoded, but so what ? Sue me... */
    if(StringItem.String[2] == 'N') {
      if(StringItem.String[3] == 'O') {
        if(StringItem.String[4] == 'T') {
          ffValue &= ~MVALUE_ISBYTE;
          return(~ALU_PVR(PRIO_NOT));
        }
      }
    }
  }
  return(ALU_GetLabelValue(NZONE_GLOBAL, len));
}

/*
 * Routine for parsing "<"-operators:  <,  <=,  <<,  <>
 */
static int ALU_PVR_LT() {/* GotByte = "<" */
  switch(GetByte()) {

    case '=':/* "<=", less or equal */
    GetByte();
    return(HOP_LE);

    case '<':/* "<<", shift left */
    GetByte();
    return(HOP_SL);

    case '>':/* "<>", not equal */
    GetByte();
    return(HOP_NOTEQUAL);

    default:/* "<", less than (no bytefetch !) */
    return(HOP_LT);
  }
}

/*
 * Routine for parsing ">"-operators:  >,  >=,  >>,  ><
 */
static int ALU_PVR_HT() {/* GotByte = ">" */
  switch(GetByte()) {

    case '=':/* ">=", more or equal */
    GetByte();
    return(HOP_GE);

    case '<':/* "><", not equal */
    GetByte();
    return(HOP_NOTEQUAL);

    case '>':/* ">>", LSR */
    GetByte();
    return(HOP_LSR);

    default:/* ">", greater than (no bytefetch !) */
    return(HOP_GT);
  }
}

/*
 * Routine for parsing "!="-operator
 */
static int ALU_PVR_EM() {/* GotByte = "!", expecting "=" */
  if(GetByte() == '=') {
    GetByte();
    return(HOP_NOTEQUAL);
  } else {
    Message(pseSyntax, EERROR);
    return(HOP_NONE);
  }
}

/*
 * Routine for parsing quoted characters. The character will be converted using
 * the current conversion table.
 */
static Value ALU_ReadCharValue() {/* GotByte = Quote */
  Value v = (Value) ' ';
  char  c = GetByte();

  if(QuoteChar != NO_QUOTE_CHAR) {
    v = (Value) ConvertChar(c, hCodeTable_Now);
    GetByte();/* Read closing quote (hopefully) */
    if(QuoteChar == NO_QUOTE_CHAR) {
      GetByte();/* If length == 1, proceed with next byte */
    } else Message(pseTooLong, EERROR);/* If longer than one character */
  } else Message(pseMissingString, EERROR);/* If shorter than one char */
  ffValue |= MVALUE_EXISTS;
  return(v);/* GotByte = char following closing quote */
}

/*
 * Routine for parsing a binary value. Apart from "0" and "1", it also accepts
 * characters "." and "#", this is much more readable. The current value is
 * returned as soon as a character is read that is none of those given above.
 */
static Value ALU_ReadBinaryValue() {/* GotByte = "%" */
  Value v     =  0;
  int   fCont = TRUE,
        c     = -1;

  do {
    c++;
    switch(GetByte()) {

      case '0':
      case '.':
      v = v << 1;
      break;

      case '1':
      case '#':
      v = (v << 1) + 1;
      break;

      default:
      fCont = FALSE;
    }
  } while(fCont);
  if(c > 8) {
    if(c > 16) {
      if(v < 65536) ffValue |= MVALUE_FORCE24;
    } else {
      if(v <   256) ffValue |= MVALUE_FORCE16;
    }
  }
  ffValue |= MVALUE_EXISTS;
  return(v);/* GotByte = non-binary character */
}

/*
 * Routine for parsing a hexadecimal value. It accepts "0" to "9", "a" to "f"
 * and "A" to "F". Capital letters will be converted to lowercase letters using
 * the flagtable. The current value is returned as soon as a character is read
 * that is none of those given above.
 */
static Value ALU_ReadHexadecimalValue() {/* GotByte = "$" or "x" */
  Value v = 0;
  char  byte;
  int   fCont,
        c = -1;

  do {
    c++;
    fCont = FALSE;
    byte = GetByte();
    /*  first, convert "A-F" to "a-f" */
    if(pFlagTable[(u_char) byte] & MBYTEUPPERCASE) byte |= 32;
    if((byte > 47) && (byte < 58)) {
      /* add values of digits */
      fCont = TRUE;
      v = (v << 4) + (byte & 15);
    }
    if((byte > 96) && (byte < 103)) {
      /* add values of characters "a-f" */
      fCont = TRUE;
      v = (v << 4) + (byte & 15) + 9;
    }
  } while(fCont);
  if(c > 2) {
    if(c > 4) {
      if(v<65536) ffValue |= MVALUE_FORCE24;
    } else {
      if(v<  256) ffValue |= MVALUE_FORCE16;
    }
  }
  ffValue |= MVALUE_EXISTS;
  return(v);/* GotByte = non-hexadecimal character */
}

/*
 * Routine for parsing an octal value. It accepts characters "0" to "7". The
 * current value is returned as soon as a character is read that is none of
 * those given above.
 */
static Value ALU_ReadOctalValue() {/* GotByte = "&" */
  Value v = 0;
  int   c = 0;

  GetByte();
  while((GotByte > 47) && (GotByte < 56)) {
    v = (v << 3) + (GotByte & 7);
    c++;
    GetByte();
  }
  if(c > 3) {
    if(c > 6) {
      if(v<65536) ffValue |= MVALUE_FORCE24;
    } else {
      if(v<  256) ffValue |= MVALUE_FORCE16;
    }
  }
  ffValue |= MVALUE_EXISTS;
  return(v);/* GotByte = non-octal character */
}

/*
 * Routine for parsing a decimal value. It accepts characters "0" to "9". The
 * current value is returned as soon as a character is read that is none of
 * those given above. Unlike the others, the "decimal" routine expects the
 * first digit to be read already, because decimal values don't use any prefix.
 * If the first two digits are "0x", this routine branches to the one for
 * parsing hexadecimal values.
 */
static Value ALU_ReadDecimalValue() {/* GotByte = first digit */
  Value v = (GotByte & 15);

  GetByte();
  if((v == 0) && (GotByte == 'x')) return(ALU_ReadHexadecimalValue());
  while((GotByte > 47) && (GotByte < 58)) {
    v *= 10;
    v += (GotByte & 15);
    GetByte();
  }
  ffValue |= MVALUE_EXISTS;
  return(v);/* GotByte = non-decimal character */
}

/*
 * Lookup (and create, if necessary) label list item and return value. The
 * stringbuffer holds the label's name (its length given by "len") and "Zone"
 * its zone.
 */
static Value ALU_GetLabelValue(Sixteen Zone, int len) {
  ListItem *p;
  int       Flags;

  p = Struct_GetPreparedLabel(Zone, len, 0);
  /* If the label was produced just now, we have to mark it as unsure */
  if(fMadeItem) p->Data.Label.Flags |= MVALUE_UNSURE;
  Flags = p->Data.Label.Flags;
#ifdef FDEBUG
  printf("Label's flags: %x\n", Flags);
#endif
  ffValue &= (Flags | ~MVALUE_DEFINED);
  ffValue |= (Flags & (MVALUE_UNSURE | MVALUE_FORCEBITS));
  ffValue |= MVALUE_EXISTS;
  return(p->Data.Label.Value);
}
