/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Flow control stuff (macro calls, loops, etc.)
 */

#include "flowpo.h"

/*
 * Start offset assembly
 */
static void FlowPO_pseudopc() {
  PC_CPU = ALU_GetValue_Strict();
  EnsureEOL();
}

/*
 * End offset assembly
 */
static void FlowPO_realpc() {
  PC_CPU = PC_Mem;/* deactivate offset assembly */
  EnsureEOL();
}

/*
 * Switch to new zone ("!zone" or "!zn")
 */
static void FlowPO_zone() {
  int len;

  Context[nContext].nZone_Now    = ++nZone_Max;
  Context[nContext].pSourceTitle = psUntitled;
  SKIPSPACE;
  if(GotByte) {
    len = Stream_ReadKeyword(&pTitle[nContext][0], LSMAX, FALSE);
    if(len) Context[nContext].pSourceTitle = &pTitle[nContext][0];
    EnsureEOL();
  }
}

/*
 * Start subzone ("!subzone" or "!sz"). Has to be re-entrant.
 */
static void FlowPO_subzone() {
  int len,
      Reason;

  /* Initialize new context */
  Flow_NewContext();
  Context[nContext].nLines       = Context[nContext-1].nLines;
  Context[nContext].nZone_Now    = ++nZone_Max;
  Context[nContext].hByteSource  = Context[nContext-1].hByteSource;
  Context[nContext].u.hFile      = Context[nContext-1].u.hFile;
  Context[nContext].pSourceFile  = Context[nContext-1].pSourceFile;
  Context[nContext].pSourceTitle = psUntitled;
  Context[nContext].pSourceType  = Context[nContext-1].pSourceType;
  Context[nContext].OldFileByte  = 0;
  Context[nContext].OldRawByte   = ' ';/* Don't increase line number */
  SKIPSPACE;
  /* Check for zone title */
  if((pFlagTable[(u_char) GotByte] & MBYTEILLEGAL) == 0) {
    len = Stream_ReadKeyword(&pTitle[nContext][0], LSMAX, FALSE);
    if(len) Context[nContext].pSourceTitle = &pTitle[nContext][0];
    SKIPSPACE;
  }
  if(GotByte == '{' ) {
    ParseBlock();
    Reason = EndReason;
    EndReason = RNONE;/* Clear global variable */
    if(Reason != RRIGHTBRACE) Message(pseeRightBrace, EERROR);
    GetByte();
    EnsureEOL();
  } else Message(pseeLeftBrace, ESERIOUS);
  /* Adjust old context's data */
  Context[nContext-1].nLines = Context[nContext].nLines;
  Context[nContext-1].OldFileByte = 0;
  Context[nContext-1].OldRawByte  = 0;/* Increase line number */
  nContext--;/* Old context */
}

/*
 * End of source file ("!endoffile" or "!eof")
 */
static void FlowPO_eof() {
  EndReason = RENDOFFILE;
  EnsureEOL();
}

/*
 * Include source file ("!source" or "!src"). Has to be re-entrant.
 */
static void FlowPO_source() {/* GotByte = illegal character */
  char pnfSub[LNPMAX + 1];/* file name */
  int  len = Stream_ReadFilename(pnfSub, LNPMAX, TRUE);

  if(len) {
    if(Stream_OpenSub(pnfSub)) {
      /* Initialize new context */
      Flow_NewContext();
      Context[nContext].nLines       = 0;
      Context[nContext].nZone_Now    = Context[nContext-1].nZone_Now;
      Context[nContext].hByteSource  = BYTESOURCE_FILE;
      Context[nContext].u.hFile      = hfSub;
      Context[nContext].pSourceFile  = pnfSub;
      Context[nContext].pSourceTitle = Context[nContext-1].pSourceTitle;
      Context[nContext].pSourceType  = Context[nContext-1].pSourceType;
      Context[nContext].OldFileByte  = 0;
      Context[nContext].OldRawByte   = 0;
      /* Parse block and check end reason */
      ParseBlock();
      if(EndReason != RENDOFFILE) Message(pseeEndOfFile, EERROR);
      EndReason = RNONE;/* Clear global variable */
      Stream_CloseSub();
      nContext--;/* Old context */
    }
    EnsureEOL();
  } else {
    SkipRest();
  }
}

/*
 * Conditional assembly ("!ifdef")
 */
static void FlowPO_ifdef() {/* GotByte = illegal character */
  int       len;
  ListItem *p;
  Sixteen   Zone = NZONE_GLOBAL;

  SKIPSPACE;
  if(GotByte == '.') {
    Zone = Context[nContext].nZone_Now;
    GetByte();
  }
  len = Stream_ReadKeyword(&StringItem.String[2], LSMAX - 2, FALSE);
  if(len) {
    p = Struct_GetZoned(Zone, len, HTYPE_LABEL, FALSE);
    if(p) {
      if((p->Data.Label.Flags & MVALUE_DEFINED) == 0) p = NULL;
    }
    SKIPSPACE;
    if(GotByte == '{') {
      FlowPO_2Blocks((int) (p != NULL));
    } else {
      if(p == NULL) SkipRest();
    }
  }
}

/*
 * Conditional assembly ("!if"). Has to be re-entrant.
 */
static void FlowPO_if() {/* GotByte = illegal character */
  Value v = ALU_GetValue_Strict();

  if(GotByte == '{') FlowPO_2Blocks((int) v);
  else               Message(pseeLeftBrace, ESERIOUS);
}

/*
 * Parse {block} [else {block}]
 */
static void FlowPO_2Blocks(int f) {
  ListItem *p;
  int       len,
            Reason;

  if(f) {
    ParseBlock();
  } else {
    Flow_StoreBlock(NULL);
  }
  Reason = EndReason;
  EndReason = RNONE;/* Clear global variable */
  if(Reason == RRIGHTBRACE) {
    /* check for "else" */
    NEXTANDSKIPSPACE;
    if(GotByte) {
      len = Stream_ReadKeyword(&StringItem.String[0], LSMAX, FALSE);
      if(len) {
        /* Search for list item. Don't create */
        String_ToLower(&StringItem, len);
        p = Struct_Search(&StringItem, HTYPE_ELSE, len, 0);
        if(p) {
          if(p->Data.Bytes.Code == ID_ELSE) {
            SKIPSPACE;
            if(GotByte == '{') {
              if(f) {
                Flow_StoreBlock(NULL);
              } else {
                ParseBlock();
              }
              Reason = EndReason;
              EndReason = RNONE;/* Clear global variable */
              if(Reason != RRIGHTBRACE) Message(pseeRightBrace, EERROR);
              GetByte();
            } else Message(pseeLeftBrace, ESERIOUS);
          } else Message(pseSyntax, EERROR);
        } else Message(pseSyntax, EERROR);
      }
      EnsureEOL();
    }
  } else Message(pseeRightBrace, EERROR);
}

/*
 * Looping assembly ("!for"). Has to be re-entrant.
 */
static void FlowPO_for() {/* GotByte = illegal character */
  Value     vEnd,
            vNow;
  int       line,
            len,
            fb,
            ffvEnd;
  char     *pb,
           *pbOwn;
  ListItem *p;
  Sixteen   Zone = NZONE_GLOBAL;

  line = Context[nContext].nLines;/* Remember line number */
  SKIPSPACE;
  if(GotByte == '.') {
    Zone = Context[nContext].nZone_Now;
    GetByte();
  }
  len = Stream_ReadKeyword(&StringItem.String[2], LSMAX - 2, FALSE);
  /* (now: GotByte = illegal char) */
  if(len) {
    fb = Mnemo_GetForceBit();/* skips spaces */
    p = Struct_GetPreparedLabel(Zone, len, fb);
    if(fb) {
      p->Data.Label.Flags &= ~MVALUE_FORCEBITS;
      p->Data.Label.Flags |= fb;
    }
    if(GotByte == ',') {
      GetByte();/* proceed with next char */
      vEnd = ALU_GetValue_Strict();
/* prepare flags, size of *end* value doesn't matter for *all* loops */
      ffvEnd = ffValue & ~MVALUE_FORCEBITS;
      if(vEnd >= 0) {
        if(GotByte == '{') {
          pb = Flow_StoreBlock(pFlowBuffer);/* Read block to buffer */
          EndReason = RNONE;/* Clear global variable */
          if(GetByte() == 0) {/* proceed with next char */
            pbOwn = malloc(pb - pFlowBuffer);/* Get memory block */
            if(pbOwn) {
              memcpy(pbOwn, pFlowBuffer, pb - pFlowBuffer);
              /* Initialize new context */
              Flow_NewContext();
              Context[nContext].nZone_Now    = Context[nContext-1].nZone_Now;
              Context[nContext].hByteSource  = BYTESOURCE_RAM;
              Context[nContext].pSourceFile  = Context[nContext-1].pSourceFile;
              Context[nContext].pSourceTitle = Context[nContext-1].pSourceTitle;
              Context[nContext].pSourceType  = Context[nContext-1].pSourceType;

              if(vEnd) {/* skip loop if counter = 0 */
                vNow = 1;
                while(vNow <= vEnd) {
                  /* set counter */
                  SetLabelValue(p, vNow, ffvEnd, TRUE);
                  /* now actually do some work */
                  Context[nContext].OldFileByte  = 0;
                  Context[nContext].OldRawByte = ' ';/* Don't inc line count */
                  Context[nContext].u.pRAM = pbOwn;/* Begin at beginning */
                  Context[nContext].nLines = line;/* Restore line number */
                  ParseBlock();
                  if(EndReason == RENDOFBLOCK) {
                    EndReason = RNONE;/* Clear global variable */
                  } else Message(pseeRightBrace, ESERIOUS);
                  vNow++;/* increase counter */
                }
              } else SetLabelValue(p, 0, ffvEnd, TRUE);/* set counter */
              free(pbOwn);/* Free memory */
              nContext--;/* Old context */
              EnsureEOL();
            } else Message(pseNoMemLeft, ESERIOUS);
          } else Message(pseSyntax, ESERIOUS);
        } else Message(pseeLeftBrace, ESERIOUS);
      } else Message(pseTooBig, ESERIOUS);/* No negative loops ! */
    } else {
      Message(pseSyntax, EERROR);
      SkipRest();
    }
  }
}

/*
 * Looping assembly ("!do"). Has to be re-entrant.
 */
static void FlowPO_do() {/* GotByte = illegal character */
  Value  v;
  int    line,
         fGoOn = TRUE;
  char  *pb,
        *pbOwn,
         id;

  line = Context[nContext].nLines;/* Remember line number */
  pb = Flow_StoreCondition(pFlowBuffer);/* Read head to buffer */
  if(GotByte == '{') {
    pb = Flow_StoreBlock(pb);/* Read block to buffer */
    EndReason = RNONE;/* Clear global variable */
    GetByte();/* proceed with next char */
    pb = Flow_StoreCondition(pb);/* Read tail to buffer */
    if(GotByte == 0) {
      pbOwn = malloc(pb - pFlowBuffer);/* Get memory for this incarnation */
      if(pbOwn) {
        memcpy(pbOwn, pFlowBuffer, pb - pFlowBuffer);
        /* Initialize new context */
        Flow_NewContext();
        Context[nContext].nZone_Now    = Context[nContext-1].nZone_Now;
        Context[nContext].hByteSource  = BYTESOURCE_RAM;
        Context[nContext].pSourceFile  = Context[nContext-1].pSourceFile;
        Context[nContext].pSourceTitle = Context[nContext-1].pSourceTitle;
        Context[nContext].pSourceType  = Context[nContext-1].pSourceType;
        Context[nContext].OldFileByte  = 0;
        Context[nContext].OldRawByte   = ' ';/* Don't increase line number */
        do {
          /* Start of loop */
          Context[nContext].u.pRAM = pbOwn;/* Begin at beginning */
          Context[nContext].nLines = line;/* Use remembered line number */
          /* Check head condition */
          id = GetByte();/* get condition type (until/while) */
          GetByte();/* proceed with next char */
          v = ALU_GetValue_Strict();
          if(EndReason == RENDOFBLOCK) {
            EndReason = RNONE;/* Clear global variable */
            if(id == ID_UNTIL) v = !v;
            if(v) {
              Context[nContext].nLines = line-1;/* Kluge !! */
              ParseBlock();
              if(EndReason == RENDOFBLOCK) {
                Context[nContext].nLines-=1;/* Kluge !! */
                EndReason = RNONE;/* Clear global variable */
                /* Check tail condition */
                id = GetByte();/* get condition type (until/while) */
                GetByte();/* proceed with next char */
                v = ALU_GetValue_Strict();
                if(EndReason == RENDOFBLOCK) {
                  EndReason = RNONE;/* Clear global variable */
                  if(id == ID_UNTIL) v = !v;
                  if(v == 0) fGoOn = FALSE;
                } else Message(pseSyntax, ESERIOUS);
              } else Message(pseeRightBrace, ESERIOUS);
            } else fGoOn = FALSE;
          } else Message(pseSyntax, ESERIOUS);
        } while(fGoOn);
        free(pbOwn);/* Free memory */
        nContext--;/* Old context */
        EnsureEOL();
      } else Message(pseNoMemLeft, ESERIOUS);
    } else Message(pseSyntax, ESERIOUS);
  } else Message(pseeLeftBrace, ESERIOUS);
}

/*
 * Try to read a condition and store it at the given location. Add separator.
 */
static char *Flow_StoreCondition(char *pb) {
  ListItem *p;
  int       len;

  SKIPSPACE;
  if((GotByte == '{') || (GotByte == 0)) {/* Accept both head *and* tail */
    /* Write pseudo condition (always TRUE) into buffer, so we don't have */
    /* to worry about nonexistant conditions */
    *(pb++) = ID_WHILE;
    *(pb++) = '1';
  } else {
    /* Write given condition into buffer */
    len = Stream_ReadKeyword(&StringItem.String[0], LSMAX, FALSE);
    if(len) {
      /* Search for list item. Don't create */
      String_ToLower(&StringItem, len);
      p = Struct_Search(&StringItem, HTYPE_DOLOOP, len, 0);
      if(p) {
        *(pb++) = p->Data.Bytes.Code;/* store condition type handle */
        while((GotByte != '{') && GotByte) {
          if(pb == &pFlowBuffer[MAXBLOCKSIZE])
            Message(pseNoMemLeft, ESERIOUS);
          *(pb++) = GotByte;
          GetByte();
        }
      } else Message(pseSyntax, EERROR);
    }
  }
  *(pb++) = SEPARATOR;/* Terminate / separate */
  return(pb);
}

/*
 * Macro definition ("!macro").
 */
static void FlowPO_macro() {
  ListItem    *p;
  char        *pb,
              *pSource,
               byte;
  MacroStruct *pMacro,
              *pMacroGlobal;
  Sixteen      Zone         = NZONE_GLOBAL;
  int          len,
               line         = Context[nContext].nLines;/* Remember line nr */

  pMacroGlobal = &MacroBuffer;/* Macro/flow buffer */
  pb = &pMacroGlobal->Chars[0];/* pb points to start of sequential data */
  SKIPSPACE;
  if(ffPass & FPASSDOONCE) {
    /* Parse macro definition */
    if(GotByte == '.') {
      Zone = Context[nContext].nZone_Now;
      GetByte();
    }
    len = Stream_ReadKeyword(&StringItem.String[2], LSMAX - 2, FALSE);
    /* (now: GotByte = illegal char) */
    if(len) {
      p = Struct_GetZoned(Zone, len, HTYPE_MACRO, TRUE);
      if(fMadeItem) {
        /* Copy file name to buffer (includes terminator) */
        pSource = Context[nContext].pSourceFile;
        do {
          byte = *(pSource++);
          *(pb++) = byte;
          if(pb == &pFlowBuffer[MAXBLOCKSIZE])
            Message(pseNoMemLeft, ESERIOUS);
        } while(byte);
        /* Copy parameters to buffer */
        SKIPSPACE;
        while((GotByte != '{') && GotByte) {
          *(pb++) = GotByte;
          if(pb == &pFlowBuffer[MAXBLOCKSIZE])
            Message(pseNoMemLeft, ESERIOUS);
          GetByte();
        }
        if(GotByte) {
          *(pb++) = SEPARATOR;/* Terminate / separate */
          Context[nContext].OldRawByte = ' ';/* Kluge to skip spaces */
          pb = Flow_StoreBlock(pb);/* Store macro body */
          if(EndReason == RRIGHTBRACE) {
            EndReason = RNONE;/* Clear global variable */
            GetByte();/* Proceed with next character */
            /* Get own memory for macro */
            pMacro = ALLOC_TRY(pb - (char *) (&MacroBuffer));
            /* Set up macro structure in new memory block */
            memcpy(&pMacro->Chars[0], pFlowBuffer, pb - pFlowBuffer);
            pMacro->nLinesDef = line;
            /* Remember pointer to new memory block in list item */
            p->Data.Body = pMacro;
          } else Message(pseeRightBrace, ESERIOUS);
        } else Message(pseeLeftBrace, ESERIOUS);
      } else Message(pseMacroTwice, ESERIOUS);
    }
  } else {
    /* Skip macro definition */
    while((GotByte != '{') && GotByte) GetByte();
    if(GotByte) {
      Flow_StoreBlock(NULL);
      EndReason = RNONE;/* Clear global variable */
      GetByte();/* Proceed with next character */
    } else Message(pseeLeftBrace, EERROR);
  }
  EnsureEOL();
}

/*
 * Macro call ("+<macroname>"). Has to be re-entrant.
 */
static void Flow_MacroCall() {/* GotByte = "+" */
  ListItem    *p;
  int          TempReason,
               a,
               len,
               nParaC = 0,/* Number of parameters given in call */
               nParaD = 0;/* Number of parameters given in definition */
  MacroStruct *pMacro;
  char        *pb;
  Sixteen      newZone,
               oldZone = NZONE_GLOBAL;

  GetByte();
  if(GotByte == '.') {
    /* If macro is local, use current zone value */
    oldZone = Context[nContext].nZone_Now;
    GetByte();
  }
  len = Stream_ReadKeyword(&StringItem.String[2], LSMAX - 2, FALSE);
  /* (now: GotByte = illegal char) */
  if(len) {
    p = Struct_GetZoned(oldZone, len, HTYPE_MACRO, FALSE);
    if(p) {
      pMacro = p->Data.Body;/* Get pointer to macro */
      /* Skip file name stored in macro */
      pb = &pMacro->Chars[0];
      while(*(pb++));
      /* Read parameters of call */
      SKIPSPACE;
      if(GotByte) {
        do {
          /* Read parameters up to end of statement, store in static array */
          if(nParaC == MAXMACROPARAMETERS) Message(pseTooManyMP, ESERIOUS);
          MacroParaValue[nParaC] = ALU_GetValue_Medium();
          MacroParaFlags[nParaC] = ffValue;
          nParaC++;
        } while(Stream_Comma());
      }
      /* nParaC = number of parameters in call */
      TempReason = EndReason;/* buffer current end reason */

      /* Initialize new context */
      Flow_NewContext();
      Context[nContext].nLines       = pMacro->nLinesDef;/* Line of def. */
      Context[nContext].nZone_Now    = ++nZone_Max;
      Context[nContext].hByteSource  = BYTESOURCE_RAM;
      Context[nContext].u.pRAM       = pb;
      Context[nContext].pSourceFile  = &pMacro->Chars[0];
      Context[nContext].pSourceTitle = &p->String[2];/* Macro title */
      Context[nContext].pSourceType  = psMacro;
      Context[nContext].OldFileByte  = 0;
      Context[nContext].OldRawByte   = ' ';/* Don't increase line number */
      /* Search for parameter labels and store pointers in static array */
      GetByte();/* Get first byte of "new" context */
      if(GotByte) {
        do {
          SKIPSPACE;
          newZone = NZONE_GLOBAL;
          if(GotByte == '.') {
            /* If parameter is local (probably), use new zone value */
            newZone = Context[nContext].nZone_Now;
            GetByte();
          }
          len = Stream_ReadKeyword(&StringItem.String[2], LSMAX - 2, FALSE);
          /* (now: GotByte = illegal char) */
          if(len) {
            if(nParaD == MAXMACROPARAMETERS) Message(pseTooManyMP, ESERIOUS);
            MacroParameter[nParaD] =
              Struct_GetZoned(newZone, len, HTYPE_LABEL, TRUE);
            nParaD++;
          }
        } while(Stream_Comma());
        EnsureEOL();
      }
      Context[nContext].OldRawByte = ' ';/* Don't increase line number */
      EndReason = RNONE;/* Clear global variable */
      /* Compare number of parameters in call and definition */
      if(nParaC == nParaD) {
        /* Set parameters */
        for(a = nParaC-1; a >=0; a--) {
          MacroParameter[a]->Data.Label.Value = MacroParaValue[a];
          MacroParameter[a]->Data.Label.Flags = MacroParaFlags[a];
        }
        /* Actually *parse* the macro body */
        ParseBlock();
        if(EndReason == RENDOFBLOCK) {
          EndReason = TempReason;/* Restore global variable */
          nContext--;/* Old context */
        } else Message(pseeRightBrace, ESERIOUS);
      } else {
        /* Show two error messages, pointing to both definition and use */
        Message(pseWrongNr, EERROR);
        nContext--;/* Old context */
        Message(pseWrongNr, ESERIOUS);
      }
    } else Message(pseUkMacro, EERROR);
  }
  EnsureEOL();
}

/*
 * Shared utility routines
 */

/*
 * Check whether context depth is exceeded and increase counter if not.
 */
static void Flow_NewContext() {
  if((nContext++) == MAXCONTEXTS) {
    nContext--;
    Message(pseTooDeep, ESERIOUS);
  }
}

/*
 * Read block to address (starting with next byte) and terminate. The updated
 * pointer is returned. If the end address is exceeded, an error (serious)
 * will be generated. If the given RAM pointer is NULL, the block will simply
 * be skipped.
 */
static char *Flow_StoreBlock(char *pb) {
  int  depth = 1;
  char byte;

  while(depth && (EndReason == RNONE)) {
    byte = Stream_GetRawByte();
    if(pb) {
      if(pb == &pFlowBuffer[MAXBLOCKSIZE])
        Message(pseNoMemLeft, ESERIOUS);
      *(pb++) = byte;
    }
    if((QuoteChar == NO_QUOTE_CHAR) && (byte == '{')) depth++;
    if((QuoteChar == NO_QUOTE_CHAR) && (byte == '}')) depth--;
  }
  if(depth) {
    Message(pseeRightBrace, ESERIOUS);
  } else {
    EndReason = RRIGHTBRACE;
    GotByte = 0;/* Kluge */
    if(pb) *(pb - 1) = SEPARATOR;/* add separator */
  }
  return(pb);
}
