/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Buffers and tables
 */

#include "global.h"

static char pnfOut [LNFMAX + 1];/* Filename of output file */
static char pnfDump[LNFMAX + 1];/* Filename of label dump */

/*
 * Main program
 */

int main(int argc, char *argv[]) {
  pToBeFreed_Task = NULL;
  Platform_FindLib();/* only do once */

  do {
    pToBeFreed_Try = NULL;
    /* Pre-initialise, because may be needed on error */
    nContext = 0;
    Context[0].nLines       = 0;
    Context[0].pSourceFile  = pnfTop;
    Context[0].pSourceTitle = psUntitled;
    Context[0].pSourceType  = psZone;

    ReadParameters(argc, argv);/* get command line arguments */

    DataTables_Init();/* link predefined list items to lists */

    /* First pass */
    if(lVerbosity > 1) printf(psvFirst);

    Pass(FPASSDOONCE);

    if(nNeedvalue) {
      /* If there were undefined values, another pass will have to be done.*/
      do {
        nNeedvalue_Last = nNeedvalue;/* Buffer previous error counter */

        /* Further passes */
        if(lVerbosity > 1) printf(psvFurther);
        Pass(0);
        /* keep doing passes as long as there are no real errors and the */
        /* number of undefined values is decreasing */
      } while(nNeedvalue && (nNeedvalue < nNeedvalue_Last));
    }

    if(nNeedvalue) {
      /* Extra pass for finding "NeedValue" type errors */
      if(lVerbosity > 1) printf(psvFind);
      Pass(FPASSDOERROR);
    } else {
      /* If wanted, save assembled code */
      if(ffRepeat & FREPEAT_OUTFILECHOSEN)
        Stream_Output(PC_Lowest, PC_Highest);
    }

#ifdef FDEBUG
    system("OS_File, 10, RAM:$.H, &ffd,, pHashTestTable, pHashTestTable+256");
#endif

    Stream_Dump();/* if wanted, dump labels */
    FREE_TRY();/* free memory (items) at end of try */
  } while(0);/* add repetition switch here */

  CleanExit(EXIT_SUCCESS);/* that's all folks ! */
  return(0);
}

/*
 * Main routines
 */

/*
 * The core of every pass
 */
static void Pass(int Flags) {
  pToBeFreed_Pass = NULL;
  ffPass     = Flags;/* Flags im Pass */
  nNeedvalue = 0;/* no "NeedValue" errors yet */
  nErrors    = 0;/* no errors yet */
  fPCdefined = FALSE;/* Flag for "PC defined" */
  PC_Lowest  = 0;/* must be zero'd so "!initmem" can be used before "*=" */
  PC_Highest = 0;
  SegmentStart = 0;
  SegmentMax = OUTBUFFERSIZE-1;
  pSegmentList = NULL;
  PC_inc     = 0;/* Increase PCs by this amount at end of line */
  hCodeTable_Now = HCODETABLE_RAW;/* Handle of current code table */
  /* Open toplevel file */
  if(!Stream_OpenTop()) {
    printf(pseCannotOpenTopFile);
    CleanExit(EXIT_FAILURE);
  }
  /* Initialize context */
  nContext = 0;
  Context[0].nLines       = 0;
  Context[0].nZone_Now    = NZONE_START;/* current zone is default zone */
  Context[0].hByteSource  = BYTESOURCE_FILE;
  Context[0].u.hFile      = hfTop;
  Context[0].pSourceFile  = pnfTop;
  Context[0].pSourceTitle = psUntitled;
  Context[0].pSourceType  = psZone;
  Context[0].OldFileByte  = 0;
  Context[0].OldRawByte   = 0;

  EndReason = RNONE;

  nZone_Max = NZONE_START;/* highest zone number used yet */
  GotByte = 0;
  hCPU_Now = HCPU_6502;/* CPU = 6502 as default */
  fCPU_LongA  = FALSE;/* short accu and registers */
  fCPU_LongR  = FALSE;
  fCPU_LongAb = FALSE;
  fCPU_LongRb = FALSE;

  /* do the work */
  ParseBlock();
  if(EndReason != RENDOFFILE) Message(pseeEndOfFile, EERROR);
  EndReason = RNONE;/* Clear global variable */

  String_EndSegment();
  /* Reached end of source, so end of pass... */
  Stream_CloseTop();
  FREE_PASS();
#ifdef FDEBUG
  printf("Errors: %d, Needvalues: %d\n", nErrors, nNeedvalue);
#endif
  if(nErrors) CleanExit(EXIT_FAILURE);
}

/*
 * Parse block, beginning with next byte. Has to be re-entrant.
 */
static void ParseBlock() {
  int fPC2Label,
      len;

  do {
    /* As long as there are statements */
    fPC2Label = FALSE;/* no "label = pc" definition yet */
    GetByte();/* read start of statement */
    do {
/* Parse until end of statement is reached. Only loops if statement contains
 * "label = pc" definition and something else; or when "!ifdef" is true.
 */
      switch(GotByte) {

        case 0:/* Ignore now, act later (stops zero from being "default") */
        break;

        case '+':
        Flow_MacroCall();
        break;

        case '!':
        PseudoOpcode();
        break;

        case '*':
        SetPC();
        break;

        case '.':
        if(fPC2Label) {
          Message(pseSyntax, EERROR);
          SkipRest();
        } else {
          len = Stream_ReadKeyword(&StringItem.String[2], LSMAX - 2, TRUE);
          if(len) SetLabel(Context[nContext].nZone_Now, len);
          fPC2Label = TRUE;
        }
        break;

        default:
        if(pFlagTable[(u_char) GotByte] & MBYTEILLEGAL) {
          Message(pseSyntax, EERROR);
          SkipRest();
        } else {
          len = Stream_ReadKeyword(&StringItem.String[2], LSMAX - 2, FALSE);
          /* (now: GotByte = illegal char) */
          /* It is only a label if it isn't a mnemonic */
          if(Mnemo_IsInvalid(len)) {
            if(fPC2Label) {
              Message(pseSyntax, EERROR);
              SkipRest();
            } else {
              SetLabel(NZONE_GLOBAL, len);
              fPC2Label = TRUE;
            }
          }
        }
      }
    } while(GotByte);

#ifdef FDEBUG
    printf("%d \n", Context[nContext].nLines);
#endif

    PC_CPU += PC_inc;
    PC_Mem += PC_inc;
    if(PC_Mem > PC_Highest) PC_Highest = PC_Mem;
    PC_inc = 0;
  } while(EndReason == RNONE);
}

/*
 * Skip remainder of statement, for example on error
 */
static void SkipRest() {
  while(GotByte) GetByte();/* Read characters until zero */
}

/*
 * Ensure that the remainder of the current statement is empty, for example
 * after mnemonics using implicit addressing.
 */
static void EnsureEOL() {/* GotByte = first char to test */
  SKIPSPACE;
  if(GotByte) {
    Message(pseSyntax, EERROR);
    SkipRest();
  }
}

/*
 * Parse pseudo opcodes. Has to be re-entrant.
 */
static void PseudoOpcode() {/* (GotByte = "!") */
  ListItem  *p;
  void     (*fn)(void);
  int        len;

#ifdef FDEBUG
  printf(" PO ");
#endif
  len = Stream_ReadKeyword(&StringItem.String[0], LSMAX, TRUE);
  if(len) {
    /* Search for list item */
    String_ToLower(&StringItem, len);
    p = Struct_Search(&StringItem, HTYPE_PSEUDO, len, 0);
    if(p) {
      fn = (void (*)()) p->Data.Body;
      if(fn != NULL) (fn)();
      else           Message(pseUkPO, EERROR);
    } else Message(pseUkPO, EERROR);
  }
}

/*
 * Parse (re-)definitions of program counter
 */
static void SetPC() {/* GotByte = "*" */
  Value v;

#ifdef FDEBUG
  printf(" *= ");
#endif
  NEXTANDSKIPSPACE;/* proceed with next char */
  /* re-definitions of program counter change segment */
  if(GotByte == '=') {
    GetByte();/* proceed with next char */
    v = ALU_GetValue_Strict();
    if(fPCdefined) {
      /* not first segment, so show status of previous one */
      if(v < PC_Lowest) PC_Lowest = v;/* "Lowest" and "Highest" still needed? */
      String_EndSegment();
      if(ffPass & FPASSDOONCE) {
        Segment_Check(v);
        Segment_FindMax(v);
      }
    } else {
      /* first segment, so buffer some things */
      PC_Lowest = v;
      PC_Highest = v;
    }
    SegmentStart = v;
    PC_CPU = v;
    PC_Mem = v;
    fPCdefined = TRUE;
    EnsureEOL();
  } else {
    Message(pseSyntax, EERROR);
    SkipRest();
  }
}

/*
 * Defines a label (may be global or local). The stringbuffer holds the label's
 * name ("len" giving the length, "Zone" its zone).
 */
static void SetLabel(Sixteen Zone, int len) {/* GotByte = illegal */
  ListItem *p;
  Value     v;
  int       fb = Mnemo_GetForceBit();/* skips spaces */

  p = Struct_GetPreparedLabel(Zone, len, fb);
  if(GotByte == '=') {
    /* label = parsed value */
    GetByte();/* proceed with next char */
    /* Bug fix: Don't include the following call as a parameter inside the
     * next call. It changes ffValue and some compilers might not notice this.
     */
    v = ALU_GetValue_Medium();
    SetLabelValue(p, v, ffValue, FALSE);
    EnsureEOL();
  } else {
    /* label = PC */
    if(fPCdefined == FALSE) {
      Message(pseNoPC, EERROR);
      fPCdefined = TRUE;
    }
    SetLabelValue(p, PC_CPU, MVALUE_DEFINED, FALSE);
  }
}

/*
 * Assign value to label list item. The routine acts upon the list item's flag
 * bits and produces errors if needed. "p" points to the list item, "v" is the
 * value to assign, ff are the flags to set.
 */
static void SetLabelValue(ListItem *p, Value v, int ff, int fRedefine) {
  Value vOld;
  char  Flags = p->Data.Label.Flags;

  if((Flags & MVALUE_DEFINED) && (fRedefine==FALSE)) {
    /* Label is already defined, so compare new and old values */
    vOld = p->Data.Label.Value;
    if(vOld != v) Message(pseLabelTwice, EERROR);
#ifdef FDEBUG
    printf("Line: %d \n Old: %d \n New: %d \n", Context[nContext].nLines, vOld, v);
#endif
  } else {
    /* Label is not yet defined OR redefinitions are allowed */
    p->Data.Label.Value = v;
  }
  /* Ensure that "unsure" labels without "isByte" state don't get that */
  if((Flags & (MVALUE_UNSURE | MVALUE_ISBYTE))==MVALUE_UNSURE) {
    ff &= ~MVALUE_ISBYTE;
  }
  if(fRedefine) {
    Flags = (Flags & MVALUE_UNSURE) | ff;
  } else {
    if((Flags & MVALUE_FORCEBITS) == 0) {
      if((Flags & (MVALUE_UNSURE | MVALUE_DEFINED)) == 0) {
        Flags |= ff & MVALUE_FORCEBITS;
      }
    }
    Flags |= ff & ~MVALUE_FORCEBITS;
  }
  p->Data.Label.Flags = Flags;
}

/*
 * Error handling
 */

/*
 * This routine handles problems that may be solved by performing further
 * passes: "NeedValue" type errors.
 * If the current pass has the corresponding flagbit set, this routine will not
 * only count, but also show these errors to the user.
 */
static void NeedValue() {
  nNeedvalue++;
  if(ffPass & FPASSDOERROR) Message(pseNeedValue, EERROR);
}

/*
 * Output of warning or error messages. "psm" points to the string, "type" is
 * the message's type:
 *
 * EWARNING is not a "real" error but just a warning; the produced code looks
 *   as expected. But there has been a situation that should be reported to the
 *   user, for example there may has been assembled a 16-bit parameter with an
 *   8-bit value.
 *
 * EERROR is a "real" error; something went wrong in a way that implies that
 *   the output almost for sure won't look like expected, for example when
 *   there was a syntax error. The assembler will try to go on with the
 *   assembly though, as the user surely would like to know about more than
 *   one of his typos at a time.
 *
 * ESERIOUS is a serious error, an error that makes it impossible to go on with
 *   the assembly. Example: "!fill" without a parameter - the program counter
 *   cannot be adjusted correctly in this case, so proceeding would be of no
 *   use at all.
 *
 * The routine will show the given error string, as well as the current
 * context: Filename, line number, source type and source title.
 */
static void Message(char *psm, int type) {
  /* psm points to desired message */
  PLATFORM_MESSAGE(psm, type);
  printf("%s - File %s", eType[type], Context[nContext].pSourceFile);
  printf(", Line %d",                 Context[nContext].nLines);
  printf(" (%s ",                     Context[nContext].pSourceType);
  printf("%s): %s\n",                 Context[nContext].pSourceTitle, psm);
  if(type == ESERIOUS) CleanExit(EXIT_FAILURE);
  if(type == EERROR) nErrors++;
  if(nErrors >= MAXERRORS) CleanExit(EXIT_FAILURE);
}

/*
 * Get command line arguments
 */

/*
 * This routine sets "pnfTop" to point to the given filename string and acts
 * upon the supported option character(s).
 */
static void ReadParameters(int argc, char *argv[]) {
  char c;
  int  i = 1,
       j;

  while((i < argc) && (argv[i][0] == '-')) {
    j = 1;
    while((c = argv[i][j++]) != 0) {
      switch(c) {

        PLATFORM_SWITCHCODE /* platform specific switches are inserted here */

        case 'h':
        String_ShowInfo();/* show syntax message */
        break;

        case 'v':
        lVerbosity += 1;/* verbose mode */
        if((argv[i][j] >= '0') && (argv[i][j] <= '9')) {
          lVerbosity = argv[i][j++] - '0';
        }
        break;

        default:
        printf(pseUkSwitch, c);
      }
    }
    i++;
  }
  if(i == argc) {
    if(i - 1) {
      printf(pseNoTop);
      CleanExit(EXIT_FAILURE);
    } else {
      String_ShowInfo();
      CleanExit(EXIT_SUCCESS);
    }
  } else {
    pnfTop = argv[i];
  }
}

/*
 * Tidy up before exiting
 */
static void CleanExit(int exitstatus) {
  Stream_Dump();/* if wanted and not yet done, dump labels */
  PLATFORM_EXIT;
  Stream_CloseTop();/* close files */
  Stream_CloseSub();
#ifdef FDEBUG
  fclose(filter);
  fclose(filterRaw);
#endif
  FREE_TASK();/* Free memory */
  exit(exitstatus);
}

/*
 * Conversion routines...
 */

/*
 * Convert character using given conversion table
 */
static char ConvertChar(char byte, int hCT) {
  switch(hCT) {

    case HCODETABLE_RAW:
    return(byte);

    case HCODETABLE_PET:
    if((byte > 64) && (byte <  91)) return((char) (byte + 128));
    if((byte > 96) && (byte < 123)) return((char) (byte -  32));
    return(byte);

    case HCODETABLE_SCR:
    if((byte > 96) && (byte <123))
      return((char) (byte - 96));/* shift uppercase down */
    if((byte > 90) && (byte < 96))
      return((char) (byte - 64));/* shift [\]^_ down */
    if(byte == 96) return(64);/* shift ` down */
    if(byte == 64) return(0);/* shift @ down */
    return(byte);

    case HCODETABLE_FILE:
    return(pConvTable[(u_char) byte]);

    default:
    Message(pseNotYet, EERROR);
    return(byte);
  }
}

/*
 * Convert string using given conversion table
 */
static void ConvertString(char *p, int len, int hCT) {
  int a;

  for(a = len-1; a >= 0; a--) {
    p[a] = ConvertChar(p[a], hCT);
  }
}

/*
 * Fill output buffer with given byte value
 */
static void FillOutBuffer(u_char c) {
  int a;

  for(a = OUTBUFFERSIZE-1; a >= 0; a--) OutputBuffer[a] = c;
}

/*
 * Link segment data into segment chain
 */
static void Segment_Link(Value Start, Value Length) {
  SegmentStruct *p;

  /* may be faster if list is ordered !! */
  p = ALLOC_PASS(sizeof(SegmentStruct));
  p->Start = Start;
  p->Length = Length;
  p->Next = pSegmentList;
  pSegmentList = p;
}

/*
 * Set up new SegmentMax value according to the given program counter. Just
 * find the next segment start and subtract 1.
 */
static void Segment_FindMax(Value newPC) {
  SegmentStruct *p;

  /* may be faster if list is ordered !! */
  SegmentMax = OUTBUFFERSIZE;/* will be decremented later ! */
  p = pSegmentList;
  while(p) {
    if(p->Start > newPC) {
      if(p->Start < SegmentMax) SegmentMax = p->Start;
    }
    p = p->Next;
  }
  SegmentMax--;/* last free address available */
#ifdef FDEBUG
  printf("Set SegmentMax to %x\n", SegmentMax);
#endif
}

/*
 * Check whether given PC is inside segment.
 */
static void Segment_Check(Value newPC) {
  SegmentStruct *p;

  p = pSegmentList;
  while(p) {
    if(newPC >= p->Start) {
      if(newPC < (p->Start)+(p->Length)) Message(pswSegIllegal, EWARNING);
    }
    p = p->Next;
  }
}
