/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * List item stuff
 */

#include "item.h"

/*
 * Compute "big hash" by exclusive ORing the struct's type and "len" bytes
 * of its name. Write output to struct's hash, reduce "big hash" to character
 * and return that.
 */
static u_char Struct_Hash(ListItem *p, int len) {
  char         *pb;
  int           a;
  unsigned int  Hash = p->Type;

  pb = &(p->String[0]);
  for(a = len-1; a >= 0; a--) {
    Hash = ((Hash << 7) | (Hash >> (8 * sizeof(unsigned int) - 7))) ^ pb[a];
  }
  p->Hash = Hash;
  PLATFORM_INT2CHAR(Hash);
  return((u_char) Hash);
}

/*
 * Compute the hash of the given string and then use that to try to find a
 * list item that matches the given data (big hash, type, name). Return pointer
 * to list item found.
 * If no matching list item can be found and size is not zero, allocate "size"
 * bytes of memory, link it to the correct list, copy comparison data into
 * this new list item and return its pointer (and set the global "fMadeItem"
 * flag).
 * If no matching list item can be found and size is zero, return NULL.
 */
static ListItem *Struct_Search(ListItem *pConst, u_char type, int len, int Size) {
  ListItem *pCheck,
           *pNew;
  char     *p1,
           *p2,
            byte1,
            byte2;
  u_char    Hash;

  fMadeItem = FALSE;
  pConst->Type = type;
  Hash = Struct_Hash(pConst, len);
  pCheck = (ListItem *) &pPointerTable[Hash];/* point to first pointer */
  while(pCheck->Next) {
    if((pCheck->Next)->Hash > pConst->Hash) break;
#ifdef FDEBUG
    printf("{Check}");
#endif
    pCheck = pCheck->Next;
    /* Compare item at pCheck with pConst item */
    /* Compare big hash, type and first two characters (no terminator check) */
    if(pCheck->Hash == pConst->Hash) {
#ifdef FDEBUG
      printf("{=B}");
#endif
      if(pCheck->Type == pConst->Type) {/* check type */
        p1 = &(pCheck->String[0]);
        p2 = &(pConst->String[0]);
        if(*(p1++) == *(p2++)) {/* check first character */
          if(*(p1++) == *(p2++)) {/* check second character */
            /* check rest of name up to terminator */
            do {
              byte1 = *(p1++);
              byte2 = *(p2++);
            } while((byte1 == byte2) && byte1);
            if(byte1 == byte2) {
#ifdef FDEBUG
              printf("{found}");
#endif
              return(pCheck);/* return pointer to item */
            }
          }
        }
      }
    }
  }
  if(Size == 0) {
#ifdef FDEBUG
    printf("{failed}");
#endif
    return(NULL);/* indicate failure */
  }
  /* create new item and link into list */
  pNew = (ListItem *) ALLOC_TRY(Size);
  pNew->Next = pCheck->Next;
  pCheck->Next = pNew;
  pNew->Hash = pConst->Hash;
  pNew->Type = pConst->Type;/* copy type */
  p1 = &(pNew->String[0]);
  p2 = &(pConst->String[0]);
  *(p1++) = *(p2++);/* copy first char */
  *(p1++) = *(p2++);/* copy second char */
  do {
    /* copy rest of name */
    byte2 = *(p2++);
    *(p1++) = byte2;
  } while(byte2);
  fMadeItem = TRUE;
#ifdef FDEBUG
  printf("{created}");
#endif
  return(pNew);/* return pointer to new item */
}

/*
 * Looks out for (and creates, if necessary) label/macro and returns pointer.
 * Stringbuffer contains name and "Zone" contains desired zone, "len" is
 * the length of the name string. This routine increases "len" by 2 to cater
 * for the zone characters.
 */
static ListItem *Struct_GetZoned(Sixteen Zone,int len,int type,int create) {
  int size = 0;

  if(create) size = sizeof(SizeStruct) + len + 2;
  StringItem.String[0] = (char) (Zone & 255);/* zone = given zone */
  StringItem.String[1] = (char) ((Zone >> 8) & 255);
  /* search for list item. If not found, create with given size */
  return(Struct_Search(&StringItem, type, len + 2, size));
}

/*
 * Search for label. If it was created, it is now given the size info "fb".
 */
static ListItem *Struct_GetPreparedLabel(Sixteen Zone, int len, int fb) {
  ListItem *p;

  p = Struct_GetZoned(Zone, len, HTYPE_LABEL, TRUE);
  if(fMadeItem) {
    /* Finish empty label item */
    p->Data.Label.Flags = (u_char) (LABELFLAG_DEFAULT | fb);
    p->Data.Label.Value = 0;
  } else {
    if(fb) {
      if((p->Data.Label.Flags & MVALUE_FORCEBITS) != fb) {
        Message(pseTooLate, EERROR);/* too late for postfix */
      }
    }
  }
  return(p);
}

/*
 * Allocate memory and prepare for auto-freeing at end of try
 */
static void *MyAlloc(size_t size, void **Type) {
  void **p;

  p = malloc(size + sizeof(void *));
  if(p) {
    p[0] = Type[0];
    Type[0] = p;
    return(&p[1]);
  } else Message(pseNoMemLeft, ESERIOUS);
  return(NULL);
}

/*
 * Free memory at end of try
 */
static void MyFree(void *Type) {
  void **p1,
       **p2;

  p1 = Type;
  while ((p2 = p1)) {
    p1 = p2[0];
    free(p2);
  }
}
