/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Mnemonics
 */

#include "mnemo.h"

/*
 * Branches to different checks according to current cpu
 */
static int Mnemo_IsInvalid(int len) {
  switch(hCPU_Now) {

    case HCPU_6502:
    case HCPU_6510:
    case HCPU_65C02:
    case HCPU_65816:
    return(Mnemo65_IsInvalid(len));

    case HCPU_Z80:
    return(Mnemo80_IsInvalid(len));

    default:
    Message(pseNotYet, EERROR);
    return(TRUE);
  }
}

/*
 * 65xx stuff
 */

/*
 * Test whether mnemonic is known by current 65xx cpu
 */
static int Mnemo65_IsInvalid(int len) {
  if(len != 3) return(TRUE);
  MnemoItem.String[0] = StringItem.String[2] | 32;/* to lower case */
  MnemoItem.String[1] = StringItem.String[3] | 32;
  MnemoItem.String[2] = StringItem.String[4] | 32;
  MnemoItem.String[3] = 0;/* terminate (needed by item searcher) */

/* Look out for 6502 mnemonics */
  if(Mnemo65_Check(HTYPE_6502MNEMO))  return(FALSE);
  if(hCPU_Now == HCPU_6502)           return(TRUE);

/* Look out for illegal 6510 mnemonics */
  if(hCPU_Now == HCPU_6510) return(!Mnemo65_Check(HTYPE_6510MNEMO));

/* Look out for 65c02 mnemonics */
  if(Mnemo65_Check(HTYPE_65C02MNEMO)) return(FALSE);
  if(hCPU_Now == HCPU_65C02)          return(TRUE);

/* Look out for 65816 mnemonics */
  return(!Mnemo65_Check(HTYPE_65816MNEMO));
}

/*
 * Work routine
 */
static int Mnemo65_Check(int type) {
  ListItem *p;
  u_char    group,
            value;

  /* search for list item, length = 3 chars */
  p = Struct_Search(&MnemoItem, type, 3, 0);
  if(p == 0) return(FALSE);
  value = p->Data.Bytes.Code;/* get byte value */
  group = p->Data.Bytes.Group;/* get group */
  switch(group) {

    case  0:
    Mnemo65_00(value);
    break;

    case  1:
    Mnemo65_01(value);
    break;

    case  2:
    Mnemo65_02(value);
    break;

    case  3:
    Mnemo65_03_04(value, TRUE);
    break;

    case  4:
    Mnemo65_03_04(value, FALSE);
    break;

    case  5:
    Mnemo65_05_06_07(value, 5);
    break;

    case  6:
    Mnemo65_05_06_07(value, 6);
    break;

    case  7:
    Mnemo65_05_06_07(value, 7);
    break;

    case  8:
    Mnemo65_08(value);
    break;

    case  9:
    Mnemo65_09(value);
    break;

    case 10:
    Mnemo65_10(value);
    break;

    default:Message(pseUkGroup, ESERIOUS);
  }
  return(TRUE);
}

/*
 * Mnemonics using only implicit addressing.
 */
static void Mnemo65_00(u_char opcode) {
  Stream_PutCodeByte(opcode);
  EnsureEOL();
}

/*
 * Mnemonics using only 8bit relative addressing (short branch instructions).
 */
static void Mnemo65_01(u_char opcode) {
  Value v = ALU_GetValue_Medium();

  if(ffValue & MVALUE_DEFINED) {
    v -= (PC_CPU + 2);
    if((v > 127) || (v < -128)) Message(pseTooFar, EERROR);
  }
  Stream_PutCodeByte(opcode);
  Stream_Put8(v);
  EnsureEOL();
}

/*
 * Mnemonics using only 16bit relative addressing (BRL and PER).
 */
static void Mnemo65_02(u_char opcode) {
  Value v = ALU_GetValue_Medium();

  if(ffValue & MVALUE_DEFINED) {
    v -= (PC_CPU + 3);
    if((v > 32767) || (v < -32768)) Message(pseTooFar, EERROR);
  }
  Stream_PutCodeByte(opcode);
  Stream_Put16(v);
  EnsureEOL();
}

/*
 * The main accumulator stuff (ADC, AND, CMP, EOR, LDA, ORA, SBC, STA).
 * Group 4 means "without immediate addressing mode" (applies to STA only).
 */
static void Mnemo65_03_04(u_char base, int ImmFlag) {
  u_char offset = 0;/* offset, added to base to make opcode */
  int    fb     = Mnemo_GetForceBit();/* skips spaces */
  Value  v      = Mnemo_GetArgument();

  switch(hAM_Now) {

    case HAM_IMM:/* #$ff or #$ffff (depending on accu length) = offset $09 */
    if(ImmFlag) {
      if(hCPU_Now == HCPU_65816) {
        /* 65816 knows 16bit immediate addressing */
        if(fb == 0) {/* if no force bits given, use flag */
          if(fCPU_LongA) fb = MVALUE_FORCE16;
          else           fb = MVALUE_FORCE08;
        }
        Stream_PutCodeByte(base + 9);
        switch(Mnemo_DoChecks(fb, v, ffValue, MAYBE_1_2__)) {

          case MVALUE_FORCE08:
          Stream_Put8(v);
          break;

          case MVALUE_FORCE16:
          Stream_Put16(v);
        }
      } else {
        /* other 65xx only know 8bit immediate addressing */
        if(Mnemo_DoChecks(fb, v, ffValue, MAYBE_1____)) {
          Stream_PutCodeByte(base + 9);
          Stream_Put8(v);
        }
      }
    } else Message(pseCombCA, EERROR);/* kill "STA #..." */
    break;

    case HAM_ABSX:/* $ff,x, $ffff,x $ffffff,x = offsets $15, $1d, $1f */
    offset = 16;/* adjust offset and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_ABS:/*  $ff,   $ffff,  $ffffff   = offsets $05, $0d, $0f */
    switch(Mnemo_DoChecks(fb, v, ffValue, MAYBE_1_2_3)) {

      case MVALUE_FORCE08:
      Stream_PutCodeByte(base + offset + 5);
      Stream_Put8(v);
      break;

      case MVALUE_FORCE16:
      Stream_PutCodeByte(base + offset + 13);
      Stream_Put16(v);
      break;

      case MVALUE_FORCE24:
      Stream_PutCodeByte(base + offset + 15);
      Stream_Put24(v);
    }
    break;

    case HAM_ABSY:/* $ffff,y = offset $19 */
    if(Mnemo_DoChecks(fb, v, ffValue, MAYBE___2__)) {
      Stream_PutCodeByte(base + 25);
      Stream_Put16(v);
    }
    break;

    case HAM_ABSS:/* $ff,s = offset $03 */
    if(Mnemo_DoChecks(fb, v, ffValue, MAYBE_1____)) {
      Stream_PutCodeByte(base + 3);
      Stream_Put8(v);
    }
    break;

    case HAM_XIND:/* ($ff,x) = offset $01 */
    if(Mnemo_DoChecks(fb, v, ffValue, MAYBE_1____)) {
      Stream_PutCodeByte(base + 1);
      Stream_Put8(v);
    }
    break;

    case HAM_IND:/*  ($ff)   = offset $12 */
    offset = 1;/* adjust offset and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_INDY:/* ($ff),y = offset $11 */
    if(Mnemo_DoChecks(fb, v, ffValue, MAYBE_1____)) {
      Stream_PutCodeByte(base + offset + 17);
      Stream_Put8(v);
    }
    break;

    case HAM_INDLY:/* [$ff],y = offset $17 */
    offset = 16;/* adjust offset and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_INDL:/*  [$ff]   = offset $07 */
    if(Mnemo_DoChecks(fb, v, ffValue, MAYBE_1____)) {
      Stream_PutCodeByte(base + offset + 7);
      Stream_Put8(v);
    }
    break;

    case HAM_SINDY:/* ($ff,s),y = offset $13 */
    if(Mnemo_DoChecks(fb, v, ffValue, MAYBE_1____)) {
      Stream_PutCodeByte(base + 19);
      Stream_Put8(v);
    }
    break;

    default:

#ifdef FDEBUG
    printf("addressing mode: %d", hAM_Now);
#endif

    Message(pseCombCA, EERROR);
  }
}

/*
 * Various mnemonics with different addressing modes.
 * Group 5 isn't affected by different immediate addressing,
 * Group 6 means accumulator-related,
 * Group 7 means index-register-related.
 */
static void Mnemo65_05_06_07(u_char row, u_char group) {
  int   bits,
        code1,
        code2,
        column = 0,
        fb     = Mnemo_GetForceBit();/* read length info (skips spaces) */
  Value v      = Mnemo_GetArgument();

  switch(hAM_Now) {

    case HAM_IMP:/* column 0 */
    Mnemo65_567(0, row);/* output code byte */
    break;

    case HAM_IMM:/* #$ff or #$ffff (depending on register length), column 1 */
    if(Mnemo65_567(1, row)) {;/* output code byte */
      /* check CPU and group (5 = don't care, 6 = accu, 7 = index registers) */
      if((hCPU_Now == HCPU_65816) && (group != 5)) {
        /* 65816 knows 16bit immediate addressing of some commands */
        if(fb == 0) {/* if no force bits given, use relevant switch */
          fb = MVALUE_FORCE08;
          if(group == 6) {
            if(fCPU_LongA) fb = MVALUE_FORCE16;/* group 6 depends on M flag*/
          } else {
            if(fCPU_LongR) fb = MVALUE_FORCE16;/* others depend on R flag */
          }
        }
        switch(Mnemo_DoChecks(fb, v, ffValue, MAYBE_1_2__)) {

          case MVALUE_FORCE08:
          Stream_Put8(v);
          break;

          case MVALUE_FORCE16:
          Stream_Put16(v);
        }
      } else {
        /* only 8bit immediate addressing */
        if(Mnemo_DoChecks(fb, v, ffValue, MAYBE_1____)) {
          Stream_Put8(v);
        }
      }
    }
    break;

    case HAM_ABSY:/* $ff,y  or  $ffff,y, column 6(8bit) or 7(16bit) */
    column += 2;/* adjust column and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_ABSX:/* $ff,x  or  $ffff,x, column 4(8bit) or 5(16bit) */
    column += 2;/* adjust column and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_ABS: /* $ff    or  $ffff,   column 2(8bit) or 3(16bit) */
    column += 2;/* adjust column and go on (no break here) */
    code1 = pGroup5Table[column + (row << 3)];
    code2 = pGroup5Table[column + (row << 3) + 1];
    bits = MAYBE______;
    if(code1) bits |= MAYBE_1____;/*  8 bit */
    if(code2) bits |= MAYBE___2__;/* 16 bit */
    switch(Mnemo_DoChecks(fb, v, ffValue, bits)) {

      case MVALUE_FORCE08:
      Stream_PutCodeByte(code1);
      Stream_Put8(v);
      break;

      case MVALUE_FORCE16:
      Stream_PutCodeByte(code2);
      Stream_Put16(v);
    }
    break;

    default:
    Message(pseCombCA, EERROR);
  }
}

/*
 * Utility routine for groups 5, 6 and 7
 */
static int Mnemo65_567(int x, int y) {
  u_char opcode = pGroup5Table[x + (y << 3)];

  if(opcode) {
    Stream_PutCodeByte(opcode);
    return(TRUE);
  }
  Message(pseCombCA, EERROR);
  return(FALSE);
}

/*
 * Mnemonics using "8bit, 8bit" addressing. Only applies to "MVN" and "MVP".
 */
static void Mnemo65_08(u_char opcode) {
  Value b2,
        b1 = ALU_GetValue_Medium();

  if(Stream_Comma()) {
    b2 = ALU_GetValue_Medium();
    Stream_PutCodeByte(opcode);
    Stream_Put8(b2);/* changed syntax to "opcode, source, target" */
    Stream_Put8(b1);
    EnsureEOL();
  } else Message(pseSyntax, EERROR);
}

/*
 * Mnemonics using only the "($..)" addressing mode. Only applies to PEI.
 */
static void Mnemo65_09(u_char opcode) {
  Value v = Mnemo_GetArgument();

  switch(hAM_Now) {

    case HAM_IND:
    Stream_PutCodeByte(opcode);
    Stream_Put8(v);
    break;

    default:
    Message(pseCombCA, EERROR);
  }
}

/*
 * The jump instructions.
 */
static void Mnemo65_10(u_char column) {
  u_char code2,
         code3;
  int    bits,
         offset = 0,
         fb     = Mnemo_GetForceBit();/* read length info (skips spaces) */
  Value  v      = Mnemo_GetArgument();

  switch(hAM_Now) {

    case HAM_ABS:/* $ffff  or  $ffffff, row 0(16bit) or 3(24bit) */
    code2 = pGroup0Table[column];
    code3 = pGroup0Table[column + 12];
    bits = MAYBE______;
    if(code2) bits  = MAYBE___2__;/* 16 bit */
    if(code3) bits |= MAYBE_____3;/* 24 bit */
    switch(Mnemo_DoChecks(fb, v, ffValue, bits)) {

      case MVALUE_FORCE16:
      Stream_PutCodeByte(code2);
      Stream_Put16(v);
      break;

      case MVALUE_FORCE24:
      Stream_PutCodeByte(code3);
      Stream_Put24(v);
    }
    break;

    case HAM_INDL:/* [$ffff],   row 4 */
    offset += 8;/* adjust offset and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_XIND:/* ($ffff,x), row 2 */
    offset += 4;/* adjust offset and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_IND: /* ($ffff),   row 1 */
    offset += 4;/* adjust offset and go on (no break here) */
    if(Mnemo_DoChecks(fb, v, ffValue, MAYBE___2__)) {
      code2 = pGroup0Table[column + offset];
      if(code2) {
        Stream_PutCodeByte(code2);
        Stream_Put16(v);
      } else Message(pseCombCA, EERROR);
    }
    break;

    default:Message(pseCombCA, EERROR);
  }
}

/*
 * Service routine to read optional info about parameter length.
 */
static int Mnemo_GetForceBit() {
  char byte,
       fb = 0;

  if(GotByte == '+') {
    byte = GetByte();
    if(byte == '1') fb = MVALUE_FORCE08;
    if(byte == '2') fb = MVALUE_FORCE16;
    if(byte == '3') fb = MVALUE_FORCE24;
    if(fb) GetByte();
    else   Message(pseUkPF, EERROR);
  }
  SKIPSPACE;
  return(fb);
}

/*
 * Address mode parsing
 */

/*
 * This routine returns the command's argument (using the valueparser). The
 * addressing mode will be stored in hAM_Now.
 */
static Value Mnemo_GetArgument() {
  Value  v;
  char   First;

  hAM_Now = HAM_ABS;
  SKIPSPACE;
  if(GotByte == '[') {
    GetByte();/* proceed with next char */
    v = ALU_GetValue_Medium();
    if(GotByte == ']') hAM_Now |= HAM_INDL | Mnemo_GetIndex(TRUE);
    else               Message(pseSyntax, EERROR);
  } else {
    if(GotByte == '#') {
      GetByte();/* proceed with next char */
      hAM_Now |= HAM_IMM;
    }
    First = GotByte;
    v = ALU_GetValue_Liberal(FALSE);/* liberal, to allow for "(...," */
    if(ffValue & MVALUE_EXISTS) {
      if((ffValue & MVALUE_DEFINED) == 0) NeedValue();
    } else {
      hAM_Now |= HAM_IMP;
    }
    /* check for index in parentheses */
    if(nParenth) {
      /* in case there are still open parentheses, read internal index */
      hAM_Now |= (Mnemo_GetIndex(FALSE) << 2);
      if(GotByte == ')') GetByte();/* proceed with next char */
      else Message(pseSyntax, EERROR);
    }
    /* check for indirect addressing */
    if((First == '(') && (VP_Work == FALSE)) hAM_Now |= HAM_IND;
    /* check for external index */
    hAM_Now |= Mnemo_GetIndex(FALSE);
  }
  /* ensure end of line */
  EnsureEOL();
  return(v);
}

/*
 * Utility routine for parsing indices.
 */
static int Mnemo_GetIndex(int next) {
  int am = HAM__;

  if(next) GetByte();
  if(!Stream_Comma()) return(am);
  switch(GotByte) {

    case 's':
    case 'S':
    am = HAM_S;
    break;

    case 'x':
    case 'X':
    am = HAM_X;
    break;

    case 'y':
    case 'Y':
    am = HAM_Y;
    break;

    default:
    Message(pseSyntax, EERROR);
  }
  if(am != HAM__) NEXTANDSKIPSPACE;
  return(am);
}

/*
 * Utility routine for comparing force bits, argument value, argument size,
 * "unsure"-flag and possible addressing modes. Returns force bit matching
 * number of parameter bytes to send. If it returns zero, an error occurred
 * (and has already been delivered).
 */
static int Mnemo_DoChecks(int fb, Value v, int ff, int am) {
  int Result = 0;

/*
 *      fb      Force bit (none, 8b, 16b, 24b)
 *      v       Value of parameter
 *      ff      Flags of value (ffValue)
 *      am      Adressing modes (8b, 16b, 24b or any combination)
 *      c       Return value, number of parameter bytes to send (0 = error)
 */
  if(am != MAYBE______) {
    if(fb) {
/* Command has force bit */
      if(am & fb) Result = fb;/* Command allows this force bit */
      else        Message(pseCombCP, EERROR);/* ...or doesn't */
    } else {
/* Command has no force bit => using value's force bit */
      if(ff & MVALUE_FORCEBITS) {
  /* Value has force bit set, so return this or bigger size */
        if(MVALUE_FORCE08 & am & ff) Result = MVALUE_FORCE08;
        else {
          if(MVALUE_FORCE16 & am & ff) Result = MVALUE_FORCE16;
          else {
            if(MVALUE_FORCE24 & am) Result = MVALUE_FORCE24;
            else Message(pseTooBig, EERROR);/* else error */
          }
        }
      } else {
  /* Value has no force bit => see if there's only one addressing mode */
        switch(am) {

    /* There's only one addressing mode, so use it */
          case MVALUE_FORCE08:
          case MVALUE_FORCE16:
          case MVALUE_FORCE24:
          Result = am;
          break;

    /* There's more than one, so check whether value is sure */
          default:
          if(ff & MVALUE_UNSURE) {
      /* Value is unsure, so use default size */
            Result = MVALUE_FORCE08;
            if(((MVALUE_FORCE08 & am)==0) || ((ff & MVALUE_ISBYTE)==0)) {
              if(MVALUE_FORCE24 & am) Result = MVALUE_FORCE24;
              if(MVALUE_FORCE16 & am) Result = MVALUE_FORCE16;
              if(ff & MVALUE_DEFINED) {
                if(Result == MVALUE_FORCE16) {
                  if((v <= 0xff) && (v >= -0x80)) {
                    Message(pswHighbyteZero, EWARNING);
                  }
                }
                if(Result == MVALUE_FORCE24) {
                  if((v <= 0xffff) && (v >= -0x8000)) {
                    Message(pswHighbyteZero, EWARNING);
                  }
                }
              }
            }
          } else {
      /* Value is sure, so use its own size */
            if(v >= 0) {
              if((MVALUE_FORCE08 & am) && (v < 256)) {
                Result = MVALUE_FORCE08;
              } else {
                if((MVALUE_FORCE16 & am) && (v < 65536)) {
                  Result = MVALUE_FORCE16;
                } else {
                  if(MVALUE_FORCE24 & am) {
                    Result = MVALUE_FORCE24;
                  } else Message(pseTooBig, EERROR);
                }
              }
            } else Message(pseNegative, EERROR);
          }
        }
      }
    }
  } else Message(pseCombCA, EERROR);
  return(Result);
}

/*
 * Z80 stuff
 */

static int Mnemo80_IsInvalid(int len) {
  Message(pseNotYet, EERROR);
  SkipRest();
  return(TRUE);
}
