/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Pseudo opcodes
 */

#include "po.h"

/*
 * Switch to short accu
 */
static void PO_as() {
  fCPU_LongA  = FALSE;
  fCPU_LongAb = FALSE;
  EnsureEOL();
}

/*
 * Switch to long accu
 */
static void PO_al() {
  if(hCPU_Now == HCPU_65816) {
    fCPU_LongA  = TRUE;
    fCPU_LongAb = TRUE;
  } else Message(pseWrongCPU, EERROR);
  EnsureEOL();
}

/*
 * Switch to short index registers
 */
static void PO_rs() {
  fCPU_LongR  = FALSE;
  fCPU_LongRb = FALSE;
  EnsureEOL();
}

/*
 * Switch to long index registers
 */
static void PO_rl() {
  if(hCPU_Now == HCPU_65816) {
    fCPU_LongR  = TRUE;
    fCPU_LongRb = TRUE;
  } else Message(pseWrongCPU, EERROR);
  EnsureEOL();
}

/*
 * Define default value for empty memory
 */
static void PO_initmem() {
  Value v;

  if(ffPass & FPASSDOONCE) {
    if(ffRepeat & FREPEAT_MEMINIT) {
      Message(pseMemInitTwice, EERROR);
      SkipRest();
    } else {
      ffRepeat |= FREPEAT_MEMINIT;
      v = ALU_GetValue_Strict();
      if((v<256) && (v>-129)) {
        FillOutBuffer(v & 255);
        if(PC_Lowest != PC_Highest) {
          if(nNeedvalue == 0) nNeedvalue = 1;/* tricky bit */
        }
      } else Message(pseTooBig, EERROR);
    }
  } else {
    SkipRest();
  }
}

/*
 * Insert 8-bit values
 */
static void PO_08() {
  do {
    Stream_Put8(ALU_GetValue_Medium());
  } while(Stream_Comma());
  EnsureEOL();
}

/*
 * Insert 16-bit values
 */
static void PO_16() {
  do {
    Stream_Put16(ALU_GetValue_Medium());
  } while(Stream_Comma());
  EnsureEOL();
}

/*
 * Insert 24-bit values
 */
static void PO_24() {
  do {
    Stream_Put24(ALU_GetValue_Medium());
  } while(Stream_Comma());
  EnsureEOL();
}

/*
 * Insert 32-bit values
 */
static void PO_32() {
  do {
    Stream_Put32(ALU_GetValue_Medium());
  } while(Stream_Comma());
  EnsureEOL();
}

/*
 * Reserve space by sending some bytes of a given value
 */
static void PO_fill() {
  Value v    = VFILL;
  int   Size = ALU_GetValue_Strict();

  if(Stream_Comma()) v = ALU_GetValue_Medium();
  EnsureEOL();
  while(Size) {
    Stream_Put8(v);
    Size--;
  }
}

/*
 * Insert NOPs until PC fits conditions
 */
static void PO_align() {
  u_char v    = VALIGN;
  Value  Equal,
         Test = PC_CPU,
         And  = ALU_GetValue_Strict();

  if(!Stream_Comma()) Message(pseSyntax, EERROR);
  Equal = ALU_GetValue_Strict();
  if(Stream_Comma()) v = ALU_GetValue_Medium();
  EnsureEOL();
  while((Test & And) != Equal) {
    Stream_Put8(v);
    Test++;
  }
}

/*
 * Select dump file
 */
static void PO_savelabels() {
  int len = Stream_ReadFilename(pnfDump, LNFMAX, FALSE);

  if(len) {
    ffRepeat |= FREPEAT_DOLABELDUMP;/* set flag for dump */
    EnsureEOL();
  } else {
    SkipRest();
  }
}

/*
 * Select output file
 */
static void PO_to() {
  int len;

  if(ffPass & FPASSDOONCE) {
    /* in first pass, act upon this */
    if(ffRepeat & FREPEAT_OUTFILECHOSEN) {
      /* if outfile already chosen, complain */
      Message(pseNo2ndOutfile, EERROR);
    } else {
      /* do some work */
      ffRepeat |= FREPEAT_OUTFILECHOSEN;/* remember by setting flag */
      len = Stream_ReadFilename(pnfOut, LNFMAX, FALSE);
      if(len) {
        /* !!! accept file format value !!! */
        EnsureEOL();
      } else {
        SkipRest();
      }
    }
  } else {
    /* in other passes, just ignore it */
    SkipRest();
  }
}

/*
 * Select CPU
 */
static void PO_cpu() {
  ListItem *p;
  u_char    handle;
  int       len;

  SKIPSPACE;
  len = Stream_ReadKeyword(&StringItem.String[0], LSMAX, FALSE);
  if(len) {
    /* search for list item */
    p = Struct_Search(&StringItem, HTYPE_CPU, len, 0);
    if(p) {
      handle = p->Data.Bytes.Code;/* get handle */
      hCPU_Now = handle;
      if(handle == HCPU_65816) {
        fCPU_LongA = fCPU_LongAb;/* use old values */
        fCPU_LongR = fCPU_LongRb;
      } else {
        fCPU_LongA = FALSE;/* other processors can't do it */
        fCPU_LongR = FALSE;
      }
    } else Message(pseUkCPU, EERROR);
  }
  EnsureEOL();
}

/*
 * Include binary file
 */
static void PO_binary() {
  int   Byte,
        ffSize = 0;
  Value Size,
        Skip = 0;

  /* Check for number of files is done in Stream_OpenSub() */
  if(Stream_ReadFilename(pPathname, LNPMAX, TRUE)) {
    if(Stream_Comma()) {
      Size = ALU_GetValue_Empty();
      ffSize = ffValue;
      if(Stream_Comma()) {
        Skip = ALU_GetValue_Empty();
        if((ffValue & MVALUE_EXISTS)==0) Skip = 0;
      }
    }
    if(Stream_OpenSub(pPathname)) {
      fseek(hfSub, Skip, SEEK_SET);
      Byte = fgetc(hfSub);
      if(ffSize & MVALUE_EXISTS) {
        /* explicit size info, so truncate or pad file */
        while(Size) {
          if(Byte == EOF) Byte = 0;
          Stream_Put8(Byte);
          Size--;
          Byte = fgetc(hfSub);
        }
      } else {
        /* no explicit size info, so read file until EOF */
        while(Byte != EOF) {
          Stream_Put8(Byte);
          Byte = fgetc(hfSub);
        }
      }
      Stream_CloseSub();
      if((ffPass & FPASSDOONCE) && (lVerbosity > 1)) {
        printf(psvBinary, PC_inc, Skip);
      }
    }
    EnsureEOL();
  } else {
    SkipRest();
  }
}

/*
 * Choose default conversion table
 */
static void PO_convtab() {
  ListItem *p;
  int       len;

  SKIPSPACE;
  if((GotByte == '<') || (GotByte == '"')) {
    /* !!! read conversion table from file !!! */
    Message(pseNotYet, EERROR);
    SkipRest();
  } else {
    /* parse keyword */
    len = Stream_ReadKeyword(&StringItem.String[0], LSMAX, FALSE);
    if(len) {
      /* Search for list item. Don't create */
      String_ToLower(&StringItem, len);
      p = Struct_Search(&StringItem, HTYPE_CONV, len, 0);
      if(p) hCodeTable_Now = p->Data.Bytes.Code;/* set conversion table */
      else Message(pseSyntax, EERROR);
    }
  }
  EnsureEOL();
}

/*
 * Switch to CBM mode
 */
static void PO_cbm() {
  hCodeTable_Now = HCODETABLE_PET;
  if(ffPass & FPASSDOONCE) Message(pswCbmUsed , EWARNING);
  EnsureEOL();
}

/*
 * Insert text string (default format)
 */
static void PO_text() {
  PO_string(hCodeTable_Now, 0);
}

/*
 * Insert raw string
 */
static void PO_raw() {
  PO_string(HCODETABLE_RAW, 0);
}

/*
 * Insert PetSCII string
 */
static void PO_petscii() {
  PO_string(HCODETABLE_PET, 0);
}

/*
 * Insert screencode string
 */
static void PO_screencode() {
  PO_string(HCODETABLE_SCR, 0);
}

/*
 * Insert screencode string, EOR'd
 */
static void PO_scrxor() {
  Value eor = ALU_GetValue_Medium();

  if(Stream_Comma()) {
    PO_string(HCODETABLE_SCR, (u_char) eor);
  } else Message(pseSyntax, EERROR);
}

/*
 * Insert string
 */
static void PO_string(int hCT, u_char eor) {
  int a,
      len,
      temp = hCodeTable_Now;/* buffer current conversion table */

  SKIPSPACE;
  hCodeTable_Now = hCT;/* make given conversion table current one */
  do {
    if(GotByte == '"') {
      /* parse string */
      len = Stream_FinishQuoted(&StringItem.String[0], LSMAX, 0);
      if(len) {
        ConvertString(&StringItem.String[0], len, hCodeTable_Now);
        for(a = 0; a < len; a++) {
          Stream_Put8(StringItem.String[a] ^ eor);
        }
      } else {
        SkipRest();/* error */
      }
    } else {
      /* Parse value. No problems with single characters because the current */
      /* conversion table is temporarily set to the given one. */
      Stream_Put8(ALU_GetValue_Medium());
    }
  } while(Stream_Comma());
  EnsureEOL();
  hCodeTable_Now = temp;/* reactivate buffered conversion table */
}

/*
 * (Re)set label
 */
static void PO_set() {
  ListItem *p;
  int       ff,
            len,
            fb;
  Value     v;
  Sixteen   Zone = NZONE_GLOBAL;

  SKIPSPACE;
  if(GotByte == '.') {
    Zone = Context[nContext].nZone_Now;
    GetByte();
  }
  len = Stream_ReadKeyword(&StringItem.String[2], LSMAX - 2, FALSE);
  /* (now: GotByte = illegal char) */
  if(len) {
    fb = Mnemo_GetForceBit();/* skips spaces */
    p = Struct_GetPreparedLabel(Zone, len, fb);
    if(GotByte == '=') {
      /* label = parsed value */
      GetByte();/* proceed with next char */
      v = ALU_GetValue_Medium();
      ff = ffValue;
      p->Data.Label.Flags &= ~(MVALUE_FORCEBITS | MVALUE_ISBYTE);
      if(fb) {
        p->Data.Label.Flags |= fb;
        ff &= ~(MVALUE_FORCEBITS | MVALUE_ISBYTE);
      }
      SetLabelValue(p, v, ff, TRUE);
      EnsureEOL();
    } else {
      Message(pseSyntax, EERROR);
      SkipRest();
    }
  }
}
