/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * ALU stuff (the expression parser)
 */

#ifndef alu_H
#define alu_H

#include "platform.h"

/*
 * Constants
 */

/*
 * Operator handles
 */
enum {
/* special values (pseudo operators) */
  HOP_END,       /*                "reached end of expression" */
  HOP_RETURN,    /*                "return value to caller" */
/* monadic operators */
  HOP_OPENING,   /*   (v           left parenthesis, starting subexpression */
  HOP_NOT,       /*   !v   NOT v   Bitwise NOT */
  HOP_NEGATE,    /*   -v           Negate      */
  HOP_LOWBYTEOF, /*   <v           Lowbyte of  */
  HOP_HIGHBYTEOF,/*   >v           Highbyte of */
  HOP_BANKBYTEOF,/*   ^v           Bankbyte of */
/* dyadic operators */
  HOP_CLOSING,   /*  v)            right parenthesis, ending subexpression */
  HOP_POWEROF,   /*   v^w   */
  HOP_MULTIPLY,  /*   v*w   */
  HOP_DIVIDE,    /*   v/w             Integer-Division */
  HOP_MODULO,    /*   v%w   v MOD w   Remainder        */
  HOP_SL,        /*   v<<w   v LSL w   v ASL w   Shift left */
  HOP_LSR,       /*   v>>w   v LSR w             Logical shift right */
  /* HOP_ASR is no longer supported. */
  HOP_ADD,       /*   v+w    */
  HOP_SUBTRACT,  /*   v-w    */
  HOP_EQUALS,    /*   v=w    */
  HOP_LE,        /*   v<=w   */
  HOP_LT,        /*   v< w   */
  HOP_GE,        /*   v>=w   */
  HOP_GT,        /*   v> w   */
  HOP_DIFFERENT, /*   v!=w   v<>w   v><w   */
  HOP_AND,       /*   v&w       v AND w   */
  HOP_OR,        /*   v|w       v OR w    */
  HOP_EXOR,      /*   v EOR w   v XOR w   */
};

/*
 * Operator priorities
 */
enum {
  PRIO_END        =  0,
  PRIO_RETURN     =  2,
  PRIO_CLOSING    =  4,
  PRIO_OPENING    =  6,
  PRIO_OR         =  8,
  PRIO_EXOR       = 10,
  PRIO_AND        = 12,
  PRIO_EQUALS     = 14,
  PRIO_DIFFERENT  = 16,
  /* same priority for all comparison operators */
    PRIO_LE         = 18,
    PRIO_LT         = 18,
    PRIO_GE         = 18,
    PRIO_GT         = 18,
  /* same priority for all byte extraction operators */
    PRIO_LOWBYTEOF  = 20,
    PRIO_HIGHBYTEOF = 20,
    PRIO_BANKBYTEOF = 20,
  /* same priority for both shift operators */
    PRIO_SL         = 22,
    PRIO_LSR        = 22,
  /* same priority for "+" and "-" */
    PRIO_ADD        = 24,
    PRIO_SUBTRACT   = 24,
  /* same priority for "*", "/" and "%" */
    PRIO_MULTIPLY   = 26,
    PRIO_DIVIDE     = 26,
    PRIO_MODULO     = 26,
  PRIO_NEGATE     = 28,
  PRIO_POWEROF    = 30,
  PRIO_NOT        = 32,
};

/*
 * Variables
 */

extern int    ALU_LeftParentheses;/* Number of parentheses still open */
extern value  ALU_Value;/* Expression value */
extern int    ALU_Flags;/* Expression flags */
/* Needless parentheses indicate use of indirect addressing */
#define MVALUE_INDIRECT (1u << 7)
/* There was *something* to parse */
#define MVALUE_EXISTS   (1u << 6)
/* Value once was related to undefined expression */
#define MVALUE_UNSURE   (1u << 5)
/* Value is defined (if this is cleared, the value will be zero) */
#define MVALUE_DEFINED  (1u << 4)
/* Value is guaranteed to fit in one byte */
#define MVALUE_ISBYTE   (1u << 3)
/* Value usage forces 24-bit usage */
#define MVALUE_FORCE24  (1u << 2)
/* Value usage forces 16-bit usage */
#define MVALUE_FORCE16  (1u << 1)
/* Value usage forces 8-bit usage */
#define MVALUE_FORCE08  (1u << 0)
/* Bit mask for force bits */
#define MVALUE_FORCEBITS (MVALUE_FORCE08 | MVALUE_FORCE16 | MVALUE_FORCE24)
/* Bit mask for fixed values (defined and existing) */
#define MVALUE_GIVEN     (MVALUE_DEFINED | MVALUE_EXISTS)

/*
 * Prototypes
 */

value  ALU_GetValue_Strict(void);
value  ALU_GetValue_EmptyStrict(void);
value  ALU_GetValue_Medium(void);
value  ALU_GetValue_Liberal(void);

#endif
