/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Buffers and tables
 */

#include <stdio.h>

#include "alu.h"
#include "data.h"
#include "flowpo.h"
#include "core.h"
#include "mnemo.h"
#include "stream.h"
#include "strings.h"

/*
 * Constants
 */

char Exception_SegInSeg[] = "Segment starts inside another one, overwriting it.";
char Exception_NoLeftBrace[]  = "Missing '{'.";
char Exception_NoRightBrace[] = "Missing '}'.";

/* Error messages on startup or closedown */
char Error_NoTop[] = "Error: No top level source given.\n";
char Error_UkSwitch[] = "Error: Unknown CLI switch: '%c'.\n";
char Error_CannotOpenTopFile[] = "Error: Cannot open toplevel file.\n";


/* Other messages */
char psvFirst[]   = "First pass.\n";
char psvFurther[] = "Further pass.\n";
char psvFind[]    = "Further pass needed to find error.\n";
char BugInACME[] = "Bug in ACME, code follows";

/* Message types */
char psWarning[] = "Warning";
char psError[]   = "Error";
char psSerious[] = "Serious error";

/*
 * Variables
 */

int   Process_Flags = 0;/* Program flags variable and its bitfields */
int   Pass_Flags;/* Pass flags and its bitfields */
int   Process_Verbosity = 0;/* Level of additional output */
byte  GotByte;/* Last byte read (processed) */
byte* pnfTop;/* => top level file name as given in CLI */

/* Global counters */
int   nNeedValue_Now;/* "NeedValue" type errors */
int   nNeedValue_Last;/* "NeedValue" type errors of previous pass */
int   nErrors;/* Errors yet */

value PC_Mem;/* Current program counter (real memory address) */
value PC_inc;/* Increase PC by this amount after statement */
value PC_Lowest;/* Start address of program (first PC given) */
value PC_Highest;/* End address of program plus one */
value SegmentStart;/* Start of current segment */
value SegmentMax;/* Highest address segment may use */

byte  MiscString[LSMAX+1];/* temporary string buffer */
/* Static buffers for string storage, hash creation, tree item lookup */
byte  pConvTable[256];/* holds conversion table from file */

SegmentStruct* pSegmentList;/* points to linked list of segment structures */

void* pToBeFreed_Process;/* linked list, holds malloc blocks */
void* pToBeFreed_Try;/* linked list, holds malloc blocks */
void* pToBeFreed_Pass;/* linked list, holds malloc blocks */

/*
 * Main program
 */
int main(int argc, char *argv[]) {

  Process_Init();

  /* loop for repetition switch */
  do {
    Try_Init();

    ReadParameters(argc, (byte**) argv);/* get command line arguments */

    Tree_ClearPointerTable(TreesRAM);/* clear tree pointer table */
    Output_InitMem(0);/* init output buffer */

    /* First pass */
    if(Process_Verbosity > 1) printf(psvFirst);

    Core_DoPass(PASS_ISFIRST);

    if(nNeedValue_Now) {
      /* If there were undefined values, another pass will have to be done.*/
      do {
        nNeedValue_Last = nNeedValue_Now;/* Buffer previous error counter */

        /* Further passes */
        if(Process_Verbosity > 1) printf(psvFurther);
        Core_DoPass(0);
        /* keep doing passes as long as there are no real errors and the */
        /* number of undefined values is decreasing */
      } while(nNeedValue_Now && (nNeedValue_Now < nNeedValue_Last));
    }

    if(nNeedValue_Now) {
      /* Extra pass for finding "NeedValue" type errors */
      if(Process_Verbosity > 1) printf(psvFind);
      Core_DoPass(PASS_ISERROR);
    } else {
      Output_ToFile(PC_Lowest, PC_Highest);/* save output file */
    }

    /* if wanted, dump labels */
    Label_DumpAll();

    FREE_TRY();/* free memory (items) at end of try */
  } while(0);/* add repetition switch here */

  CleanExit(EXIT_SUCCESS);/* that's all folks ! */
  return(0);
}

/*
 * Main routines
 */

/*
 * The core of every pass
 */
void Core_DoPass(int Flags) {
  FILE* Filehandle;

  pToBeFreed_Pass   = NULL;
  Pass_Flags        = Flags;/* Flags im Pass */
  nNeedValue_Now        = 0;/* no "NeedValue" errors yet */
  nErrors           = 0;/* no errors yet */
  CPU_fPCdefined    = FALSE;/* Flag for "PC defined" */
  PC_Lowest         = 0;/* must be zero'd so "!initmem" can be used before "*=" */
  PC_Highest        = 0;
  SegmentStart      = 0;
  SegmentMax        = OUTBUFFERSIZE-1;
  pSegmentList      = NULL;
  PC_inc            = 0;/* Increase PCs by this amount at end of line */
  Context_CodeTable = CODETABLE_RAW;/* Handle of current code table */

  /* Open toplevel file */
  if((Filehandle = File_OpenFile(pnfTop, FILE_READBINARY)) == NULL) {
    printf(Error_CannotOpenTopFile);
    CleanExit(EXIT_FAILURE);
  }

  EndReason = RNONE;

  Context_InitForPass(Filehandle);
  CPU_InitForPass();

  /* do the work */
  ParseBlock();
  if(EndReason != RENDOFFILE) ThrowError(Exception_eEndOfFile);
  EndReason = RNONE;/* Clear global variable */

  String_EndSegment();
  /* Reached end of source, so end of pass... */
  File_CloseFile(Filehandle);/* close toplevel source */
  FREE_PASS();
  if(nErrors) CleanExit(EXIT_FAILURE);
}

/*
 * Parse block, beginning with next byte. Has to be re-entrant.
 */
void ParseBlock() {
  int fPC2Label;

  do {
    /* As long as there are statements */
    fPC2Label = FALSE;/* no "label = pc" definition yet */
    GetByte();/* read start of statement */
    do {
/* Parse until end of statement is reached. Only loops if statement contains
 * "label = pc" definition and something else; or when "!ifdef" is true.
 */
      switch(GotByte) {

        case 0:/* Ignore now, act later (stops zero from being "default") */
        break;

        case '+':
        Macro_Call();
        break;

        case '!':
        ParsePseudoOpcode();
        break;

        case '*':
        CPU_SetPC();
        break;

        case '.':
        if(fPC2Label) {
          ThrowError(Exception_Syntax);
          SkipRest();
        } else {
          if(Stream_ReadKeyword(MiscString, TRUE))
            Label_ParseDefinition(Context_CurrentZone);
          fPC2Label = TRUE;
        }
        break;

        default:
        if(pFlagTable[GotByte] & BYTEIS_ILLEGAL) {
          ThrowError(Exception_Syntax);
          SkipRest();
        } else {
          /* after ReadKeyword: GotByte = illegal char */
          /* It is only a label if it isn't a mnemonic */
          if(Mnemo_IsInvalid(Stream_ReadKeyword(MiscString, FALSE))) {
            if(fPC2Label) {
              ThrowError(Exception_Syntax);
              SkipRest();
            } else {
              Label_ParseDefinition(ZONE_GLOBAL);
              fPC2Label = TRUE;
            }
          }
        }
      }
    } while(GotByte);
    CPU_PC += PC_inc;
    PC_Mem += PC_inc;
    if(PC_Mem > PC_Highest) PC_Highest = PC_Mem;
    PC_inc = 0;
  } while(EndReason == RNONE);
}

/*
 * Skip remainder of statement, for example on error
 */
void SkipRest() {
  while(GotByte) GetByte();/* Read characters until zero */
}

/*
 * Ensure that the remainder of the current statement is empty, for example
 * after mnemonics using implicit addressing.
 */
void EnsureEOL() {/* GotByte = first char to test */
  SKIPSPACE;
  if(GotByte) {
    ThrowError(Exception_Syntax);
    SkipRest();
  }
}

/*
 * Error handling
 */

/*
 * This routine handles problems that may be solved by performing further
 * passes: "NeedValue" type errors.
 * If the current pass has the corresponding flagbit set, this routine will not
 * only count, but also show these errors to the user.
 */
void NeedValue() {
  nNeedValue_Now++;
  if(Pass_Flags & PASS_ISERROR) ThrowError(Exception_NeedValue);
}

/*
 * Output a warning.
 * This means the produced code looks as expected. But there has been a
 * situation that should be reported to the user, for example ACME may have
 * assembled a 16-bit parameter with an 8-bit value.
 */
void ThrowWarning(char* Message) {
  PLATFORM_WARNING(Message);
  ThrowMessage(Message, psWarning);
}

/*
 * Output an error.
 * This means something went wrong in a way that implies that the output
 * almost for sure won't look like expected, for example when there was a
 * syntax error. The assembler will try to go on with the assembly though, so
 * the user gets to know about more than one of his typos at a time.
 */
void ThrowError(char* Message) {
  PLATFORM_ERROR(Message);
  ThrowMessage(Message, psError);
  nErrors++;
  if(nErrors >= MAXERRORS) CleanExit(EXIT_FAILURE);
}

/*
 * Output a serious error, stopping assembly.
 * Serious errors are those that make it impossible to go on with the
 * assembly. Example: "!fill" without a parameter - the program counter cannot
 * be set correctly in this case, so proceeding would be of no use at all.
 */
void ThrowSerious(char* Message) {
  PLATFORM_SERIOUS(Message);
  ThrowMessage(Message, psSerious);
  CleanExit(EXIT_FAILURE);
}

/*
 * This routine will do the actual output for warnings, errors and serious
 * errors. It shows the given message string, as well as the current
 * context: Filename, line number, source type and source title.
 */
void ThrowMessage(char* Message, char* Type) {
  printf("%s - File %s", Type, Context[nContext].pSourceFile);
  printf(", Line %d",          Context[nContext].nLines);
  printf(" (%s ",              Context[nContext].pSourceType);
  printf("%s): %s\n",          Context[nContext].pSourceTitle, Message);
}

/*
 * Handle bugs
 */
void BugFound(char* Message) {
  ThrowWarning(BugInACME);
  ThrowSerious(Message);
}

/*
 * Get command line arguments
 */

/*
 * This routine sets "pnfTop" to point to the given filename string and acts
 * upon the supported option character(s).
 */
void ReadParameters(int argc, byte* argv[]) {
  byte c;
  int  i = 1,
       j;

  while((i < argc) && (argv[i][0] == '-')) {
    j = 1;
    while((c = argv[i][j++]) != 0) {
      switch(c) {

        PLATFORM_SWITCHCODE /* platform specific switches are inserted here */

        case 'h':
        String_ShowInfo();/* show syntax message */
        break;

        case 'v':
        Process_Verbosity += 1;/* verbose mode */
        if((argv[i][j] >= '0') && (argv[i][j] <= '9')) {
          Process_Verbosity = argv[i][j++] - '0';
        }
        break;

        default:
        printf(Error_UkSwitch, c);
      }
    }
    i++;
  }
  if(i == argc) {
    if(i - 1) {
      printf(Error_NoTop);
      CleanExit(EXIT_FAILURE);
    } else {
      String_ShowInfo();
      CleanExit(EXIT_SUCCESS);
    }
  } else {
    pnfTop = argv[i];
  }
}

/*
 * Tidy up before exiting
 */
void CleanExit(int exitstatus) {
  Label_DumpAll();/* if wanted and not yet done, dump labels */
  PLATFORM_EXIT;
  File_CloseAll();/* close files */
  FREE_PROCESS();/* Free memory */
  exit(exitstatus);
}

/*
 * Conversion routines...
 */

/*
 * Convert character using given conversion table
 */
byte ConvertChar(byte b, int hCT) {
  switch(hCT) {

    case CODETABLE_RAW:
    return(b);

    case CODETABLE_PET:
    if((b > 64) && (b <  91)) return(b + 128);
    if((b > 96) && (b < 123)) return(b -  32);
    return(b);

    case CODETABLE_SCR:
    if((b > 96) && (b <123))
      return(b - 96);/* shift uppercase down */
    if((b > 90) && (b < 96))
      return(b - 64);/* shift [\]^_ down */
    if(b == 96) return(64);/* shift ` down */
    if(b == 64) return(0);/* shift @ down */
    return(b);

    case CODETABLE_FILE:
    return(pConvTable[b]);

    default:
    ThrowError(Exception_NotYet);
    return(b);
  }
}

/*
 * Link segment data into segment chain
 */
void Segment_Link(value Start, value Length) {
  SegmentStruct* p;

  /* may be faster if list is ordered !! */
  p = ALLOC_PASS(sizeof(SegmentStruct));
  p->Start = Start;
  p->Length = Length;
  p->Next = pSegmentList;
  pSegmentList = p;
}

/*
 * Set up new SegmentMax value according to the given program counter. Just
 * find the next segment start and subtract 1.
 */
void Segment_FindMax(value NewPC) {
  SegmentStruct* p;

  /* may be faster if list is ordered !! */
  SegmentMax = OUTBUFFERSIZE;/* will be decremented later ! */
  p = pSegmentList;
  while(p) {
    if(p->Start > NewPC) {
      if(p->Start < SegmentMax) SegmentMax = p->Start;
    }
    p = p->Next;
  }
  SegmentMax--;/* last free address available */
}

/*
 * Check whether given PC is inside segment.
 */
void Segment_Check(value NewPC) {
  SegmentStruct* p;

  p = pSegmentList;
  while(p) {
    if(NewPC >= p->Start) {
      if(NewPC < (p->Start)+(p->Length)) ThrowWarning(Exception_SegInSeg);
    }
    p = p->Next;
  }
}

/*
 * Allocate memory and die if not available
 */
void* SafeAlloc(size_t Size) {
  void* p;

  if((p = malloc(Size)) == NULL) ThrowSerious(Exception_NoMemLeft);
  return(p);
}

/*
 * Allocate memory and prepare for auto-freeing at end of pass/try/process
 */
void* MyAlloc(size_t Size, void** Type) {
  void** p;

  p = malloc(Size + sizeof(void*));
  if(p) {
    p[0] = Type[0];
    Type[0] = p;
    return(&p[1]);
  } else ThrowSerious(Exception_NoMemLeft);
  return(NULL);
}

/*
 * Free memory at end of pass/try/process
 */
void MyFree(void* Type) {
  void **p1,
       **p2;

  p1 = Type;
  while((p2 = p1)) {
    p1 = p2[0];
    free(p2);
  }
}

/*
 * Initialisation at start of process
 */
void Process_Init() {
  pToBeFreed_Process = NULL;
  Platform_FindLib();/* only do once */
  Tree_InitROM();/* only needed once, not in every try */
}

/*
 * Initialisation at start of try
 */
void Try_Init() {
    pToBeFreed_Try = NULL;
}
