/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Flow control stuff (loops etc.)
 */

#include "alu.h"
#include "data.h"
#include "flowpo.h"
#include "core.h"
#include "mnemo.h"
#include "stream.h"
#include "strings.h"

/*
 * Constants
 */

/*
 * Variables
 */

context Context[MAXCONTEXTS];
byte    pTitle [MAXCONTEXTS][LSMAX + 1];

/*
 * Start offset assembly (add block support and deprecate old method !!!)
 */
int FlowPO_pseudopc() {
  CPU_PC = ALU_GetValue_Strict();
  return(ENSURE_EOL);
}

/*
 * End offset assembly (deprecate !!!)
 */
int FlowPO_realpc() {
  CPU_PC = PC_Mem;/* deactivate offset assembly */
  return(ENSURE_EOL);
}

/*
 * Switch to new zone ("!zone" or "!zn")
 */
int FlowPO_zone() {
  CONTEXT_NEWZONE;
  Context[nContext].pSourceTitle = psUntitled;
  SKIPSPACE;
  if(GotByte) {
    if(Stream_ReadKeyword(&pTitle[nContext][0], FALSE))
      Context[nContext].pSourceTitle = &pTitle[nContext][0];
  }
  return(ENSURE_EOL);
}

/*
 * Start subzone ("!subzone" or "!sz"). Has to be re-entrant.
 */
int FlowPO_subzone() {
  zone  OldZone;
  byte* OldTitle;
  byte* NewTitle = NULL;
  int   Len;

  /* remember old values */
  OldZone  = Context_CurrentZone;
  OldTitle = Context[nContext].pSourceTitle;

  /* set new values */
  CONTEXT_NEWZONE;
  Context[nContext].pSourceTitle = psUntitled;

  /* Check for zone title */
  SKIPSPACE;
  if((pFlagTable[GotByte] & BYTEIS_ILLEGAL) == 0) {
    Len = Stream_ReadKeyword(BlockBuffer, FALSE);
    if(Len) {
      NewTitle = (byte*) SafeAlloc(Len + 1);
      memcpy(NewTitle, BlockBuffer, Len + 1);
      Context[nContext].pSourceTitle = NewTitle;
    }
    SKIPSPACE;
  }
  if(GotByte != '{') ThrowSerious(Exception_NoLeftBrace);

  ParseBlock();
  if(EndReason != RRIGHTBRACE) ThrowError(Exception_NoRightBrace);
  EndReason = RNONE;/* Clear global variable */
  GetByte();

  if(NewTitle) free(NewTitle);

  /* restore old values */
  Context_CurrentZone = OldZone;
  Context[nContext].pSourceTitle = OldTitle;

  return(ENSURE_EOL);
}

/*
 * End of source file ("!endoffile" or "!eof")
 */
int FlowPO_eof() {
  EndReason = RENDOFFILE;
  return(ENSURE_EOL);
}

/*
 * Include source file ("!source" or "!src"). Has to be re-entrant.
 */
int FlowPO_source() {/* GotByte = illegal character */
  FILE* Filehandle;
  byte  Filename[LNPMAX + 1];/* file name */

  if(Stream_ReadFilename(Filename, LNPMAX, TRUE) == 0) return(SKIP_REST);
  if((Filehandle = File_OpenFile(Filename, FILE_READBINARY))) {
    /* Initialize new context */
    Context_New();
    Context[nContext].nLines       = 0;
    Context[nContext].hByteSource  = BYTESOURCE_FILE;
    Context[nContext].u.hFile      = Filehandle;
    Context[nContext].pSourceFile  = Filename;
    Context[nContext].pSourceTitle = Context[nContext-1].pSourceTitle;
    Context[nContext].pSourceType  = Context[nContext-1].pSourceType;
    Context[nContext].OldFileByte  = 0;
    Context[nContext].OldRawByte   = 0;
    /* Parse block and check end reason */
    ParseBlock();
    if(EndReason != RENDOFFILE) ThrowError(Exception_eEndOfFile);
    EndReason = RNONE;/* Clear global variable */
    File_CloseFile(Filehandle);/* close sublevel source */
    nContext--;/* Old context */
  }
  return(ENSURE_EOL);
}

/*
 * Conditional assembly ("!ifdef")
 */
int FlowPO_ifdef() {/* GotByte = illegal character */
  int       fDefined = FALSE;
  treeItem* Item;
  label*    Label;
  zone      Zone = ZONE_GLOBAL;

  SKIPSPACE;
  if(GotByte == '.') {
    Zone = Context_CurrentZone;
    GetByte();
  }
  if(Stream_ReadKeyword(MiscString, FALSE) == 0) return(SKIP_REST);
  Item = Tree_ScanRAM(Zone | BM_LABEL, FALSE);
  if(Item) {
    Label = (label*) Item->Body.Pointer;
    /* in first pass, count usage */
    if(Pass_Flags & PASS_ISFIRST) Label->Usage++;
    if(Label->Flags & MVALUE_DEFINED) fDefined = TRUE;
  }
  SKIPSPACE;
  if(GotByte == '{') {
    Block_2Blocks(fDefined);
  } else {
    if(fDefined == FALSE) return(SKIP_REST);
    else                  return(PARSE_REST);
  }
  return(ENSURE_EOL);
}

/*
 * Conditional assembly ("!if"). Has to be re-entrant.
 */
int FlowPO_if() {/* GotByte = illegal character */
  ALU_GetValue_Strict();
  if(GotByte != '{') ThrowSerious(Exception_NoLeftBrace);
  Block_2Blocks((int) ALU_Value);
  return(ENSURE_EOL);
}

/*
 * Looping assembly ("!for"). Has to be re-entrant.
 */
int FlowPO_for() {/* GotByte = illegal character */
  value     vEnd,
            vNow;
  int       Line,
            fb,
            ffvEnd;
  byte     *pb,
           *pbOwn;
  label*    Label;
  zone      Zone = ZONE_GLOBAL;

  Line = Context[nContext].nLines;/* Remember line number */
  SKIPSPACE;
  if(GotByte == '.') {
    Zone = Context_CurrentZone;
    GetByte();
  }
  /* after ReadKeyword: GotByte = illegal char */
  if(Stream_ReadKeyword(MiscString, FALSE) == 0) return(SKIP_REST);

  fb = Mnemo_GetForceBit();/* skips spaces */
  Label = Label_Find(Zone, fb);
  if(fb) {
    Label->Flags &= ~MVALUE_FORCEBITS;
    Label->Flags |= fb;
  }
  if(GotByte != ',') {
    ThrowError(Exception_Syntax);
    return(SKIP_REST);
  }
  GetByte();/* proceed with next char */
  vEnd = ALU_GetValue_Strict();
/* prepare flags, size of *end* value doesn't matter for *all* loops */
  ffvEnd = ALU_Flags & ~MVALUE_FORCEBITS;
  if(vEnd < 0) ThrowSerious(Exception_TooBig);/* No negative loops ! */

  if(GotByte != '{') ThrowSerious(Exception_NoLeftBrace);

  pb = Block_Store(BlockBuffer);/* Read block to buffer */
  EndReason = RNONE;/* Clear global variable */
/* bug fix */
  NEXTANDSKIPSPACE;/* proceed with next char */
  if(GotByte != 0) ThrowSerious(Exception_Syntax);/* necessary? !! */
/* now blanks after "}" shouldn't give errors anymore */

  pbOwn = SafeAlloc(pb - BlockBuffer);/* Get memory block */

  memcpy(pbOwn, BlockBuffer, pb - BlockBuffer);
  /* Initialize new context */
  Context_New();
  Context[nContext].hByteSource  = BYTESOURCE_RAM;
  Context[nContext].pSourceFile  = Context[nContext-1].pSourceFile;
  Context[nContext].pSourceTitle = Context[nContext-1].pSourceTitle;
  Context[nContext].pSourceType  = Context[nContext-1].pSourceType;

  if(vEnd) {/* skip loop if counter = 0 */
    vNow = 1;
    while(vNow <= vEnd) {
      /* set counter */
      Label_SetValue(Label, vNow, ffvEnd, TRUE);
      /* now actually do some work */
      Context[nContext].OldFileByte  = 0;
      Context[nContext].OldRawByte = ' ';/* Don't inc line count */
      Context[nContext].u.pRAM = pbOwn;/* Begin at beginning */
      Context[nContext].nLines = Line;/* Restore line number */
      ParseBlock();
      if(EndReason == RENDOFBLOCK) {
        EndReason = RNONE;/* Clear global variable */
      } else ThrowSerious(Exception_NoRightBrace);
      vNow++;/* increase counter */
    }
  } else Label_SetValue(Label, 0, ffvEnd, TRUE);/* set counter */
  free(pbOwn);/* Free memory */
  nContext--;/* Old context */
  return(ENSURE_EOL);
}

/*
 * Looping assembly ("!do"). Has to be re-entrant.
 */
int FlowPO_do() {/* GotByte = illegal character */
  value  Condition;
  int    line,
         fGoOn = TRUE;
  byte  *pb,
        *pbOwn,
         id;

  line = Context[nContext].nLines;/* Remember line number */
  pb = Flow_StoreCondition(BlockBuffer);/* Read head to buffer */
  if(GotByte != '{') ThrowSerious(Exception_NoLeftBrace);

  pb = Block_Store(pb);/* Read block to buffer */
  EndReason = RNONE;/* Clear global variable */
  GetByte();/* proceed with next char */
  pb = Flow_StoreCondition(pb);/* Read tail to buffer */
  if(GotByte != 0) ThrowSerious(Exception_Syntax);/* necessary? !! */

  pbOwn = SafeAlloc(pb - BlockBuffer);/* Get memory for this incarnation */

  memcpy(pbOwn, BlockBuffer, pb - BlockBuffer);
  /* Initialize new context */
  Context_New();
  Context[nContext].hByteSource  = BYTESOURCE_RAM;
  Context[nContext].pSourceFile  = Context[nContext-1].pSourceFile;
  Context[nContext].pSourceTitle = Context[nContext-1].pSourceTitle;
  Context[nContext].pSourceType  = Context[nContext-1].pSourceType;
  Context[nContext].OldFileByte  = 0;
  Context[nContext].OldRawByte   = ' ';/* Don't increase line number */
  do {
    /* Start of loop */
    Context[nContext].u.pRAM = pbOwn;/* Begin at beginning */
    Context[nContext].nLines = line;/* Use remembered line number */
    /* Check head condition */
    id = GetByte();/* get condition type (until/while) */
    GetByte();/* proceed with next char */
    Condition = ALU_GetValue_Strict();
    if(EndReason != RENDOFBLOCK) ThrowSerious(Exception_Syntax);

    EndReason = RNONE;/* Clear global variable */
    if(id == ID_UNTIL) Condition = !Condition;
    if(Condition) {
      Context[nContext].nLines = line-1;/* Kluge !! */
      ParseBlock();
      if(EndReason != RENDOFBLOCK) ThrowSerious(Exception_NoRightBrace);

      Context[nContext].nLines-=1;/* Kluge !! */
      EndReason = RNONE;/* Clear global variable */
      /* Check tail condition */
      id = GetByte();/* get condition type (until/while) */
      GetByte();/* proceed with next char */
      Condition = ALU_GetValue_Strict();
      if(EndReason != RENDOFBLOCK) ThrowSerious(Exception_Syntax);

      EndReason = RNONE;/* Clear global variable */
      if(id == ID_UNTIL) Condition = !Condition;
      if(Condition == 0) fGoOn = FALSE;
    } else fGoOn = FALSE;
  } while(fGoOn);
  free(pbOwn);/* Free memory */
  nContext--;/* Old context */
  return(ENSURE_EOL);
}

/*
 * Try to read a condition and store it at the given location. Add separator.
 */
byte* Flow_StoreCondition(byte* pb) {
  treeItem* p;

  SKIPSPACE;
  if((GotByte == '{') || (GotByte == 0)) {/* Accept both head *and* tail */
    /* Write pseudo condition (always TRUE) into buffer, so we don't have */
    /* to worry about nonexistant conditions */
    *(pb++) = ID_WHILE;
    *(pb++) = '1';
  } else {
    /* Write given condition into buffer */
    if(Stream_ReadKeyword(MiscString, FALSE)) {
      /* Search for tree item. Don't create */
      MiscString_ToLower();
      p = Tree_ScanROM(MiscString, HTYPE_UNTILWHILE);
      if(p) {
        *(pb++) = p->Body.Opcode.Code;/* store condition type handle */
        while((GotByte != '{') && GotByte) {
          if(pb == BlockBufEnd) ThrowSerious(Exception_NoMemLeft);
          *(pb++) = GotByte;
          GetByte();
        }
      } else ThrowError(Exception_Syntax);
    }
  }
  *(pb++) = BLOCK_FIELDSEPARATOR;/* Terminate / separate */
  return(pb);
}
