/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Macro stuff
 */

#include "macro.h"

/*
 * Constants
 */

char psMacro[] = "Macro";
/* Error messages */
char Exception_MacroTwice[] = "Macro already defined.";
char Exception_TooManyMP[] = "Too many macro parameters.";
char Exception_UkMacro[] = "Macro not defined.";
char Exception_WrongNr[] = "Wrong number of macro parameters.";

/*
 * Variables
 */

/* Temporary storage during macro call */
treeItem* MaPa_Ref [MACRO_MAXPARAMETERS];/* pointers to named parameter */
label*    MaPa_Expr[MACRO_MAXPARAMETERS];/* pointers to expression result */

/*
 * Macro definition ("!macro").
 */
int Macro_Definition() {
  treeItem*  Item;
  byte      *WritePointer,
            *ReadPointer,
             b;
  macro*     Macro;
  zone       Zone = ZONE_GLOBAL;
  int        Line = Context[nContext].nLines;/* Remember line nr */

  WritePointer = BlockBuffer;/* points to start of sequential data */
  SKIPSPACE;
  if(Pass_Flags & PASS_ISFIRST) {
    /* only parse in first pass */

    /* Parse macro definition */
    if(GotByte == '.') {/* if local, set zone number */
      Zone = Context_CurrentZone;
      GetByte();
    }
    /* after ReadKeyword: GotByte = illegal char */
    if(Stream_ReadKeyword(MiscString, FALSE)) {/* read macro title */
      Item = Tree_ScanRAM(Zone | BM_MACRO, TRUE);/* find macro */
      if(Tree_ItemCreated == FALSE) ThrowSerious(Exception_MacroTwice);

      /* Item was created, so fill it */

      /* Copy file name to buffer (includes terminator) */
      ReadPointer = Context[nContext].pSourceFile;
      do {
        b = *(ReadPointer++);
        *(WritePointer++) = b;
        if(WritePointer == BlockBufEnd) ThrowSerious(Exception_NoMemLeft);

      } while(b);

      /* Copy parameters to buffer */
      SKIPSPACE;
      while((GotByte != '{') && GotByte) {
        *(WritePointer++) = GotByte;
        if(WritePointer == BlockBufEnd) ThrowSerious(Exception_NoMemLeft);

        GetByte();
      }

      if(GotByte == 0) ThrowSerious(Exception_NoLeftBrace);

      *(WritePointer++) = BLOCK_FIELDSEPARATOR;/* Terminate / separate */
      Context[nContext].OldRawByte = ' ';/* Kluge to skip spaces */
      WritePointer = Block_Store(WritePointer);/* Store macro body */
      if(EndReason != RRIGHTBRACE) ThrowSerious(Exception_NoRightBrace);

      EndReason = RNONE;/* Clear global variable */
      GetByte();/* Proceed with next character */
      /* Get own memory for macro */
      Macro = (macro*) ALLOC_TRY(sizeof(macro) + (WritePointer - BlockBuffer));
      /* Set up macro structure in new memory block */
      Macro->LineNumber = Line;
      memcpy(&Macro->Body[0], BlockBuffer, WritePointer - BlockBuffer);
      /* Remember pointer to new memory block in tree item */
      Item->Body.Pointer = Macro;
    }
  } else {
    /* in further passes, just skip definition */

    /* Skip macro definition */
    while((GotByte != '{') && GotByte) GetByte();
    if(GotByte) {
      Block_Store(NULL);
      EndReason = RNONE;/* Clear global variable */
      GetByte();/* Proceed with next character */
    } else ThrowError(Exception_NoLeftBrace);
  }
  return(ENSURE_EOL);
}

/*
 * Macro call ("+<macroname>"). Has to be re-entrant.
 */
void Macro_Call() {/* GotByte = "+" */
  treeItem* Item;
  int       TempReason,
            nParaC = 0,/* Number of parameters in call (and also counter) */
            nParaD = 0;/* Number of parameters in definition */
  macro*    Macro;
  byte*     ReadPointer;
  zone      OuterZone;
  zone      InnerZone,/* zone value inside macro */
            DefZone = ZONE_GLOBAL;/* zone macro was defined in */

  /* remember old values */
  OuterZone = Context_CurrentZone;

  GetByte();
  if(GotByte == '.') {
    /* Macro is local, so use current zone value */
    DefZone = Context_CurrentZone;
    GetByte();
  }
  /* after ReadKeyword: GotByte = illegal char */
  if(Stream_ReadKeyword(MiscString, FALSE)) {
    Item = Tree_ScanRAM(DefZone | BM_MACRO, FALSE);
    if(Item) {
      Macro = Item->Body.Pointer;/* Get pointer to macro */
      /* Skip file name stored in macro */
      ReadPointer = &Macro->Body[0];
      while(*(ReadPointer++));
      /* Read parameters of call */
      SKIPSPACE;/* necessary ? !! */
      if(GotByte) {
        do {
          /* Read parameters up to end of statement, store in static array */
          if(nParaC == MACRO_MAXPARAMETERS) ThrowSerious(Exception_TooManyMP);
          MaPa_Expr[nParaC] = Label_Create(0, 0);
/* When call by reference comes along, don't create. */
          MaPa_Expr[nParaC]->Value = ALU_GetValue_Medium();
          MaPa_Expr[nParaC]->Flags = ALU_Flags;
          nParaC++;
        } while(Stream_Comma());
      }
      /* nParaC = number of parameters in call */
      TempReason = EndReason;/* buffer current end reason */

      /* Initialize new context */
      Context_New();
      CONTEXT_NEWZONE;
      Context[nContext].nLines       = Macro->LineNumber;/* Line of def. */
      Context[nContext].hByteSource  = BYTESOURCE_RAM;
      Context[nContext].u.pRAM       = ReadPointer;
      Context[nContext].pSourceFile  = &Macro->Body[0];
      Context[nContext].pSourceTitle = Item->ID_String;/* Macro title */
      Context[nContext].pSourceType  = psMacro;
      Context[nContext].OldFileByte  = 0;
      Context[nContext].OldRawByte   = ' ';/* Don't increase line number */

      /* Search for parameter labels and store pointers in static array */
      if(GetByte()) {/* Get first byte of "new" context */
        do {
          SKIPSPACE;/* necessary ? !! */
          InnerZone = ZONE_GLOBAL;
          if(GotByte == '.') {
            /* parameter is local, so use new zone value */
            InnerZone = Context_CurrentZone;
            GetByte();
          }
          /* after ReadKeyword: GotByte = illegal char */
          if(Stream_ReadKeyword(MiscString, FALSE)) {
            if(nParaD == MACRO_MAXPARAMETERS) ThrowSerious(Exception_TooManyMP);
            MaPa_Ref[nParaD] = Tree_ScanRAM(InnerZone | BM_LABEL, TRUE);
            if(Tree_ItemCreated) MaPa_Ref[nParaD]->Body.Pointer = NULL;
            nParaD++;
          }
        } while(Stream_Comma());
        EnsureEOL();
      }
      Context[nContext].OldRawByte = ' ';/* Don't increase line number */
      EndReason = RNONE;/* Clear global variable */
      /* Compare number of parameters in call and definition */
      if(nParaC == nParaD) {
        /* Set parameters */
        while(nParaC--) {
          if(MaPa_Ref[nParaC]->Body.Pointer) {
            Label_Remove(MaPa_Ref[nParaC]->Body.Pointer);
          }
          MaPa_Ref[nParaC]->Body.Pointer = MaPa_Expr[nParaC];
          MaPa_Expr[nParaC]->Refs++;
        }
        /* Actually *parse* the macro body */
        ParseBlock();
        if(EndReason == RENDOFBLOCK) {
          EndReason = TempReason;/* Restore global variable */
          nContext--;/* Old context */
        } else ThrowSerious(Exception_NoRightBrace);
      } else {
        /* Show two error messages, pointing to both definition and use */
        ThrowError(Exception_WrongNr);
        nContext--;/* Old context */
        ThrowSerious(Exception_WrongNr);
      }
    } else ThrowError(Exception_UkMacro);
  }

  /* restore old values */
  Context_CurrentZone = OuterZone;

  EnsureEOL();
}
