/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Mnemonics stuff
 */

#include "alu.h"
#include "data.h"
#include "core.h"
#include "mnemo.h"
#include "stream.h"
#include "strings.h"

/*
 * Constants
 */

/*
 * The mnemonics are split up into groups, each group has its own routine to
 * be dealt with:
 *
 * G_IMP  : Mnemonics using only implicit addressing. Byte value = opcode.
 * G_RELS : The short branch instructions.            Byte value = opcode.
 * G_RELL : Mnemonics with 16bit relative addressing. Byte value = opcode.
 * G_ACCM : The main accumulator stuff. Byte value = "base" opcode;
 *          an offset (depending on the addressing mode) has to be added.
 * G_ACCS : As G_ACCM, but without immediate addressing mode.
 *          Only applies to STA and some 6510 illegals.
 * The "MISC" groups contain various mnemonics with different addressing
 * modes. Byte value = row to use when reading the code table.
 * G_MISCO: not affected by different immediate addressing,
 * G_MISCA: affected by length of accumulator
 * G_MISCR: affected by length of index registers
 * G_MOVE : The "move" commands.                      Byte value = opcode.
 * G_PEI  : "pei", it only has one addressing mode.   Byte value = opcode.
 * G_JUMP : The jump instructions.
 *          Byte value = column to use when reading the code table.
 */
enum {
  G_ACCM ,/* =  3,/ * accu main */
  G_ACCS ,/* =  4,/ * STA */
  G_IMP  ,/* =  0,/ * implicit */
  G_JUMP ,/* = 10,/ * jump */
  G_MISCA,/* =  6,/ * Misc, depends on accu length */
  G_MISCO,/* =  5,/ * Misc, other */
  G_MISCR,/* =  7,/ * Misc, depends on index register length */
  G_MOVE ,/* =  8,/ * MVP, MVN */
  G_PEI  ,/* =  9,/ * PEI */
  G_RELL ,/* =  2,/ * long relative */
  G_RELS ,/* =  1,/ * short relative */
};

/*
 * Code table for groups MISCO, MISCA, MISCR:
 *
 * This table is needed for finding out the correct code in cases when
 * there are no general rules. By reading the mnemonic's byte value (from the
 * mnemotable), the assembler finds out the row to use here. The column
 * depends on the used addressing mode. A zero entry in these tables means
 * that the combination of mnemonic and addressing mode is illegal.
 *
 * Format:
 *
 *  implicit/accu     16bit
 *  |     #8bit       |     8bit,x
 *  |     |     8bit  |     |     16bit,x
 *  |     |     |     |     |     |     8bit,y
 *  |     |     |     |     |     |     |     16bit,y
 *  |     |     |     |     |     |     |     |
 */

byte GroupMISC[160] = {
  0x0a,    0, 0x06, 0x0e, 0x16, 0x1e,    0,    0, /* asl */
     0, 0x89, 0x24, 0x2c, 0x34, 0x3c,    0,    0, /* bit */
     0, 0xe0, 0xe4, 0xec,    0,    0,    0,    0, /* cpx */
     0, 0xc0, 0xc4, 0xcc,    0,    0,    0,    0, /* cpy */
  0x3a,    0, 0xc6, 0xce, 0xd6, 0xde,    0,    0, /* dec */
  0x1a,    0, 0xe6, 0xee, 0xf6, 0xfe,    0,    0, /* inc */
     0, 0xa2, 0xa6, 0xae,    0,    0, 0xb6, 0xbe, /* ldx */
     0, 0xa0, 0xa4, 0xac, 0xb4, 0xbc,    0,    0, /* ldy */
  0x4a,    0, 0x46, 0x4e, 0x56, 0x5e,    0,    0, /* lsr */
  0x2a,    0, 0x26, 0x2e, 0x36, 0x3e,    0,    0, /* rol */
  0x6a,    0, 0x66, 0x6e, 0x76, 0x7e,    0,    0, /* ror */
     0,    0, 0x86, 0x8e,    0,    0, 0x96,    0, /* stx */
     0,    0, 0x84, 0x8c, 0x94,    0,    0,    0, /* sty */
     0,    0, 0x64, 0x9c, 0x74, 0x9e,    0,    0, /* stz */
     0,    0, 0x14, 0x1c,    0,    0,    0,    0, /* trb */
     0,    0, 0x04, 0x0c,    0,    0,    0,    0, /* tsb */
     0,    0, 0x02,    0,    0,    0,    0,    0, /* cop */
     0,    0,    0, 0xf4,    0,    0,    0,    0, /* pea */
     0, 0xc2,    0,    0,    0,    0,    0,    0, /* rep */
     0, 0xe2,    0,    0,    0,    0,    0,    0  /* sep */
};

/*
 * Code table for group JUMP:
 *
 * This table is needed for finding out the correct code when the mnemonic is
 * "jmp" or "jsr" (or the long versions "jml" and "jsl").
 * By reading the mnemonic's byte value (from the mnemotable), the assembler
 * finds out the column to use here. The row depends on the used addressing
 * mode. A zero entry in these tables means that the combination of mnemonic
 * and addressing mode is illegal.
 *
 * Format:
 *
 * jmp   jsr   jml    jsl
 */

byte GroupJUMP[20] = {
  0x4c, 0x20,    0,     0, /*  16bit    */
  0x6c,    0,    0,     0, /* (16bit)   */
  0x7c, 0xfc,    0,     0, /* (16bit,x) */
  0x5c, 0x22, 0x5c,  0x22, /*  24bit    */
  0xdc,    0, 0xdc,     0  /* [16bit]   */
};

/* error message strings */
char Exception_HighbyteZero[] = "Using oversized addressing mode.";
char Exception_CombCA[] = "Illegal combination of command and addressing mode.";
char Exception_CombCP[] = "Illegal combination of command and postfix.";
char Exception_TooFar[] = "Target out of range.";
char Exception_Negative[] = "Negative value - cannot choose addressing mode.";
char Exception_UkPF[] = "Illegal postfix.";
char BIA_UkGI[] = "illegal group index";

/*
 * Variables
 */

byte MnemoString[MAXSIZE_MNEMO+1];/* temporary mnemonic buffer */
int  hAM_Now;/* Handle of current addressing mode */

/*
 * Check whether the given code byte runs on the current CPU and output it.
 */
void Mnemo_PutCodeByte(byte b) {
  if(pFlagTable[b] & CPU_TypeBit) {
    Output_Byte((value) b);
  } else {
    ThrowError(Exception_WrongCPU);
    SkipRest();
  }
}

/*
 * Branches to different checks according to current cpu
 */
int Mnemo_IsInvalid(int len) {
  switch(CPU_TypeBit) {

    case CPUBIT_6502:
    case CPUBIT_6510:
    case CPUBIT_65C02:
    case CPUBIT_65816:
    return(Mnemo65_IsInvalid(len));

    case CPUBIT_Z80:
    return(Mnemo80_IsInvalid(len));

    default:
    ThrowError(Exception_NotYet);
    return(TRUE);
  }
}

/*
 * 65xx stuff
 */

/*
 * Test whether mnemonic is known by current 65xx cpu
 */
int Mnemo65_IsInvalid(int len) {
  if(len != 3) return(TRUE);
  MnemoString[0] = MiscString[0] | 32;/* to lower case */
  MnemoString[1] = MiscString[1] | 32;
  MnemoString[2] = MiscString[2] | 32;
  MnemoString[3] = 0;/* terminate (needed by item searcher) */

/* Look out for 6502 mnemonics */
  if(Mnemo65_Check(HTYPE_6502MNEMO)) return(FALSE);

  if(CPU_TypeBit == CPUBIT_6502) return(TRUE);
/* Look out for illegal 6510 mnemonics */
  if(CPU_TypeBit == CPUBIT_6510) return(!Mnemo65_Check(HTYPE_6510MNEMO));

/* Look out for 65c02 mnemonics */
  if(Mnemo65_Check(HTYPE_65C02MNEMO)) return(FALSE);

  if(CPU_TypeBit == CPUBIT_65C02) return(TRUE);
/* Look out for 65816 mnemonics */
  return(!Mnemo65_Check(HTYPE_65816MNEMO));
}

/*
 * Work routine
 */
int Mnemo65_Check(int Type) {
  treeItem* p;
  byte      Group,
            Value;

  /* search for tree item */
  p = Tree_ScanROM(MnemoString, Type);
  if(p == NULL) return(FALSE);
  Value = p->Body.Opcode.Code;/* get byte value */
  Group = p->Body.Opcode.Group;/* get group */
  switch(Group) {

    case G_IMP:/* mnemonics with only implicit addressing */
    Mnemo65_GroupIMP(Value);
    break;

    case G_RELS:/* short relative */
    Mnemo65_GroupRELS(Value);
    break;

    case G_RELL:/* long relative */
    Mnemo65_GroupRELL(Value);
    break;

    case G_ACCM:/* main accumulator stuff */
    Mnemo65_GroupACC(Value, TRUE);
    break;

    case G_ACCS:/* "sta" */
    Mnemo65_GroupACC(Value, FALSE);
    break;

    case G_MISCO:/* misc, does not depend on register lengths */
    /*FALLTHROUGH*/

    case G_MISCA:/* misc, depends on accu length */
    /*FALLTHROUGH*/

    case G_MISCR:/* misc, depends on index register lengths */
    Mnemo65_GroupMISC(Value, Group);
    break;

    case G_MOVE:/* "mvp" and "mvn" */
    Mnemo65_GroupMOVE(Value);
    break;

    case G_PEI:/* "pei" */
    Mnemo65_GroupPEI(Value);
    break;

    case G_JUMP:/* the jump instructions */
    Mnemo65_GroupJUMP(Value);
    break;

    default:
    BugFound(BIA_UkGI);
  }
  return(TRUE);
}

/*
 * Mnemonics using only implicit addressing.
 */
void Mnemo65_GroupIMP(byte Opcode) {
  Mnemo_PutCodeByte(Opcode);
  EnsureEOL();
}

/*
 * Mnemonics using only 8bit relative addressing (short branch instructions).
 */
void Mnemo65_GroupRELS(byte Opcode) {
  value v = ALU_GetValue_Medium();

  if(ALU_Flags & MVALUE_DEFINED) {
    v -= (CPU_PC + 2);
    if((v > 127) || (v < -128)) ThrowError(Exception_TooFar);
  }
  Mnemo_PutCodeByte(Opcode);
  Output_8b(v);
  EnsureEOL();
}

/*
 * Mnemonics using only 16bit relative addressing (BRL and PER).
 */
void Mnemo65_GroupRELL(byte Opcode) {
  value v = ALU_GetValue_Medium();

  if(ALU_Flags & MVALUE_DEFINED) {
    v -= (CPU_PC + 3);
    if((v > 32767) || (v < -32768)) ThrowError(Exception_TooFar);
  }
  Mnemo_PutCodeByte(Opcode);
  Output_16b(v);
  EnsureEOL();
}

/*
 * The main accumulator stuff (ADC, AND, CMP, EOR, LDA, ORA, SBC, STA).
 * STA has no immediate addressing mode, so ImmFlag is set in that case.
 */
void Mnemo65_GroupACC(byte Base, int ImmFlag) {
  byte  Offset = 0;/* offset, added to base to make opcode */
  int   fb     = Mnemo_GetForceBit();/* skips spaces */
  value v      = Mnemo_GetArgument();

  switch(hAM_Now) {

    case HAM_IMM:/* #$ff or #$ffff (depending on accu length) = offset $09 */
    if(ImmFlag) {
      if(CPU_TypeBit == CPUBIT_65816) {
        /* 65816 knows 16bit immediate addressing */
        if(fb == 0) {/* if no force bits given, use flag */
          if(CPU_fLongA) fb = MVALUE_FORCE16;
          else           fb = MVALUE_FORCE08;
        }
        Mnemo_PutCodeByte(Base + 9);
        switch(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE_1_2__)) {

          case MVALUE_FORCE08:
          Output_8b(v);
          break;

          case MVALUE_FORCE16:
          Output_16b(v);
        }
      } else {
        /* other 65xx only know 8bit immediate addressing */
        if(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE_1____)) {
          Mnemo_PutCodeByte(Base + 9);
          Output_8b(v);
        }
      }
    } else ThrowError(Exception_CombCA);/* kill "STA #..." */
    break;

    case HAM_ABSX:/* $ff,x, $ffff,x $ffffff,x = offsets $15, $1d, $1f */
    Offset = 16;/* adjust offset and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_ABS:/*  $ff,   $ffff,  $ffffff   = offsets $05, $0d, $0f */
    switch(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE_1_2_3)) {

      case MVALUE_FORCE08:
      Mnemo_PutCodeByte(Base + Offset + 5);
      Output_8b(v);
      break;

      case MVALUE_FORCE16:
      Mnemo_PutCodeByte(Base + Offset + 13);
      Output_16b(v);
      break;

      case MVALUE_FORCE24:
      Mnemo_PutCodeByte(Base + Offset + 15);
      Output_24b(v);
    }
    break;

    case HAM_ABSY:/* $ffff,y = offset $19 */
    if(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE___2__)) {
      Mnemo_PutCodeByte(Base + 25);
      Output_16b(v);
    }
    break;

    case HAM_ABSS:/* $ff,s = offset $03 */
    if(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE_1____)) {
      Mnemo_PutCodeByte(Base + 3);
      Output_8b(v);
    }
    break;

    case HAM_XIND:/* ($ff,x) = offset $01 */
    if(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE_1____)) {
      Mnemo_PutCodeByte(Base + 1);
      Output_8b(v);
    }
    break;

    case HAM_IND:/*  ($ff)   = offset $12 */
    Offset = 1;/* adjust offset and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_INDY:/* ($ff),y = offset $11 */
    if(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE_1____)) {
      Mnemo_PutCodeByte(Base + Offset + 17);
      Output_8b(v);
    }
    break;

    case HAM_INDLY:/* [$ff],y = offset $17 */
    Offset = 16;/* adjust offset and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_INDL:/*  [$ff]   = offset $07 */
    if(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE_1____)) {
      Mnemo_PutCodeByte(Base + Offset + 7);
      Output_8b(v);
    }
    break;

    case HAM_SINDY:/* ($ff,s),y = offset $13 */
    if(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE_1____)) {
      Mnemo_PutCodeByte(Base + 19);
      Output_8b(v);
    }
    break;

    default:
    ThrowError(Exception_CombCA);
  }
}

/*
 * Various mnemonics with different addressing modes.
 * MISCO isn't affected by different immediate addressing,
 * MISCA means accumulator-related,
 * MISCR means index-register-related.
 */
void Mnemo65_GroupMISC(byte Row, byte Group) {
  int   Bits,
        Code1,
        Code2,
        Column = 0,
        fb     = Mnemo_GetForceBit();/* read length info (skips spaces) */
  value v      = Mnemo_GetArgument();

  switch(hAM_Now) {

    case HAM_IMP:/* column 0 */
    Mnemo65_OutputMISC(0, Row);/* output code byte */
    break;

    case HAM_IMM:/* #$ff or #$ffff (depending on register length), column 1 */
    if(Mnemo65_OutputMISC(1, Row)) {;/* output code byte */
      /* check CPU and group */
      /* MISCO = don't care, MISCA = accu, MISCR = index registers */
      if((CPU_TypeBit == CPUBIT_65816) && (Group != G_MISCO)) {
        /* 65816 knows 16bit immediate addressing of some commands */
        if(fb == 0) {/* if no force bits given, use relevant switch */
          fb = MVALUE_FORCE08;
          if(Group == G_MISCA) {
            /* group G_MISC_A depends on M flag*/
            if(CPU_fLongA) fb = MVALUE_FORCE16;
          } else {
            /* others depend on R flag */
            if(CPU_fLongR) fb = MVALUE_FORCE16;
          }
        }
        switch(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE_1_2__)) {

          case MVALUE_FORCE08:
          Output_8b(v);
          break;

          case MVALUE_FORCE16:
          Output_16b(v);
        }
      } else {
        /* only 8bit immediate addressing */
        if(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE_1____)) {
          Output_8b(v);
        }
      }
    }
    break;

    case HAM_ABSY:/* $ff,y  or  $ffff,y, column 6(8bit) or 7(16bit) */
    Column += 2;/* adjust column and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_ABSX:/* $ff,x  or  $ffff,x, column 4(8bit) or 5(16bit) */
    Column += 2;/* adjust column and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_ABS: /* $ff    or  $ffff,   column 2(8bit) or 3(16bit) */
    Column += 2;/* adjust column and go on (no break here) */
    Code1 = GroupMISC[Column + (Row << 3)];
    Code2 = GroupMISC[Column + (Row << 3) + 1];
    Bits = MAYBE______;
    if(Code1) Bits |= MAYBE_1____;/*  8 bit */
    if(Code2) Bits |= MAYBE___2__;/* 16 bit */
    switch(Mnemo_DoChecks(fb, v, ALU_Flags, Bits)) {

      case MVALUE_FORCE08:
      Mnemo_PutCodeByte(Code1);
      Output_8b(v);
      break;

      case MVALUE_FORCE16:
      Mnemo_PutCodeByte(Code2);
      Output_16b(v);
    }
    break;

    default:
    ThrowError(Exception_CombCA);
  }
}

/*
 * Utility routine for groups MISCO, MISCA and MISCR
 */
int Mnemo65_OutputMISC(int x, int y) {
  byte Opcode = GroupMISC[x + (y << 3)];

  if(Opcode) {
    Mnemo_PutCodeByte(Opcode);
    return(TRUE);
  }
  ThrowError(Exception_CombCA);
  return(FALSE);
}

/*
 * Mnemonics using "8bit, 8bit" addressing. Only applies to "MVN" and "MVP".
 */
void Mnemo65_GroupMOVE(byte Opcode) {
  value Target,
        Source = ALU_GetValue_Medium();

  if(Stream_Comma()) {
    Target = ALU_GetValue_Medium();
    Mnemo_PutCodeByte(Opcode);
    Output_8b(Target);/* assembler syntax:    "opcode, source, target" */
    Output_8b(Source);/* in machine language: "opcode, target, source" */
    EnsureEOL();
  } else ThrowError(Exception_Syntax);
}

/*
 * Mnemonics using only the "($..)" addressing mode. Only applies to PEI.
 */
void Mnemo65_GroupPEI(byte Opcode) {
  value v = Mnemo_GetArgument();

  switch(hAM_Now) {

    case HAM_IND:
    Mnemo_PutCodeByte(Opcode);
    Output_8b(v);
    break;

    default:
    ThrowError(Exception_CombCA);
  }
}

/*
 * The jump instructions.
 */
void Mnemo65_GroupJUMP(byte Column) {
  byte  Code2,
        Code3;
  int   Bits,
        Offset = 0,
        fb     = Mnemo_GetForceBit();/* read length info (skips spaces) */
  value v      = Mnemo_GetArgument();

  switch(hAM_Now) {

    case HAM_ABS:/* $ffff  or  $ffffff, row 0(16bit) or 3(24bit) */
    Code2 = GroupJUMP[Column];
    Code3 = GroupJUMP[Column + 12];
    Bits = MAYBE______;
    if(Code2) Bits  = MAYBE___2__;/* 16 bit */
    if(Code3) Bits |= MAYBE_____3;/* 24 bit */
    switch(Mnemo_DoChecks(fb, v, ALU_Flags, Bits)) {

      case MVALUE_FORCE16:
      Mnemo_PutCodeByte(Code2);
      Output_16b(v);
      break;

      case MVALUE_FORCE24:
      Mnemo_PutCodeByte(Code3);
      Output_24b(v);
    }
    break;

    case HAM_INDL:/* [$ffff],   row 4 */
    Offset += 8;/* adjust offset and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_XIND:/* ($ffff,x), row 2 */
    Offset += 4;/* adjust offset and go on (no break here) */
    /*FALLTHROUGH*/

    case HAM_IND: /* ($ffff),   row 1 */
    Offset += 4;/* adjust offset and go on (no break here) */
    if(Mnemo_DoChecks(fb, v, ALU_Flags, MAYBE___2__)) {
      Code2 = GroupJUMP[Column + Offset];
      if(Code2) {
        Mnemo_PutCodeByte(Code2);
        Output_16b(v);
      } else ThrowError(Exception_CombCA);
    }
    break;

    default:ThrowError(Exception_CombCA);
  }
}

/*
 * Service routine to read optional info about parameter length.
 */
int Mnemo_GetForceBit() {
  byte b;
  int  fb = 0;

  if(GotByte == '+') {
    b = GetByte();
    if(b == '1') fb = MVALUE_FORCE08;
    if(b == '2') fb = MVALUE_FORCE16;
    if(b == '3') fb = MVALUE_FORCE24;
    if(fb) GetByte();
    else   ThrowError(Exception_UkPF);
  }
  SKIPSPACE;
  return(fb);
}

/*
 * Address mode parsing
 */

/*
 * This routine returns the command's argument (using the valueparser). The
 * addressing mode will be stored in hAM_Now.
 */
value Mnemo_GetArgument() {
  value  v;

  hAM_Now = HAM_ABS;
  SKIPSPACE;
  switch(GotByte) {

    case '[':
    GetByte();/* proceed with next char */
    v = ALU_GetValue_Medium();
    if(GotByte == ']') hAM_Now |= HAM_INDL | Mnemo_GetIndex(TRUE);
    else               ThrowError(Exception_Syntax);
    break;

    case '#':
    GetByte();/* proceed with next char */
    hAM_Now |= HAM_IMM;
    v = ALU_GetValue_Medium();
    break;

    default:
    v = ALU_GetValue_Liberal();/* liberal, to allow for "(...," */
    /* check for implicit addressing */
    if((ALU_Flags & MVALUE_EXISTS) == 0) hAM_Now |= HAM_IMP;
    /* check for indirect addressing */
    if(ALU_Flags & MVALUE_INDIRECT) hAM_Now |= HAM_IND;
    /* check for internal index (before closing parenthesis) */
    if(ALU_LeftParentheses) {
      /* in case there are still open parentheses, read internal index */
      hAM_Now |= (Mnemo_GetIndex(FALSE) << 2);
      if(GotByte == ')') GetByte();/* proceed with next char */
      else ThrowError(Exception_Syntax);
    }
    /* check for external index (after closing parenthesis) */
    hAM_Now |= Mnemo_GetIndex(FALSE);
    break;

  }
  /* ensure end of line */
  EnsureEOL();
  /*printf("AM: %x\n", hAM_Now);*/
  return(v);
}

/*
 * Utility routine for parsing indices.
 */
int Mnemo_GetIndex(int Next) {
  int am = HAM__;

  if(Next) GetByte();
  if(!Stream_Comma()) return(am);
  switch(GotByte) {

    case 's':
    case 'S':
    am = HAM_S;
    break;

    case 'x':
    case 'X':
    am = HAM_X;
    break;

    case 'y':
    case 'Y':
    am = HAM_Y;
    break;

    default:
    ThrowError(Exception_Syntax);
  }
  if(am != HAM__) NEXTANDSKIPSPACE;
  return(am);
}

/*
 * Utility routine for comparing force bits, argument value, argument size,
 * "unsure"-flag and possible addressing modes. Returns force bit matching
 * number of parameter bytes to send. If it returns zero, an error occurred
 * (and has already been delivered).
 */
int Mnemo_DoChecks(int fb, value v, int ff, int am) {
  int Result = 0;

/*
 *      fb      Force bit (none, 8b, 16b, 24b)
 *      v       Value of parameter
 *      ff      Flags of value (ALU_Flags)
 *      am      Adressing modes (8b, 16b, 24b or any combination)
 *      Result  Return value, number of parameter bytes to send (0 = error)
 */
  if(am != MAYBE______) {
    if(fb) {
/* Command has force bit */
      if(am & fb) Result = fb;/* Command allows this force bit */
      else        ThrowError(Exception_CombCP);/* ...or doesn't */
    } else {
/* Command has no force bit => using value's force bit */
      if(ff & MVALUE_FORCEBITS) {
  /* Value has force bit set, so return this or bigger size */
        if(MVALUE_FORCE08 & am & ff) Result = MVALUE_FORCE08;
        else {
          if(MVALUE_FORCE16 & am & ff) Result = MVALUE_FORCE16;
          else {
            if(MVALUE_FORCE24 & am) Result = MVALUE_FORCE24;
            else ThrowError(Exception_TooBig);/* else error */
          }
        }
      } else {
  /* Value has no force bit => see if there's only one addressing mode */
        switch(am) {

    /* There's only one addressing mode, so use it */
          case MVALUE_FORCE08:
          case MVALUE_FORCE16:
          case MVALUE_FORCE24:
          Result = am;
          break;

    /* There's more than one, so check whether value is sure */
          default:
          if(ff & MVALUE_UNSURE) {
      /* Value is unsure, so use default size */
            Result = MVALUE_FORCE08;
            if(((MVALUE_FORCE08 & am)==0) || ((ff & MVALUE_ISBYTE)==0)) {
              if(MVALUE_FORCE24 & am) Result = MVALUE_FORCE24;
              if(MVALUE_FORCE16 & am) Result = MVALUE_FORCE16;
              if(ff & MVALUE_DEFINED) {
                if(Result == MVALUE_FORCE16) {
                  if((v <= 0xff) && (v >= -0x80)) {
                    ThrowWarning(Exception_HighbyteZero);
                  }
                }
                if(Result == MVALUE_FORCE24) {
                  if((v <= 0xffff) && (v >= -0x8000)) {
                    ThrowWarning(Exception_HighbyteZero);
                  }
                }
              }
            }
          } else {
      /* Value is sure, so use its own size */
            if(v >= 0) {
              if((MVALUE_FORCE08 & am) && (v < 256)) {
                Result = MVALUE_FORCE08;
              } else {
                if((MVALUE_FORCE16 & am) && (v < 65536)) {
                  Result = MVALUE_FORCE16;
                } else {
                  if(MVALUE_FORCE24 & am) {
                    Result = MVALUE_FORCE24;
                  } else ThrowError(Exception_TooBig);
                }
              }
            } else ThrowError(Exception_Negative);
          }
        }
      }
    }
  } else ThrowError(Exception_CombCA);
  return(Result);
}

/*
 * Z80 stuff
 */

int Mnemo80_IsInvalid(int len) {
  ThrowError(Exception_NotYet);
  SkipRest();
  return(TRUE);
}
