/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Output stuff
 */

#include "output.h"

/*
 * Constants
 */

char Exception_TooMuch[] = "Produced too much code.";
char Exception_SegReached[] = "Segment reached another one, overwriting it.";
char Exception_CannotOpenOutFile[] = "Cannot open output file.";
char psvOutput[] = "Saving %d ($%x) bytes to output file ($%x - $%x exclusive).\n";

/*
 * Variables
 */

byte  OutputBuffer[OUTBUFFERSIZE];/* to hold assembled code */
int   hOutformat = OUTFORMAT_NONE;/* Output file format */
byte  pnfOut[LNFMAX + 1];/* Filename of output file */

/*
 * Output 8-bit value with range check
 */
void Output_8b(value v) {
  if((v <= 0xff) && (v >= -0x80)) {
    Output_Byte(v);
  } else ThrowError(Exception_TooBig);
}

/*
 * Output 16-bit values with range check
 */
void Output_16b(value v) {
  if((v <= 0xffff) && (v >= -0x8000)) {
    Output_Byte(v);
    Output_Byte(v >> 8);
  } else ThrowError(Exception_TooBig);
}

/*
 * Output 24-bit values with range check
 */
void Output_24b(value v) {
  if((v <= 0xffffff) && (v >= -0x800000)) {
    Output_Byte(v);
    Output_Byte(v >> 8);
    Output_Byte(v >> 16);
  } else ThrowError(Exception_TooBig);
}

/*
 * Output 32-bit values (without range check)
 */
void Output_32b(value v) {
/*  if((v <= 0x7fffffff) && (v >= -0x80000000)) {*/
    Output_Byte(v);
    Output_Byte(v >> 8);
    Output_Byte(v >> 16);
    Output_Byte(v >> 24);
/*  } else ThrowError(Exception_TooBig);*/
}

/*
 * Send low byte to output file, automatically increasing the program counter
 */
void Output_Byte(value Byte) {
  int Offset;

  if(CPU_fPCdefined == FALSE) {
    ThrowError(Exception_NoPC);
    CPU_fPCdefined = TRUE;
  }
  Offset = PC_Mem + PC_inc;
  if(Offset == SegmentMax+1) {
    if(Offset == OUTBUFFERSIZE) {
      ThrowSerious(Exception_TooMuch);
    } else {
      ThrowWarning(Exception_SegReached);
      if(Pass_Flags & PASS_ISFIRST) Segment_FindMax(Offset+1);
    }
  }
  OutputBuffer[Offset] = Byte & 255;
  PC_inc++;
}

/*
 * Dump code from buffer into output file (add block support? !!!)
 */
void Output_ToFile(value Start, value End) {
  FILE* Filehandle;
  value a;

  /* if no output file chosen, do nothing */
  if(hOutformat == OUTFORMAT_NONE) return;

  if(Process_Verbosity) printf(psvOutput, End-Start, End-Start, Start, End);
  /* Try to open output file */
  Filehandle = File_OpenFile(pnfOut, FILE_WRITEBINARY);

  if(Filehandle) {
    switch(hOutformat) {

      case OUTFORMAT_CBM:
      PLATFORM_SETFILETYPE(pnfOut, FILETYPE_CBM);
      fputc(Start & 255, Filehandle);/* load address, low  byte */
      fputc(Start >>  8, Filehandle);/* load address, high byte */
      break;

      case OUTFORMAT_PLAIN:
      PLATFORM_SETFILETYPE(pnfOut, FILETYPE_PLAIN);
      break;

      case OUTFORMAT_O65:
      /* !!! */
      break;

    }
    for(a = Start; a < End; a++) {
      fputc(OutputBuffer[a], Filehandle);
    }
    File_CloseFile(Filehandle);/* close output file */
  } else ThrowSerious(Exception_CannotOpenOutFile);
}

/*
 * Fill output buffer with given byte value
 */
void Output_InitMem(byte c) {
  int a;

  for(a = OUTBUFFERSIZE-1; a >= 0; a--) OutputBuffer[a] = c;
}
