/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Tree stuff
 */

#ifndef tree_H
#define tree_H

/*
 * Constants
 */

/* ROM tree item types */
enum {            /* Tree item type handles */
 HTYPE_PSEUDO    ,/* Pseudo opcodes */
 HTYPE_OPERATOR  ,/* String versions of arithmetic operators */
 HTYPE_CPU       ,/* Processor names */
 HTYPE_UNTILWHILE,/* Keywords "until" and "while" */
 HTYPE_ELSE      ,/* Keyword "else" */
 HTYPE_CONV      ,/* Keywords "raw", "pet", "scr", "file" */
 HTYPE_6502MNEMO ,/* Mnemonics of standard 6502 instruction set */
 HTYPE_65C02MNEMO,/* Mnemonics of 65c02 extensions to instruction set */
 HTYPE_65816MNEMO,/* Mnemonics of 65816 extensions to instruction set */
 HTYPE_6510MNEMO ,/* Mnemonics of 6510's illegal instructions */
 HTYPE_Z80       ,/* Mnemonics of the z80 instruction set */
 HTYPE_OUTFORMAT  /* Output file formats (hint) */
};

/* the "treeItem" definition */
struct treeItem {
  treeItem*     GreaterThan;/* pointer to sub-tree */
  treeItem*     LessThanOrEqual;/* pointer to sub-tree */
  u_int         IntHash;/* big (integer) hash value */
  u_int         ID_Number;/* item type [and zone number] */
  byte*         ID_String;/* name, zero-terminated */
  union {
    struct {
      byte          Code;/* opcode value, operator handle */
      byte          Group;/* opcode group, operator priority */
    } Opcode;
    void*         Pointer;/* macro/label body, PO handler function */
  } Body;
};

/* Official ANSI-C does not allow to predefine any union elements but the
 * first. Because of that, here the structure is defined once again (just with
 * a different ordering of union elements). :( */
typedef struct treeItemP treeItemP;
struct treeItemP {
  treeItem*     GreaterThan;/* pointer to sub-tree */
  treeItem*     LessThanOrEqual;/* pointer to sub-tree */
  u_int         IntHash;/* big (integer) hash value */
  u_int         ID_Number;/* item type [and zone number] */
  byte*         ID_String;/* name, zero-terminated */
  union {
    void*         Pointer;/* macro/label body, PO handler function */
    struct {
      byte          Code;/* opcode data */
      byte          Group;/* opcode data */
    } Opcode;
  } Body;
};

/* This is for allocating space for a tree item and its name string */
typedef struct namedItem namedItem;
struct namedItem {
  treeItem Item;
  byte     Name[1];/* place for terminator */
};

/*
 * Variables
 */

extern treeItem* TreesROM[];/* Array of pointers to start of item */
extern treeItem* TreesRAM[];/*      trees (because of 8-bit hash) */
extern treeItem  PredefinedB[];/* predefined stuff using byte values */
extern treeItemP PredefinedP[];/* predefined stuff using pointers */
extern int       Tree_ItemCreated;/* Global flag for tree item lookup */

/*
 * Prototypes
 */

void      Tree_InitROM(void);
void      Tree_ClearPointerTable(treeItem* []);
treeItem* Tree_ScanROM(byte*, int);
treeItem* Tree_ScanRAM(int, int);
void      Tree_Dump256(treeItem**, int, void (*)(treeItem*));
void      Tree_DumpTree(treeItem*, int, void (*)(treeItem*));

#endif
