/*
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *  Copyright 1999 Michael Klein <michael.klein@puffin.lb.shuttle.de>
*/

#include "cbm4linux.h"
#include "d64copy.h"

#include <errno.h>
#include <error.h>
#include <fcntl.h>
#include <getopt.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

char sector_map[EXT_TRACKS+1] =
{ 0,
  21, 21, 21, 21, 21, 21, 21, 21, 21, 21,
  21, 21, 21, 21, 21, 21, 21, 19, 19, 19,
  19, 19, 19, 19, 18, 18, 18, 18, 18, 18,
  17, 17, 17, 17, 17,
  17, 17, 17, 17, 17
};

int fd_cbm;

static unsigned char warp_read_1541[] = {
#include "warpread1541.inc"
};

static unsigned char warp_write_1541[] = {
#include "warpwrite1541.inc"
};

static unsigned char warp_read_1571[] = {
#include "warpread1571.inc"
};

static unsigned char warp_write_1571[] = {
#include "warpwrite1571.inc"
};

static unsigned char turbo_read_1541[] = {
#include "turboread1541.inc"
};

static unsigned char turbo_write_1541[] = {
#include "turbowrite1541.inc"
};

static unsigned char turbo_read_1571[] = {
#include "turboread1571.inc"
};

static unsigned char turbo_write_1571[] = {
#include "turbowrite1571.inc"
};

static unsigned char drv_detect[] = {
#include "detect.inc"
};

static struct drive_prog {
    int size;
    char *prog;
} drive_progs[] = {
    {sizeof(turbo_read_1541), turbo_read_1541},
    {sizeof(turbo_write_1541), turbo_write_1541},
    {sizeof(warp_read_1541), warp_read_1541},
    {sizeof(warp_write_1541), warp_write_1541},
    {sizeof(turbo_read_1571), turbo_read_1571},
    {sizeof(turbo_write_1571), turbo_write_1571},
    {sizeof(warp_read_1571), warp_read_1571},
    {sizeof(warp_write_1571), warp_write_1571}
};

static int send_turbo(int fd, int drv, int write, int warp, int drv_type)
{
    struct drive_prog *prog;

    prog = &drive_progs[drv_type * 4 + warp * 2 + write];

    return cbm_upload(fd, drv, 0x500, prog->prog, prog->size);
}

static int is_cbm(char *name)
{
    return((strcmp(name, "8" ) == 0) || (strcmp(name, "9" ) == 0) ||
           (strcmp(name, "10") == 0) || (strcmp(name, "11") == 0) );
}

#define TM_FILESYSTEM 0
#define TM_ORIGINAL   1
#define TM_PARALLEL   2
#define TM_SERIAL1    3
#define TM_SERIAL2    4

#define BM_IGNORE     0
#define BM_ALLOCATED  1
#define BM_SAVE       2

extern struct transfer_funcs fs_transfer,
                             std_transfer,
                             pp_transfer,
                             s1_transfer,
                             s2_transfer;

static void help()
{
    printf(
"Usage: d64copy [OPTION]... [SOURCE] [TARGET]\n\
Copy .d64 disk images to a CBM-1541 or compatible drive and vice versa\n\
\n\
  -h                display this help and exit\n\
  -q                more quiet output\n\
\n\
  -e TRACK          set start track\n\
  -s TRACK          set end track (start <= end <= 40)\n\
\n\
  -t TRANSFERMODE   set transfermode; valid modes are:\n\
                      original   (slowest)\n\
                      serial1\n\
                      serial2\n\
                      parallel   (fastest)\n\
                    `original' and `serial1' should work in any case;\n\
                    `serial2' won't work if more than one device is\n\
                    connected to the IEC bus;\n\
                    `parallel' needs a XP1541 cable\n\
\n\
  -i INTERLEAVE     set interleave value; ignored when reading with warp\n\
                    mode; default values are:\n\
\n\
                      original     16
\n\
                                turbo r/w   warp write\n\
\n\
                      serial1       4            6\n\
                      serial2      13           12\n\
                      parallel      7            4\n\
\n\
                    INTERLEAVE is ignored when reading with warp mode;\n\
                    if data transfer is very slow, increasing this value\n\
                    may help.\n\
\n\
  -w                enable warp mode; this is not possible when\n\
                    TRANSFERMODE is set to `original'\n\
\n\
  -b                BAM-only copy; only allocated blocks are copied; for\n\
                    extended tracks (36-40), SpeedDOS BAM format is assumed.\n\
                    Use with caution.\n\
\n\
  -B                Save BAM-only copy; this is like the `-b option but\n\
                    copies always the entire directory track.\n\
\n\
  -d TYPE           specify drive type:\n\
                      0 = 1541\n\
                      1 = 1570/1571\n\
\n\
");
}

static void hint(char *s)
{
    fprintf(stderr, "Try `%s' -h for more information.\n", s);
}

static void reset(int dummy)
{
    fprintf(stderr, "\nSIGINT caught X-(  Resetting IEC bus...\n");
    sleep(1);
    cbm_reset(fd_cbm);
    close(fd_cbm);
    exit(1);
}


/* setable via command line */
int drive_type;

static int quiet       = 0;
static int warp        = 0;
static int bam_mode    = BM_IGNORE;
static int interleave  = -1;
static int start_track = 1;
static int end_track   = STD_TRACKS;


static int go_for_it(struct transfer_funcs *src, struct transfer_funcs *dst)
{
    int tr = 0;
    int se = 0;
    int st;
    int cnt  = 0;
    int scnt = 0;
    int bamofs;
    char trackmap[MAX_SECTORS+1];
    unsigned char bam[BLOCKSIZE];
    unsigned char block[BLOCKSIZE];
    unsigned char gcr[GCRBUFSIZE];


    if(bam_mode != BM_IGNORE) {
        if(warp && src->is_cbm_drive) {
            memset(trackmap, BS_DONT_COPY, sector_map[18]);
            trackmap[0] = BS_MUST_COPY;
            scnt = 1;
            src->send_track_map(18, trackmap, scnt);
            st = src->read_gcr_block(&se, gcr);
            if(st == 0) st = gcr_decode(gcr, bam);
        } else {
            st = src->read_block(18, 0, bam);
        }
        if(st) {
            fprintf(stderr, "failed to read BAM (%d)\n", st);
            bam_mode = BM_IGNORE;
        }
    }

    fprintf(stderr, "copying tracks %d-%d...", start_track, end_track);
    for(tr = start_track; tr <= end_track; tr++) {
        scnt = sector_map[tr];
        memset(trackmap, BS_MUST_COPY, scnt);
        trackmap[scnt] = '\0';
        if(bam_mode != BM_IGNORE && (bam_mode == BM_ALLOCATED || tr != 18)) {
            for(se = scnt-1; se >= 0; se--) {
                bamofs = 4*tr + (tr > STD_TRACKS ? 48 : 0);
                if(bam[bamofs+1+(se/8)]&(1<<(se&0x07))) {
                    scnt--;
                    trackmap[se] = BS_DONT_COPY;
                }
            }
        }
        if(!quiet) {
            printf("\ntrack %02d: %s", tr, trackmap);
            fflush(stdout);
        }
        if(scnt && warp && src->is_cbm_drive) {
            src->send_track_map(tr, trackmap, scnt);
        } else {
            se = 0;
        }
        while (scnt) {
            if(warp && src->is_cbm_drive) {
                st = src->read_gcr_block(&se, gcr);
                trackmap[se] = BS_COPIED;
                if(st == 0) {
                    st = gcr_decode(gcr, block);
                } else {
                    src->send_track_map(tr, trackmap, scnt);
                }
            } else {
                while(trackmap[se] != BS_MUST_COPY) {
                    if(++se >= sector_map[tr]) se = 0;
                }
                st = src->read_block(tr, se, block);
                trackmap[se] = BS_COPIED;
            }
            if(st) {
                fprintf(stderr, "read error: %02x/%02x: %d\n", tr, se, st);
            }
            if(warp && dst->is_cbm_drive) {
                gcr_encode(block, gcr);
                st = dst->write_block(tr, se, gcr, GCRBUFSIZE-1);
            } else {
                st = dst->write_block(tr, se, block, BLOCKSIZE);
            }
            if(st) {
                fprintf(stderr, "write error: %02x/%02x: %d\n", tr, se, st);
            }
            cnt++;
            scnt--;
            if(!quiet) {
                printf("\rtrack %02d: %s", tr, trackmap);
                fflush(stdout);
            }
            if(dst->is_cbm_drive || !warp) {
                se += interleave;
                if(se >= sector_map[tr]) se -= sector_map[tr];
            }
        }
    }
    return cnt;
}

int start_turbo(int drive)
{   
    return cbm_exec_command(fd_cbm, drive, "U4:", 3);
}

int main(int argc, char *argv[])
{
    static int default_interleave[] = { 0, 17, 7, 4, 13 };
    static int warp_write_interleave[] = { 0, 0, 4, 6, 12 };

    static struct transfer_funcs *transfer[] = { &fs_transfer,
                                                 &std_transfer,
                                                 &pp_transfer,
                                                 &s1_transfer,
                                                 &s2_transfer};

    int transfer_mode = TM_ORIGINAL;

    char *tm = NULL;
    char *src_arg;
    char *dst_arg;

    struct transfer_funcs *src = NULL;
    struct transfer_funcs *dst = NULL;

    char c;
    char buf[40];
    int  rv = 1;
    int  cbm_drive = 0;

    drive_type = -1; /* auto detect */

    while((c=getopt(argc, argv, "hwqbBt:i:s:e:d:")) != -1) {
        switch(c) {
            case 'h': help();
                      return 0;
            case 'w': warp = 1;
                      break;
            case 'q': quiet = 1;
                      break;
            case 'i': interleave = atoi(optarg);
                      break;
            case 's': start_track = atoi(optarg);
                      break;
            case 'e': end_track = atoi(optarg);
                      break;
            case 't': tm = optarg;
                      break;
            case 'b': bam_mode = BM_ALLOCATED;
                      break;
            case 'B': bam_mode = BM_SAVE;
                      break;
            case 'd': if(strcmp(optarg, "1541") == 0) {
                          drive_type = 0;
                      } else if(strcmp(optarg, "1571") == 0) {
                          drive_type = 1;
                      } else {
                          drive_type = atoi(optarg) != 0;
                      }
                      break;
            default : hint(argv[0]);
                      return 1;
        }
    }

    if(tm) {
        if(strcmp(tm, "original") == 0) {
            transfer_mode = TM_ORIGINAL;
        } else if(strcmp(tm, "parallel") == 0) {
            transfer_mode = TM_PARALLEL;
        } else if(strcmp(tm, "serial1") == 0) {
            transfer_mode = TM_SERIAL1;
        } else if(strcmp(tm, "serial2") == 0) {
            transfer_mode = TM_SERIAL2;
        } else {
            fprintf(stderr, "unknown transfer mode: `%s'\n", tm);
            return 1;
        }
    }

    if(interleave != -1 && (interleave < 1 || interleave > 17)) {
        fprintf(stderr, "invalid value (%d) for interleave\n", interleave);
        return 1;
    }

    if(start_track < 1 || start_track > EXT_TRACKS) {
        fprintf(stderr, "invalid value (%d) for start track\n", start_track);
        return 1;
    }

    if(end_track < start_track || end_track > EXT_TRACKS) {
        fprintf(stderr, "invalid value (%d) for end track\n", end_track);
        return 1;
    }

    if(optind + 2 == argc) {
        src_arg = argv[optind];
        dst_arg = argv[optind+1];

        src = transfer[is_cbm(src_arg) ? transfer_mode : TM_FILESYSTEM];
        dst = transfer[is_cbm(dst_arg) ? transfer_mode : TM_FILESYSTEM];

        if(src->is_cbm_drive == dst->is_cbm_drive) {
            fprintf(stderr, "either source or target must be a CBM drive\n");
            return 1;
        }

        cbm_drive = atoi(src->is_cbm_drive ? src_arg : dst_arg);

        if(interleave == -1) {
            interleave = (dst->is_cbm_drive && warp) ?
                warp_write_interleave[transfer_mode] :
                default_interleave[transfer_mode];
        }

        if((fd_cbm = open(cbm_dev, O_RDWR)) != -1) {

            signal(SIGINT, reset);

            if(drive_type < 0) {
                cbm_upload(fd_cbm, cbm_drive, 0x500, drv_detect, 
                        sizeof(drv_detect));
                cbm_exec_command(fd_cbm, cbm_drive, "U3:", 3);
                cbm_exec_command(fd_cbm, cbm_drive, "M-R\002\005\000\001", 7);
                cbm_device_status(fd_cbm, cbm_drive, buf, 2);
                drive_type = buf[0];
            }

            cbm_exec_command(fd_cbm, cbm_drive, "I0:", 0);
            rv = cbm_device_status(fd_cbm, cbm_drive, buf, sizeof(buf));

            fprintf(stderr, "drive %02d (%s): %s\n",
                    cbm_drive,
                    drive_type ? "1571" : "1541",
                    buf);

            if(rv) {
                close(fd_cbm);
                return 1;
            }

            if(warp && (transfer[transfer_mode]->read_gcr_block == NULL)) {
                fprintf(stderr, "warning: `-w' ignored\n");
                warp = 0;
            }
            if(transfer[transfer_mode]->needs_turbo) {
                send_turbo(fd_cbm, cbm_drive, dst->is_cbm_drive, warp, drive_type);
            }

            if(src->open_disk(src_arg, 0, end_track > STD_TRACKS, start_turbo) == 0) {
                if(dst->open_disk(dst_arg, 1, end_track > STD_TRACKS, start_turbo) == 0) {

                    printf("\n%d blocks copied.\n", go_for_it(src, dst));

                    dst->close_disk();
                } else {
                    fprintf(stderr, "can't open destination\n");
                }
                src->close_disk();
            } else {
                fprintf(stderr, "can't open source\n");
            }
            close(fd_cbm);
        } else {
            error(0, errno, "%s", cbm_dev);
        }
        return rv;
    } else {
        fprintf(stderr, "Usage: %s [OPTION]... [SOURCE] [TARGET]\n", argv[0]);
        hint(argv[0]);
        return 1;
    }
}
