/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version
 *  2 of the License, or (at your option) any later version.
 *
 *  Copyright 1999-2001 Michael Klein <michael.klein@puffin.lb.shuttle.de>
*/

#include <linux/config.h>
#include <linux/version.h>

#include <linux/module.h>
#ifdef CONFIG_MODVERSIONS
#include <linux/modversions.h>
#endif

#include <asm/io.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/sched.h>

#ifdef KERNEL_VERSION                   /* kernel > 2.0.x */
#include <asm/uaccess.h>
#endif

#include "cbm_module.h"

unsigned int port        = 0x378;       /* lpt port address    */
unsigned int irq         = 7;           /* lpt irq line        */
         int cable       = -1;          /* <0 => autodetect    */
                                        /* =0 => non-inverted (XM1541) */
                                        /* >0 => inverted     (XA1541) */

#ifdef KERNEL_VERSION
MODULE_PARM(port,"i");
MODULE_PARM(irq,"i");
MODULE_PARM(cable,"i");

MODULE_AUTHOR("Michael Klein");
MODULE_DESCRIPTION("Serial CBM bus driver module");
#endif

#define NAME      "cbm"
#define CBM_MINOR 177

#define IEC_DATA   1
#define IEC_CLOCK  2
#define IEC_ATN    4

/* lpt output lines */
#define ATN_OUT    0x01
#define CLK_OUT    0x02
#ifndef OLD_C4L_CABLE
  #define DATA_OUT 0x04
  #define RESET    0x08
#else
  #define DATA_OUT 0x08
  #define RESET    0x04
#endif
#define LP_IRQ     0x10
#define LP_BIDIR   0x20

/* lpt input lines */
#define ATN_IN     0x10
#define CLK_IN     0x20
#define DATA_IN    0x40


#define cbm_init    init_module
#define cbm_cleanup cleanup_module

static unsigned char out_bits, out_eor;
static int in_port;
static int out_port;
static int busy;

#define SET(line)       (outb(out_eor^(out_bits|=line),out_port))
#define RELEASE(line)   (outb(out_eor^(out_bits&=~(line)),out_port))
#define GET(line)       ((inb(in_port)&line)==0?1:0)

#ifdef DEBUG
  #define DPRINTK(fmt,args...)     printk(fmt, ## args)
  #define SHOW(str)                show(str)
#else
  #define DPRINTK(fmt,args...)
  #define SHOW(str)
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0))
static struct wait_queue *cbm_wait_q;
#else
static wait_queue_head_t cbm_wait_q;
#endif
volatile static int eoi;
volatile static int irq_count;

#ifndef KERNEL_VERSION
#define signal_pending(p) (p->signal & ~p->blocked)
#endif

/*
 *  dump input lines
 */
static void show( char *s )
{
        printk("%s: data=%d, clk=%d, atn=%d\n", s,
                        GET(DATA_IN), GET(CLK_IN), GET(ATN_IN));
}

static void do_reset( void )
{
        printk("cbm: resetting devices\n");
        RELEASE(DATA_OUT | ATN_OUT | CLK_OUT | LP_BIDIR | LP_IRQ);
        SET(RESET);
        current->state = TASK_INTERRUPTIBLE;
#ifdef KERNEL_VERSION
        schedule_timeout(HZ/10); /* 100ms */
#else
        current->timeout = jiffies + 10;
        schedule();
#endif
        RELEASE(RESET);

        printk("cbm: sleeping 5 seconds...\n");
        current->state = TASK_INTERRUPTIBLE;
#ifdef KERNEL_VERSION
        schedule_timeout(HZ*5); /* 5s */
#else
        current->timeout = jiffies + 500;
        schedule();
#endif
        SET(CLK_OUT);
}

/*
 *  send byte
 */
static int send_byte(int b)
{
        int i, ack = 0;
        unsigned long flags;

        DPRINTK("send_byte %02x\n", b);

        save_flags(flags); cli();
        for( i = 0; i < 8; i++ ) {
                udelay(70);
                if( !((b>>i) & 1) ) {
                        SET(DATA_OUT);
                }
                RELEASE(CLK_OUT);
                udelay(20);
                SET(CLK_OUT);
                RELEASE(DATA_OUT);
        }
        restore_flags(flags);

        for( i = 0; (i < 20) && !(ack=GET(DATA_IN)); i++ ) {
                udelay(100);
        }

        DPRINTK("ack=%d\n", ack);

        return ack;
}

/*
 *  wait until listener is ready to receive
 */
static void wait_for_listener()
{
#ifdef DECLARE_WAITQUEUE
        DECLARE_WAITQUEUE(wait, current);
#else
        struct wait_queue wait = { current, NULL };
#endif

        SET(LP_IRQ);
        add_wait_queue(&cbm_wait_q, &wait);
        current->state = TASK_INTERRUPTIBLE;
        RELEASE(CLK_OUT);
        while(irq_count && !signal_pending(current)) {
            schedule();
        }
        remove_wait_queue(&cbm_wait_q, &wait);
        RELEASE(LP_IRQ);
}

/*
 *  idle
 */
static void release_all(void)
{
        RELEASE(ATN_OUT | DATA_OUT);
}

#ifdef KERNEL_VERSION
static int cbm_read(struct file *f, char *buf, size_t count, loff_t *ppos)
#else
static int cbm_read(struct inode *inode, struct file *f, char *buf, int count)
#endif
{
        int received = 0;
        int i, b, bit;
        int ok = 0;
        unsigned long flags;

        DPRINTK("cbm_read: %d bytes\n", count);

        if(eoi) {
                return 0;
        }

        do {
                i = 0;
                while(GET(CLK_IN)) {
                        if( i >= 50 ) {
                                current->state = TASK_INTERRUPTIBLE;
#ifdef KERNEL_VERSION
                                schedule_timeout(HZ/50);
#else
                                current->timeout = jiffies+4;
                                schedule();
#endif
                                if(signal_pending(current)) {
                                        return -EINTR;
                                }
                        } else {
                                i++;
                                udelay(20);
                        }
                }
                save_flags(flags); cli();
                RELEASE(DATA_OUT);
                for(i = 0; (i < 40) && !(ok=GET(CLK_IN)); i++) {
                        udelay(10);
                }
                if(!ok) {
                        /* device signals eoi */
                        eoi = 1;
                        SET(DATA_OUT);
                        udelay(70);
                        RELEASE(DATA_OUT);
                }
                for(i = 0; i < 100 && !(ok=GET(CLK_IN)); i++) {
                        udelay(20);
                }
                for(bit = b = 0; (bit < 8) && ok; bit++) {
                        for(i = 0; (i < 200) && !(ok=(GET(CLK_IN)==0)); i++) {
                                udelay(10);
                        }
                        if(ok) {
                                b >>= 1;
                                if(GET(DATA_IN)==0) {
                                        b |= 0x80;
                                }
                                for(i = 0; i < 100 && !(ok=GET(CLK_IN)); i++) {
                                        udelay(20);
                                }
                        }
                }
                if(ok) {
                        SET(DATA_OUT);
                }
                restore_flags(flags);
                if(ok) {
                        received++;
                        put_user((char)b, buf++);

                        if(count % 256) {
                            udelay(50);
                        } else {
                            schedule();
                        }
                }

        } while(received < count && ok && !eoi);

        if(!ok) {
                printk("cbm_read: I/O error\n");
                return -EIO;
        }

        DPRINTK("received=%d, count=%d, ok=%d, eoi=%d\n",
                        received, count, ok, eoi);

        return received;
}

static int cbm_raw_write(const char *buf, size_t cnt, int atn, int talk)
{
        unsigned char c;
        int i;
        int rv   = 0;
        int sent = 0;
        unsigned long flags;

        eoi = irq_count = 0;

        DPRINTK("cbm_write: %d bytes, atn=%d\n", cnt, atn);

        if(atn) {
                SET(ATN_OUT);
        }
        SET(CLK_OUT);
        RELEASE(DATA_OUT);

        for(i=0; (i<100) && !GET(DATA_IN); i++) {
            udelay(10);
        }

        if(!GET(DATA_IN)) {
                printk("cbm_write: no devices found\n");
                RELEASE(CLK_OUT | ATN_OUT);
                return -ENODEV;
        }

        current->state = TASK_INTERRUPTIBLE;
#ifdef KERNEL_VERSION
        schedule_timeout(HZ/50);   /* 20ms */
#else
        current->timeout = jiffies + 2;
        schedule();
#endif

        while(cnt > sent && rv == 0) {
                if(atn == 0) {
#ifdef KERNEL_VERSION
                    get_user(c, buf++);
#else
                    c = get_user(buf++);
#endif
                } else {
                    c = *buf++;
                }
                udelay(50);
                if(GET(DATA_IN)) {
                        irq_count = ((sent == (cnt-1)) && (atn == 0)) ? 2 : 1;
                        wait_for_listener();

                        if(signal_pending(current)) {
                                rv = -EINTR;
                        } else {
                                if(send_byte(c)) {
                                        sent++;
                                        udelay(100);
                                } else {
                                        printk("cbm_write: I/O error\n");
                                        rv = -EIO;
                                }
                        }
                } else {
                     printk("cbm_write: device not present\n");
                     rv = -ENODEV;
                }
        }
        DPRINTK("%d bytes sent, rv=%d\n", sent, rv);

        if(talk) {
                save_flags(flags); cli();
                SET(DATA_OUT);
                RELEASE(ATN_OUT);
                udelay(30);
                RELEASE(CLK_OUT);
                restore_flags(flags);
        } else {
                RELEASE(ATN_OUT);
        }
        udelay(100);

        return (rv < 0) ? rv : sent;
}

#ifdef KERNEL_VERSION
static int cbm_write(struct file *f, const char *buf, size_t cnt, loff_t *ppos)
#else
static int cbm_write(struct inode *inode, struct file *f, const char *buf, int cnt)
#endif
{
        return cbm_raw_write(buf, cnt, 0, 0);
}

static int cbm_ioctl(struct inode *inode, struct file *f,
                     unsigned int cmd, unsigned long arg)
{
        unsigned char buf[2], c, talk, mask, state, i;
        int rv = 0;

        buf[0] = (arg >> 8) & 0x1f;  /* device */
        buf[1] = arg & 0x0f;         /* secondary address */

        switch( cmd ) {
                case CBMCTRL_RESET:
                        do_reset();
                        return 0;

                case CBMCTRL_TALK:
                case CBMCTRL_LISTEN:
                        talk = (cmd == CBMCTRL_TALK);
                        buf[0] |= talk ? 0x40 : 0x20;
                        buf[1] |= 0x60;
                        rv = cbm_raw_write(buf, 2, 1, talk);
                        return rv > 0 ? 0 : rv;

                case CBMCTRL_UNTALK:
                case CBMCTRL_UNLISTEN:
                        buf[0] = (cmd == CBMCTRL_UNTALK) ? 0x5f : 0x3f;
                        rv = cbm_raw_write(buf, 1, 1, 0);
                        return rv > 0 ? 0 : rv;

                case CBMCTRL_OPEN:
                case CBMCTRL_CLOSE:
                        buf[0] |= 0x20;
                        buf[1] |= (cmd == CBMCTRL_OPEN) ? 0xf0 : 0xe0;
                        rv = cbm_raw_write(buf, 2, 1, 0);
                        return rv > 0 ? 0 : rv;

                case CBMCTRL_GET_EOI:
                        return eoi ? 1 : 0;

                case CBMCTRL_IEC_WAIT:
                        switch(arg >> 8) {
                            case IEC_DATA:
                                    mask = DATA_IN;
                                    break;
                            case IEC_CLOCK:
                                    mask = CLK_IN;
                                    break;
                            case IEC_ATN:
                                    mask = ATN_IN;
                                    break;
                            default:
                                    return -EINVAL;
                        }
                        state = (arg & 0xff) ? mask : 0;
                        i = 0;
                        while((inb(in_port) & mask) == state) {
                                if(i >= 20) {
                                    current->state = TASK_INTERRUPTIBLE;
#ifdef KERNEL_VERSION
                                    schedule_timeout(HZ/50);   /* 20ms */
#else
                                    current->timeout = jiffies + 2;
                                    schedule();
#endif
                                    if(signal_pending(current)) {
                                            return -EINTR;
                                    }
                                } else {
                                    i++;
                                    udelay(10);
                                }
                        }
                        /* fall through */

                case CBMCTRL_IEC_POLL:
                        c = inb(in_port);
                        if((c & DATA_IN) == 0) rv |= IEC_DATA;
                        if((c & CLK_IN ) == 0) rv |= IEC_CLOCK;
                        if((c & ATN_IN ) == 0) rv |= IEC_ATN;
                        return rv;

                case CBMCTRL_IEC_SET:
                        switch(arg) {
                            case IEC_DATA:
                                    SET(DATA_OUT);
                                    break;
                            case IEC_CLOCK:
                                    SET(CLK_OUT);
                                    break;
                            case IEC_ATN:
                                    SET(ATN_OUT);
                                    break;
                            default:
                                    return -EINVAL;
                        }
                        return 0;

                case CBMCTRL_IEC_RELEASE:
                        switch(arg) {
                            case IEC_DATA:
                                    RELEASE(DATA_OUT);
                                    break;
                            case IEC_CLOCK:
                                    RELEASE(CLK_OUT);
                                    break;
                            case IEC_ATN:
                                    RELEASE(ATN_OUT);
                                    break;
                            default:
                                    return -EINVAL;
                        }
                        return 0;

                case CBMCTRL_PP_READ:
                        if(!(out_bits & LP_BIDIR)) {
                            outb(0xff, port);
                            SET(LP_BIDIR);
                        }
                        return inb(port);

                case CBMCTRL_PP_WRITE:
                        if(out_bits & LP_BIDIR) {
                            RELEASE(LP_BIDIR);
                        }
                        outb(arg, port);
                        return 0;
        }
        return -EINVAL;
}

static int cbm_open(struct inode *inode, struct file *f)
{
        if(busy) return -EBUSY;
#ifdef DECLARE_WAITQUEUE
        init_waitqueue_head(&cbm_wait_q);
#endif
        busy = 1;
        MOD_INC_USE_COUNT;
        return 0;
}

static
#ifdef KERNEL_VERSION
int
#else
void
#endif
cbm_release(struct inode *inode, struct file *f)
{
        busy = 0;
        MOD_DEC_USE_COUNT;
#ifdef KERNEL_VERSION
        return 0;
#endif
}

static void cbm_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
        inb(in_port);   /* acknowledge interrupt */

        if(irq_count == 0) {
                return;
        }
        if(--irq_count == 0) {
                DPRINTK("continue to send (no EOI)\n");
                SET(CLK_OUT);
                wake_up_interruptible(&cbm_wait_q);
        }
}

static struct file_operations cbm_fops =
{
#ifdef KERNEL_VERSION
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0))
        THIS_MODULE,    /* owner */
#endif
        NULL,           /* seek */
        cbm_read,       /* read */
        cbm_write,      /* write */
        NULL,           /* readdir */
        NULL,           /* poll */
        cbm_ioctl,      /* ioctl */
        NULL,           /* mmap */
        cbm_open,       /* open */
        NULL,           /* flush */
        cbm_release,    /* release */
#else
        NULL,           /* seek */
        cbm_read,       /* read */
        cbm_write,      /* write */
        NULL,           /* readaddr */
        NULL,           /* select */
        cbm_ioctl,      /* ioctl */
        NULL,           /* mmap */
        cbm_open,       /* open */
        cbm_release,    /* release */
        NULL            /* sync */
#endif
};

static struct miscdevice cbm_dev=
{
        CBM_MINOR,
        NAME,
        &cbm_fops
};

void cbm_cleanup(void)
{
        free_irq(irq,NULL);
        release_region(port, 3);
        misc_deregister(&cbm_dev);
}

int cbm_init(void)
{
        unsigned char in, out;
        char *msg;

        if(check_region(port, 3)) {
                printk("cbm_init: port already in use\n");
                return -EBUSY;
        }
        if(request_irq(irq, cbm_interrupt, SA_INTERRUPT, NAME, NULL)) {
                printk("cbm_init: irq already in use\n");
                return -EBUSY;
        }
        request_region(port, 3, NAME);
        misc_register(&cbm_dev);

        in_port   = port+1;
        out_port  = port+2;

        if(cable < 0) {
            in    = GET(ATN_IN);
            out   = (inb(out_port) & ATN_OUT) ? 1 : 0;
            cable = (in != out);
            msg   = " (auto)";
        } else {
            msg   = "";
        }

        out_eor = cable ? 0xcb : 0xc4;

        printk("cbm_init: using %s cable%s\n",
                cable ? "active (XA1541)" : "passive (XM1541)", msg);

        irq_count = 0;

        out_bits  = (inb(out_port) ^ out_eor) &
                    (DATA_OUT|CLK_OUT|ATN_OUT|RESET);

        if(out_bits & RESET) do_reset();

        busy = 0;

        RELEASE(RESET | DATA_OUT | ATN_OUT | LP_BIDIR | LP_IRQ);
        SET(CLK_OUT);

        return 0;
}
