/*
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *  Copyright 1999 Michael Klein <michael.klein@puffin.lb.shuttle.de>
*/

#include "cbm4linux.h"

#include <ctype.h>
#include <errno.h>
#include <error.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static unsigned char dskfrmt[] = {
#include "cbmformat.inc"
};

static void help()
{
    printf(
"Usage: cbmformat [OPTION]... DRIVE NAME,ID\n\
Fast CBM-1541 disk formatter\n\
\n\
  -h    display this help and exit\n\
\n\
  -n    do not bump drive head\n\
  -x    format 40 track disk\n\
  -o    fill sectors with the original pattern (0x4b, 0x01, 0x01...)\n\
        instead of zeroes\n\
  -s    display drive status after formatting\n\
\n\
");
}

static void hint(char *s)
{
    fprintf(stderr, "Try `%s' -h for more information.\n", s);
}

int main(int argc, char *argv[])
{
    int fd, status = 0, drive, id_ofs = 0, name_len;
    unsigned char tracks = 35, bump = 1, orig = 0;
    char cmd[40], c, name[20], *arg;

    while((c=getopt(argc, argv, "nosxh")) != -1) {
        switch(c) {
            case 'n': bump = 0;
                      break;
            case 'o': orig = 1;
                      break;
            case 's': status = 1;
                      break;
            case 'x': tracks = 40;
                      break;
            case 'h': help();
                      return 0;
            default : hint(argv[0]);
                      return 1;
        }
    }

    if(optind + 2 != argc) {
        fprintf(stderr, "Usage: %s [OPTION]... DRIVE NAME,ID\n", argv[0]);
        hint(argv[0]);
        return 1;
    }

    arg = argv[optind++];
    drive = atoi(arg);
    if(drive < 8 || drive > 11) {
        fprintf(stderr, "Invalid drive number (%s)\n", arg);
        return 1;
    }
    
    arg      = argv[optind++];
    name_len = 0;
    while(*arg) {
        c = toupper(*arg);
        if(c == ',') {
            if(id_ofs) {
                fprintf(stderr, "More than one `,' in disk name\n");
                return 1;
            }
            id_ofs = name_len;
        }
        name[name_len++] = c;
        if(name_len > 19) {
            fprintf(stderr, "Disk name too long\n");
            return 1;
        }
        arg++;
    }
    
    fd = open(cbm_dev, O_RDWR);
    if(fd >= 0) {
        cbm_upload(fd, drive, 0x0500, dskfrmt, sizeof(dskfrmt));
        sprintf(cmd, "M-E%c%c%c%c%c0:%s", 64, 6, 
                     tracks+1, orig, bump, name);
        cbm_exec_command(fd, drive, cmd, 10+strlen(name));
        if(tracks > 35) {
            cbm_open(fd, drive, 2);
            write(fd, "#", 1);
            cbm_unlisten(fd);
            cbm_exec_command(fd, drive, "U1:2 0 18 0", 11);
            cbm_exec_command(fd, drive, "B-P2 192", 8);
            cbm_listen(fd, drive, 2);
            while(tracks > 35) {
                write(fd, "\021\377\377\001", 4);
                tracks--;
            }
            cbm_unlisten(fd);
            cbm_exec_command(fd, drive, "U2:2 0 18 0", 11);
            cbm_close(fd, drive, 2);
        }
        if(status) {
            cbm_device_status(fd, drive, cmd, sizeof(cmd));
            printf("%s\n", cmd);
        }
        close(fd);
        return 0;
    } else {
        error(0, errno, "%s", cbm_dev);
        return 1;
    }
}
