/*
 * linux/fs/cbmfs/inode.c
 *
 * Written 1995 by Dan Fandrich <dan@fch.wimsey.bc.ca>
 *
 * 1995-03-25	Converted from msdos/inode.c (Dan Fandrich)
 * 1999-08-20	Rewritten for kernel v2.2 (David Weinehall)
 * 1999-08-21	Code cleanup (David Weinehall)
 * 1999-08-24	Code rewritten to support non-modular mode (David Weinehall)
 * 1999-08-30	Added support for 1571 disk images (David Weinehall)
 * 1999-09-12	Fixed some bugs I managed to introduced when I added
 *		1571-support (David Weinehall)
 * 1999-09-16	Hopefully made auto-detection work properly even
 *		for extended 1541-disks (David Weinehall)
 */

/* REMEMBER: append mode probably cannot work with the fastsize option */

#include <linux/version.h>
//#define __NO_VERSION__
#include <linux/module.h>

#include <linux/cbm_fs.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/major.h>
#include <linux/blkdev.h>
#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/locks.h>
#include <linux/init.h>

#include "cbmbuffer.h"

#include <asm/segment.h>
#include <asm/uaccess.h>


#define VERSION "0.4e"	/* filesystem version number */

/* No writing for cbm yet */
#if MAY_NEED_FOR_FUTURE_WRITE_SUPPORT
void cbm_put_inode(struct inode *inode)
{
	struct super_block *sb;

	if (inode->i_nlink) {
		if (CBM_I(inode)->i_busy)
			cbm_cache_inval_inode(inode);
		return;
	}
	inode->i_size = 0;
	cbm_truncate(inode);
	sb = inode->i_sb;
	clear_inode(inode);
}
#endif


void cbm_put_super(struct super_block *sb)
{
	/*
	 * Invalidate our cache of file locations for this device
	 */

	cbm_cache_inval_dev(sb->s_dev);
	/*
	 * Restore default blocksize for the device
	 */
	set_blocksize(sb->s_dev, BLOCK_SIZE);

	MOD_DEC_USE_COUNT;	/* we're done with this filesystem */
	return;
}

static struct super_operations cbm_sops = {
	cbm_read_inode,			/* read_inode */
	NULL, /* cbm_write_inode */	/* write_inode */
	NULL,				/* put_inode */
	NULL, /* cbm_delete_inode */	/* delete_inode */
	cbm_notify_change,		/* notify_change */
	cbm_put_super,			/* put_super */
	NULL,				/* write_super */
	cbm_statfs,			/* statfs */
	NULL,				/* remount */
	NULL /* cbm_clear_inode */	/* clear_inode */
};

/*
 * Parse the mount options
 */

static int parse_options(char *options, char *check,
		         int *fstype, uid_t *uid, gid_t *gid, int *umask,
			 int *debug, int *quiet, int *fastsize, int *nous)
{
	char *this_char, *value;

	*check = 'n';
	*uid = current->uid;
	*gid = current->gid;
	*umask = current->fs->umask;
	*fstype = *debug = *quiet = *fastsize = *nous = 0;
	if (!options)
		return 1;

	for (this_char = strtok(options, ","); this_char; this_char = strtok(NULL, ",")) {
		if ((value = strchr(this_char, '=')) != NULL)
			*value++ = 0;

		/*
		 * File name checking strictness
		 */

		if (!strcmp(this_char, "check") && value) {
			if (value[0] && !value[1] && strchr("ns", *value)) {
				*check = *value;
			} else if (!strcmp(value, "normal")) {
				*check = 'n';
			} else if (!strcmp(value, "strict")) {
				*check = 's';
			} else {
				return 0;
			}
		}

		/*
		 * Commodore disk type
		 */

		else if (!strcmp(this_char, "type")) {
			if (!value || !*value)
				return 0;
			*fstype = simple_strtoul(value, &value, 0);
			if (*value || (*fstype != 1541 &&
				       *fstype != 1571 &&
				       *fstype != 1581)) {
				/* If an invalid type is supplied, ignore
				 * it and warn
				 */
				printk(KERN_NOTICE "CBMFS: Illegal type supplied!");
				*fstype = 0;
				return 0;
			}
		}

		/*
		 * User ID for files
		 */

		else if (!strcmp(this_char, "uid")) {
			if (!value || !*value)
				return 0;
			*uid = simple_strtoul(value, &value, 0);
			if (*value)
				return 0;
		}

		/*
		 * Group ID for files
		 */

		else if (!strcmp(this_char, "gid")) {
			if (!value || !*value)
				return 0;
			*gid = simple_strtoul(value, &value, 0);
			if (*value)
				return 0;
		}

		/*
		 * umask for files
		 */

		else if (!strcmp(this_char, "umask")) {
			if (!value || !*value)
				return 0;
			*umask = simple_strtoul(value, &value, 8);
			if (*value)
				return 0;
		}

		/*
		 * Display debug information on mounting
		 */

		else if (!strcmp(this_char, "debug")) {
			if (value)
				return 0;
			*debug = 1;
		}

		/*
		 * Don't return errors when changing file attributes
		 */

		else if (!strcmp(this_char, "quiet")) {
			if (value)
				return 0;
			*quiet = 1;
		}

		/*
		 * Approximate file sizes
		 */

		else if (!strcmp(this_char, "fastsize")) {
			if (value)
				return 0;
			*fastsize = 1;
		}

		/*
		 * Show spaces in file names
		 */

		else if (!strcmp(this_char, "nous")) {
			if (value)
				return 0;
			*nous = 1;
		}

		/*
		 * No valid mount-time options
		 */
		else return 0;
	}
	return 1;
}

int cbm_determine_cbmfs_type(struct super_block *sb, int silent, char check, uid_t uid, gid_t gid, int umask) {

	struct buffer_head *bh, *bht;

	int i, error;

	/* Test for a 1581 file-system */

	bht = cbm_bread(sb, cbm_1581_block(CBM_1581_TRACKS, 0));
	bh = cbm_bread(sb, CBM_1581_SUPER_LOC);

	if ((bht != NULL) && (bh != NULL)) {
		struct cbm_1581_header_sector *b1581;
		b1581 = (struct cbm_1581_header_sector *) bh->b_data;
		error = !cbm_is_1581_header(b1581) ||
			 (b1581->t != CBM_1581_DIR_TRACK);
	} else {
		error = 1; /* The disk is too small, or there's no
			    * valid superblock
			    */
	}

	cbm_brelse(bht);
	cbm_brelse(bh);

	if (!error) {
		return 1581;	/* 1581 */
	}

	/* Test for a 1571 file-system */

	bht = cbm_bread(sb, cbm_1571_block(CBM_1571_TRACKS, 0));
	bh = cbm_bread(sb, CBM_1571_SUPER_LOC);

	if ((bht != NULL) && (bh != NULL)) {
		struct cbm_1571_header_sector *b1571;
		b1571 = (struct cbm_1571_header_sector *) bh->b_data;
		error = !cbm_is_1571_header(b1571) ||
			 (b1571->t != CBM_1571_DIR_TRACK);
	} else {
		error = 1;
	}

	cbm_brelse(bht);
	cbm_brelse(bh);

	if (!error) {
		return 1571;	/* 1571 */
	}

	/* Test for a 1541 file-system */

	bh = cbm_bread(sb, CBM_1541_SUPER_LOC);

	if (bh != NULL) {
		struct cbm_1541_header_sector *b1541;
		b1541 = (struct cbm_1541_header_sector *) bh->b_data;
		error = !cbm_is_1541_header(b1541) ||
			 (b1541->t != CBM_1541_DIR_TRACK);
	} else {
		error = 1;
	}

	if (!error) {
		cbm_brelse(bh);

		for (i = 10; i >= 0; i--) {
			bht = cbm_bread(sb, cbm_1541_block(CBM_1541_TRACKS + i, 0));
			cbm_brelse(bht);

			if (bht != NULL)
				return 1541 + i;
		}
	}

	/* We didn't find a superblock, so we give up... */
	if (!silent) {
		printk(KERN_NOTICE "[CBM FS Rel. " VERSION ", check=%c, "
		       "uid=%d, gid=%d, umask=%03o]\n",
		       check, uid, gid, umask);
		printk(KERN_NOTICE "[sb=%lu, sl=%u, rs=%lu, re=%lu, tb=%lu]\n",
		       CBM_SB(sb)->super_start, CBM_SB(sb)->super_length,
		       CBM_SB(sb)->root_start, CBM_SB(sb)->root_high,
		       CBM_SB(sb)->blocks);

		printk(KERN_ERR "CBMFS: Can't find a valid CBM "
		       "filesystem on dev 0x%04x.\n", sb->s_dev);
	}

	unlock_super(sb);
	sb->s_dev = 0;
	MOD_DEC_USE_COUNT;
	return 0;
}

/*
 * Read the super block of CBMFS.
 * This is the first function called after mounting the filesystem
 */

struct super_block *cbm_read_super(struct super_block *sb, void *data,
				     int silent)
{
	struct buffer_head *bh;
	int fstype, mfstype, debug, quiet, fastsize, nous, umask;
	char check;
	uid_t uid;
	gid_t gid;

	/* Set them all to 0 to quench compile-time warnings */
	int extra_blocks = 0;
	unsigned int sbfstype = 0;
	unsigned long sbsuperstart = 0;
	int btrack = 0;
	int bsector = 0;
	int bformat = 0;
	int bflag = 0;

	int blksize = 512;

	MOD_INC_USE_COUNT;

	if (hardsect_size[MAJOR(sb->s_dev)] != NULL) {
		blksize = hardsect_size[MAJOR(sb->s_dev)][MINOR(sb->s_dev)];
	}

	/*
	 * Parse mount options
	 */
	if (!parse_options((char *) data, &check, &mfstype, &uid,
			   &gid, &umask, &debug, &quiet, &fastsize, &nous)
		|| (blksize != 512)) {
		/* Cannot handle this sector size -- abort */
		sb->s_dev = 0;
		MOD_DEC_USE_COUNT;
		return NULL;
	}

	cbm_cache_init();
	lock_super(sb);

	sb->s_blocksize = CBM_BLOCK_SIZE;
	set_blocksize(sb->s_dev, CBM_BLOCK_SIZE);

	/*
	 * See if device contains a cbm filesystem, or if a filesystem
	 * type is specified, make sure that the disk is readable
	 */

	fstype = cbm_determine_cbmfs_type(sb, silent, check, uid, gid, umask);

	if (fstype == 0) {
		return NULL;	/* Unable to determine filesystem type
				 * or to read superblock
				 */
	}

	if ((fstype != mfstype) && (mfstype != 0)) {
		printk(KERN_NOTICE "CBMFS: Overriding detected type\n");
		fstype = mfstype;
	}

	if ((fstype > 1541) && (fstype <= 1551)) {
		/* We assume that a 1541 can't possibly squeeze
		 * more than 10 tracks extra onto the disk
		 * (even this is quite unrealistic...)
		 */
		int extra = fstype - 1541;

		extra_blocks = cbm_1541_block(CBM_1541_TRACKS + extra, 0)
			       - cbm_1541_block(CBM_1541_TRACKS, 0);
		fstype = 1541;
	}

	switch (fstype) {
		case 1541: { /* 1541 */
			struct cbm_1541_header_sector *b1541;

			bh = cbm_bread(sb, CBM_1541_SUPER_LOC);
			b1541 = (struct cbm_1541_header_sector *) bh->b_data;

			/* 1541 disk */
			CBM_SB(sb)->fs_type = 1541;
			CBM_SB(sb)->super_start = CBM_1541_SUPER_LOC;
			CBM_SB(sb)->super_length = CBM_1541_SUPER_LEN;
			CBM_SB(sb)->blocks = CBM_1541_BLOCKS + extra_blocks;

			/* Should always be t18 s1 */
			CBM_SB(sb)->root_start = cbm_1541_block(b1541->t, b1541->s);

			/* Last reserved directory sector -- t18 s18 */
			CBM_SB(sb)->root_high = cbm_1541_block(b1541->t, 18);

			sbfstype = CBM_SB(sb)->fs_type;
			sbsuperstart = CBM_SB(sb)->super_start,
			btrack = (int) b1541->t;
			bsector = (int) b1541->s;
			bformat = (int) b1541->disk_format;
			bflag = (int) b1541->flag;

			cbm_brelse(bh);
			break;
		}
		case 1571: { /* 1571 */
			struct cbm_1571_header_sector *b1571;

			bh = cbm_bread(sb, CBM_1571_SUPER_LOC);
			b1571 = (struct cbm_1571_header_sector *) bh->b_data;

			/* 1571 disk */
			CBM_SB(sb)->fs_type = 1571;
			CBM_SB(sb)->super_start = CBM_1571_SUPER_LOC;
			CBM_SB(sb)->super_length = CBM_1571_SUPER_LEN;
			CBM_SB(sb)->blocks = CBM_1571_BLOCKS;

			/* Should always be t18 s1 */
			CBM_SB(sb)->root_start = cbm_1571_block(b1571->t, b1571->s);
			/* Last reserved directory sector -- t18 s18 */
			CBM_SB(sb)->root_high = cbm_1571_block(b1571->t, 18);

			sbfstype = CBM_SB(sb)->fs_type;
			sbsuperstart = CBM_SB(sb)->super_start,
			btrack = (int) b1571->t;
			bsector = (int) b1571->s;
			bformat = (int) b1571->disk_format;
			bflag = (int) b1571->flag;

			cbm_brelse(bh);
			break;
		}
		case 1581: { /* 1581 */
			struct cbm_1581_header_sector *b1581;

			bh = cbm_bread(sb, CBM_1581_SUPER_LOC);
			b1581 = (struct cbm_1581_header_sector *) bh->b_data;

			/* 1581 disk */
			CBM_SB(sb)->fs_type = 1581;
			CBM_SB(sb)->super_start = CBM_1581_SUPER_LOC;
			CBM_SB(sb)->super_length = CBM_1581_SUPER_LEN;
			CBM_SB(sb)->blocks = CBM_1581_BLOCKS;

			/* Should always be t40 s3 */
			CBM_SB(sb)->root_start = cbm_1581_block(b1581->t, b1581->s);
			/* Last reserved directory sector -- t40 s39 */
			CBM_SB(sb)->root_high = cbm_1581_block(b1581->t, 39);

			sbfstype = CBM_SB(sb)->fs_type;
			sbsuperstart = CBM_SB(sb)->super_start,
			btrack = (int) b1581->t;
			bsector = (int) b1581->s;
			bformat = (int) b1581->disk_format;
			bflag = (int) b1581->flag;

			cbm_brelse(bh);
			break;	/* This break isn't really needed, it's
				 * here purely for future expansions
				 */
		}
	}

	printk(KERN_DEBUG "[%d BAM @%ld: %d %d %2x %2x]\n",
	       sbfstype, sbsuperstart, btrack, bsector, bformat, bflag);

	/*
	 * This must be done after the brelse because the bh is
	 * a dummy allocated by cbm_bread (see buffer.c) [why? -- DF]
	 */
	sb->s_blocksize_bits = 9;	/* 512 byte blocks -- is this
					 * supposed to be the size of
					 * of the cache blocks?
					 */

	printk(KERN_DEBUG "[CBMFS Rel. " VERSION ", check=%c, uid=%d, "
	       "gid=%d, umask=%03o]\n", check, uid, gid, umask);
	printk(KERN_DEBUG "[sb=%lu, sl=%u, rs=%lu, re=%lu, tb=%lu]\n",
	       CBM_SB(sb)->super_start, CBM_SB(sb)->super_length,
	       CBM_SB(sb)->root_start, CBM_SB(sb)->root_high,
	       CBM_SB(sb)->blocks);

	sb->s_magic = CBM_SUPER_MAGIC;
	CBM_SB(sb)->name_check = check;
	unlock_super(sb);

	/*
	 * Set up enough so that it can read an inode
	 */
	sb->s_op = &cbm_sops;
	CBM_SB(sb)->fs_uid = uid;
	CBM_SB(sb)->fs_gid = gid;
	CBM_SB(sb)->fs_umask = umask;
	CBM_SB(sb)->quiet = quiet;
	CBM_SB(sb)->fastsize = fastsize;
	CBM_SB(sb)->nous = nous;
	CBM_SB(sb)->bam_wait = NULL;
	CBM_SB(sb)->bam_lock = 0;
	CBM_SB(sb)->prev_free = 0;
	CBM_SB(sb)->free_blocks = -1; /* don't know yet */
	sb->s_root = d_alloc_root(iget(sb, CBM_ROOT_INODE(sb)), NULL);
	if (!sb->s_root) {
		sb->s_dev = 0;
		printk(KERN_ERR "get root inode failed\n");
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	return sb;
}


/*
 * Return disk free space and free inodes given a superblock
 */

int cbm_statfs(struct super_block *sb, struct statfs *buf, int bufsiz)
{
	int free;

	struct statfs tmp;

	if (bufsiz < 40)
		return -EFAULT;		/*
					 * Aborting is easier than changing
					 * this routine to return only what
					 * is asked
					 */

	tmp.f_type = sb->s_magic;
	tmp.f_bsize = CBM_LBLOCK_SIZE;	/* should be CBM_DATA_SIZE */
	tmp.f_blocks = CBM_SB(sb)->blocks;

	/*
	 * Don't change the BAM while we're counting
	 */

	cbm_lock_bam(sb);

	/*
	 * Has free_blocks already been determined?
	 */

	if (CBM_SB(sb)->free_blocks >= 0) {
		free = CBM_SB(sb)->free_blocks;
	} else {
		free = cbm_free_blocks(sb);
		CBM_SB(sb)->free_blocks = free;
	}
	cbm_unlock_bam(sb);

	tmp.f_bfree = free;
	tmp.f_bavail = free;
	tmp.f_files = 0;		/* Don't support inode counting */
	tmp.f_ffree = 0;		/* Don't support inode counting */
	tmp.f_namelen = 16;

	return copy_to_user(buf, &tmp, bufsiz) ? -EFAULT : 0;
}

/*
 * Return the disk block given an inode and logical file block
 */

int cbm_bmap(struct inode *inode, int block)
{
	/*
	 * NOTE: There seems to be a problem here. Returning 0 seems
	 * to indicate an error, but on CBM disks, sector 0 is just as
	 * valid as any. -1 as error would work, but how does the
	 * calling function handle that?
	 */
	if ((block = cbm_get_sector(inode, block)) < 0)
		return 0; /* try -1 */
	return block;
}

/*
 * Fill in the inode structure at the pointer, given the inode number
 * in inode->i_ino
 *
 * Much of the directory stuff is 1581-specific and would need to be modified
 * in case directories were desired on 1541 disks.
 */

void cbm_read_inode(struct inode *inode)
{
	struct super_block *sb = inode->i_sb;
	struct buffer_head *bh;
	struct cbm_dir_entry *raw_entry;
	unsigned long blocks;

	printk(KERN_DEBUG "cbm_read_inode: %ld\n", inode->i_ino);

	CBM_I(inode)->i_busy = 0;
	inode->i_uid = CBM_SB(sb)->fs_uid;
	inode->i_gid = CBM_SB(sb)->fs_gid;

	/*
	 * Root inode (special case inode)
	 */
	if (inode->i_ino == CBM_ROOT_INODE(sb)) {
		inode->i_mode = S_IFDIR | (S_IRWXUGO & ~CBM_SB(sb)->fs_umask);
		inode->i_op = &cbm_dir_inode_operations;

		CBM_I(inode)->i_start = CBM_SB(sb)->root_start;
		CBM_I(inode)->i_type = CBM_CBM | CBM_CLOSED;
		CBM_I(inode)->i_rec_size = 0;
		CBM_I(inode)->i_side_sec = 0;
		CBM_I(inode)->i_parent = inode->i_ino;	/*
							 * not really parent,
							 * but doesn't matter
							 */
		if (CBM_SB(sb)->fastsize) {

			/*
			 * Return the size of a standard disk directory
			 */

			switch (CBM_SB(sb)->fs_type) {
				case 1541: blocks = CBM_1541_DIR_LEN;
					   break;
				case 1571: blocks = CBM_1571_DIR_LEN;
					   break;
				case 1581:
				default: blocks = CBM_1581_DIR_LEN;
			}
		} else {

			/*
			 * Count the actual number of directory blocks here,
			 * just in case the directory is a nonstandard
			 * extended one.
			 *
			 * Round up to the least number of blocks needed
			 * to hold the data. This will give strange results
			 * if the directory length == 0 (which should never
			 * happen), or if an error occurred while reading
			 * the disk (which might happen)
			 */

			blocks = 1 + ((cbm_chain_len(inode) - 1) / CBM_DATA_SIZE);
		}

		inode->i_size = CBM_DATA_SIZE * blocks;
		inode->i_blksize = CBM_LBLOCK_SIZE; /* should be CBM_DATA_SIZE */
		inode->i_blocks = blocks;

		/*
		 * Currently, all subdirectories below root have '..' pointing to
		 * the root, so this next line isn't always technically correct
		 */

		inode->i_nlink = cbm_subdirs(inode) + 2;

		/*
		 * CBM doesn't have file times
		 */

		inode->i_mtime = inode->i_atime = inode->i_ctime = 0;

		return;		/* Return with changed root inode information */
	}

	/*
	 * Inode is from a disk entry
	 *
	 * Convert the inode number into a corresponding disk directory block
	 */

	if (!(bh = cbm_bread(sb, CBM_INOBLOCK(inode->i_ino)))) {
		printk(KERN_CRIT "dev = 0x%04X, ino = %ld\n", inode->i_dev, inode->i_ino);
		panic("cbm_read_inode: unable to read i-node block");
	}
	raw_entry = &((struct cbm_dir_entry *)
		((struct cbm_data_sector *) bh->b_data)->data)[CBM_INODIRENT(inode->i_ino)];
	printk(KERN_DEBUG "cbm_read_inode: name \"%-16s\"\n", raw_entry->name);

	if (IS_CBM(raw_entry->type)) {

	/*
	 * Partition or directory entry (1581 only)
	 */
		struct buffer_head *new_bh;
		struct cbm_1581_header_sector *head;

		/*
		 * Must read the first sector of the partition to find what it is
		 */

		if (!(new_bh = cbm_bread(sb, cbm_block(sb, raw_entry->t, raw_entry->s)))) {
			printk(KERN_CRIT "dev = 0x%04X, ino = %ld\n", inode->i_dev, inode->i_ino);
			panic("cbm_read_inode: unable to read i-node block");
		}
		head = (struct cbm_1581_header_sector *) new_bh->b_data;
		if (cbm_is_1581_header(head)) {

			/*
			 * Partition is a subdirectory
			 */

			inode->i_mode = CBM_MKMODE(raw_entry->type, S_IRWXUGO &
			    ~CBM_SB(sb)->fs_umask) | S_IFDIR;
			inode->i_op = &cbm_dir_inode_operations;
			inode->i_nlink = cbm_subdirs(inode) + 2;

			CBM_I(inode)->i_start = cbm_block(sb, head->t, head->s);

			if (CBM_SB(sb)->fastsize) {
				/*
				 * Return the size of a standard disk directory
				 *
				 * If dirs are to be supported on 1541,
				 * then CBM_1541_DIR_LEN isn't correct for
				 * some tracks (not enough sectors) but
				 * shouldn't matter
				 */

				switch (CBM_SB(sb)->fs_type) {
					case 1541: blocks = CBM_1541_DIR_LEN;
						   break;
					case 1571: blocks = CBM_1571_DIR_LEN;
						   break;
					case 1581:
					default: blocks = CBM_1581_DIR_LEN;
				}
			} else {

				/*
				 * Count the actual number of directory
				 * blocks here, just in case the directory
				 * is a nonstandard extended one
				 *
				 * Round up to the least number of blocks
				 * needed to hold the data This will give
				 * strange results if the directory length == 0
				 * (which should never happen), or if an
				 * error occurred while reading the disk
				 * (which might happen)
				 */

				blocks = 1 + ((cbm_chain_len(inode) - 1) / CBM_DATA_SIZE);
			}
			inode->i_blocks = blocks;
			inode->i_size = blocks * CBM_DATA_SIZE;

		} else {

			/*
			 * Partition entry which isn't a directory
			 * (treat as a nonreadable file)
			 */

			inode->i_mode = S_IFREG; /* file is not accessible
						  * (0 mask) because the
						  * partition may not have
						  * T&S links
						  */
			inode->i_nlink = 1;	/* CBM doesn't have
						   hard links */
			inode->i_op = &cbm_file_inode_operations;
			inode->i_size = CF_LE_W(raw_entry->blocks) * CBM_DATA_SIZE;
			inode->i_blocks = CF_LE_W(raw_entry->blocks);
			CBM_I(inode)->i_start = cbm_block(sb, raw_entry->t, raw_entry->s);
		}

		cbm_brelse(new_bh);

	} else {

		/*
		* Regular file inode
		*/

		inode->i_mode = S_IFREG | CBM_MKMODE(raw_entry->type,
			((IS_NOEXEC(inode) ? S_IRUGO|S_IWUGO : S_IRWXUGO) &
			~CBM_SB(sb)->fs_umask));
		inode->i_op = &cbm_file_inode_operations;
		inode->i_nlink = 1;		/* CBM doesn't have links */
		CBM_I(inode)->i_start = cbm_block(sb, raw_entry->t, raw_entry->s);

		if (CBM_SB(sb)->fastsize) {	/* approximate file size */
			inode->i_size = CF_LE_W(raw_entry->blocks) * CBM_DATA_SIZE;
		} else {
			/*
			 * Count the actual number of bytes here
			 *
			 * BUG: An error while reading the file is ignored!
			 */
			inode->i_size = cbm_chain_len(inode);
		}

		inode->i_blocks = CF_LE_W(raw_entry->blocks);
			/* this is for CBM_DATA_SIZE-sized blocks */
	}

	/*
	 * This is common to directories, partitions and files
	 */

	CBM_I(inode)->i_side_sec = cbm_block(sb, raw_entry->ss_t, raw_entry->ss_s);
	CBM_I(inode)->i_rec_size = (unsigned long) raw_entry->rec_size;
	CBM_I(inode)->i_type = raw_entry->type;

	/*
	 * The parent won't be correct for subdirectories > 1 level down from root
	 */
	CBM_I(inode)->i_parent = CBM_ROOT_INODE(sb);

	inode->i_blksize = CBM_LBLOCK_SIZE;	/* should be CBM_DATA_SIZE */
	inode->i_mtime = inode->i_atime = inode->i_ctime = 0;

	cbm_brelse(bh);
}


#if MAY_NEED_FOR_FUTURE_WRITE_SUPPORT
void cbm_write_inode(struct inode *inode)
{
	struct super_block *sb = inode->i_sb;
	struct buffer_head *bh;
	struct cbm_dir_entry *raw_entry;

	inode->i_dirt = 0;
	if (inode->i_ino == CBM_ROOT_INO || !inode->i_nlink)
		return;
	if (!(bh = bread(inode->i_dev, inode->i_ino >> CBM_DPB_BITS,
	    SECTOR_SIZE))) {
		printk(KERN_CRIT "dev = 0x%04X, ino = %ld\n",
		       inode->i_dev, inode->i_ino);
		panic("cbm_write_inode: unable to read i-node block");
	}
	raw_entry = &((struct cbm_dir_entry *) (bh->b_data))
	    [inode->i_ino & (CBM_DPB - 1)];

	if (S_ISDIR(inode->i_mode)) {
		raw_entry->attr = ATTR_DIR;
		raw_entry->size = 0;
	} else {
		raw_entry->attr = ATTR_NONE;
		raw_entry->size = CT_LE_L(inode->i_size);
	}
	raw_entry->attr |= CBM_MKATTR(inode->i_mode) |
	    CBM_I(inode)->i_attrs;
	raw_entry->start = CT_LE_L(CBM_I(inode)->i_start);
	date_unix2dos(inode->i_mtime, &raw_entry->time, &raw_entry->date);
	raw_entry->time = CT_LE_W(raw_entry->time);
	raw_entry->date = CT_LE_W(raw_entry->date);
	mark_buffer_dirty(bh, 1);
	brelse(bh);
}
#endif


/*
 * Verify & process changes to an inode's attributes?
 */

int cbm_notify_change(struct dentry *dentry, struct iattr *attr)
{
	struct inode *inode = dentry->d_inode;
	int error;

	error = inode_change_ok(inode, attr);
	if (error)
		return error;

	if (((attr->ia_valid & ATTR_UID) &&
	     (attr->ia_uid != CBM_SB(inode->i_sb)->fs_uid)) ||
	    ((attr->ia_valid & ATTR_GID) &&
	     (attr->ia_gid != CBM_SB(inode->i_sb)->fs_gid)) ||
	    ((attr->ia_valid & ATTR_MODE) &&
	     (attr->ia_mode & ~CBM_VALID_MODE)))
		error = -EPERM;

	/*
	 * Don't report an illegal mode error in quiet mode
	 */

	if (error)
		return CBM_SB(inode->i_sb)->quiet ? 0 : error;

	inode_setattr(inode, attr);

	if (IS_NOEXEC(inode) && !S_ISDIR(inode->i_mode)) {
		inode->i_mode &= S_IFMT | S_IRUGO | S_IWUGO;
	} else {
		inode->i_mode |= S_IXUGO;
	}

	inode->i_mode = ((inode->i_mode & S_IFMT) | ((((inode->i_mode & S_IRWXU
	    & ~CBM_SB(inode->i_sb)->fs_umask) | S_IRUSR) >> 6)*S_IXUGO)) &
	    ~CBM_SB(inode->i_sb)->fs_umask;
	return 0;
}

struct file_system_type cbm_fs_type = {
	"cbm",
	FS_REQUIRES_DEV,
	cbm_read_super,
	NULL
};

__initfunc(int init_cbm_fs(void))
{
	return register_filesystem(&cbm_fs_type);
}

#ifdef MODULE
EXPORT_NO_SYMBOLS;

int init_module(void)
{
	return init_cbm_fs();
}

void cleanup_module(void)
{
	unregister_filesystem(&cbm_fs_type);
}
#endif
