/*
 * linux/fs/cbm/misc.c
 *
 * Written 1995 by Dan Fandrich <dan@fch.wimsey.bc.ca>
 *
 * 1999-08-30	Added support for 1571 disk images (David Weinehall)
 */

#include <linux/fs.h>
#include <linux/cbm_fs.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/stat.h>

#include "cbmbuffer.h"


/*
 * cbm_fs_panic reports a severe file system problem and sets the file system
 * read-only. The file system can be made writable again by remounting it.
 *
 * CBM_RDONLY would be the same as MS_RDONLY, so we don't create a special
 * flag for this.
 */

void cbm_fs_panic(struct super_block *sb, char *msg)
{
	int not_ro;

	not_ro = !(sb->s_flags & MS_RDONLY);
	if (not_ro)
		sb->s_flags |= MS_RDONLY;

	printk(KERN_CRIT "Filesystem panic (dev %s).\n  %s\n",
	       kdevname(sb->s_dev), msg);

	if (not_ro)
		printk(KERN_NOTICE "  File system has been set read-only\n");
}

/*
 * Compare raw CBM file names for equality
 * CBM_END_NAME is considered the same as a null character
 *
 * Input: name1 and name2 (must be non-NULL, may be unterminated)
 * Output: 0 if names are equal, 1 if they are not
 */

inline static int cbm_strcmp(const unsigned char *n1, const unsigned char *n2)
{
	int count;

	for (count = 0; count < CBM_NAME; ++count, ++n1, ++n2) {
		if ((*n1 != *n2) &&
		    (!((*n1 == 0 && *n2 == CBM_END_NAME) ||
			   (*n2 == 0 && *n1 == CBM_END_NAME)))) {
			return 1;		/* name mismatch */
		}

		if (*n1 == 0 || *n1 == CBM_END_NAME) {
			break;			/* end of name */
		}
	}

	return 0;		/* matching names */
}

/*
 * Given a Commodore 1541 track and sector number, returns the CBM block
 * number relative to the start of the disk
 */

unsigned long cbm_1541_block(int track, int sector)
{
	static const unsigned sectors[45] = {
		/* 21 sectors long */
		0, 21, 42, 63, 84, 105, 126, 147, 168, 189,	/* 01-10 */
                210, 231, 252, 273, 294, 315, 336, 357,		/* 11-18 */
		/* 19 sectors long */
		376, 395, 414, 433, 452, 471, 490,		/* 19-25 */
		/* 18 sectors long */
		508, 526, 544, 562, 580, 598,			/* 26-31 */
		/* 17 sectors long */
		615, 632, 649, 666,				/* 32-35 */

		/*
		 * The rest of the tracks are nonstandard,
		 * they too are 17 sectors long (I hope)
		 */
		683, 700, 717, 734, 751,			/* 36-40 */
		768, 785, 802, 819, 836				/* 41-45 */
	};

	return sectors[track - 1] + sector;
}

/*
 * Given a Commodore 1571 track and sector number, returns the CBM block number
 * relative to the start of the disk
 */

unsigned long cbm_1571_block(int track, int sector)
{
	static const unsigned sectors[70] = {
		/* These 35 sectors are "side 1" of the disk */

		/* 21 sectors long */
		0, 21, 42, 63, 84, 105, 126, 147, 168,		/* 01-09 */
                189, 210, 231, 252, 273, 294, 315, 336, 357,	/* 10-18 */
		/* 19 sectors long */
		376, 395, 414, 433, 452, 471, 490,		/* 19-25 */
		/* 18 sectors long */
		508, 526, 544, 562, 580, 598,			/* 26-31 */
		/* 17 sectors long */
		615, 632, 649, 666,				/* 32-35 */

		/* These 35 sectors are "side 2" of the disk */

		/* 21 sectors long */
		683, 704, 725, 746, 767, 788, 809, 830, 851,	/* 36-44 */
                872, 893, 914, 935, 956, 977, 998, 1019, 1040,	/* 45-53 */
		/* 19 sectors long */
		1059, 1078, 1097, 1116, 1137, 1158, 1179,	/* 54-60 */
		/* 18 sectors long */
		1197, 1215, 1233, 1251, 1269, 1287,		/* 61-66 */
		/* 17 sectors long */
		1304, 1321, 1338, 1355				/* 67-70 */

	};

	return sectors[track - 1] + sector;
}

/*
 * Given a Commodore 1581 track and sector number, returns the CBM block number
 * relative to the start of the disk
 */

unsigned long cbm_1581_block(int track, int sector)
{
	return (track - 1) * 40 + sector;
}

/*
 * Given a Commodore track and sector and superblock, return the correct CBM
 * block number for the disk type
 */

unsigned long cbm_block(struct super_block *sb, int track, int sector)
{
	/* Allow for future expansions; FD-2000/4000, etc. */

	switch (CBM_SB(sb)->fs_type) {
		case 1541: return cbm_1541_block(track, sector);
		case 1571: return cbm_1571_block(track, sector);
		case 1581:
		default: return cbm_1581_block(track, sector);
	}
}

/*
 * Given a sector number, returns the 1541 track and sector number
 */

void cbm_1541_ts(unsigned long block, int *track, int *sector)
{
	if (block < 357)			/* tracks 1-17 */
		*track = block / 21;
	else if (block < 490)			/* tracks 18-24 */
		*track = block / 19;
	else if (block < 598)			/* tracks 25-30 */
		*track = block / 18;
	else					/* tracks 31-40 */
		*track = block / 17;

	*sector = (long) block % (*track)++;	/* CBM tracks are numbered from 1 */
}

/*
 * Given a sector number, returns the 1571 track and sector number
 */

void cbm_1571_ts(unsigned long block, int *track, int *sector)
{
	if (block < 357)			/* tracks 1-17 */
		*track = block / 21;
	else if (block < 490)			/* tracks 18-24 */
		*track = block / 19;
	else if (block < 598)			/* tracks 25-30 */
		*track = block / 18;
	else if (block < 683)			/* tracks 31-35 */
		*track = block / 17;
	else if (block < 1040)
		*track = block / 21;		/* tracks 36-52 */
	else if (block < 1179)
		*track = block / 19;		/* tracks 53-59 */
	else if (block < 1355)
		*track = block / 18;		/* tracks 60-65 */
	else
		*track = block / 17;		/* tracks 66-70 */

	*sector = (long) block % (*track)++;	/* CBM tracks are numbered from 1 */
}

/*
 * Given a sector number, returns the 1581 track and sector number
 */

void cbm_1581_ts(unsigned long block, int *track, int *sector)
{
	*track = block / 40;	/* 40 logical sectors per cylinder */
	*sector = (long) block % (*track)++;	/* CBM tracks are numbered from 1 */
}

/*
 * Count the number of free blocks on the 1541 disk, not including the
 * directory track
 */

inline long cbm_1541_free_blocks(struct super_block *sb)
{
	int track, free_blocks = 0;
	struct buffer_head *bh;
	struct cbm_1541_header_sector *sec;

	if (!(bh = cbm_bread(sb, CBM_SB(sb)->super_start))) {
		printk(KERN_WARNING "cbm_bread in cbm_1541_free_blocks failed\n");
		return -2;
	}
	sec = (struct cbm_1541_header_sector *) bh->b_data;

	/*
	 * Add up the free data blocks in each track,
	 * which means skip the free directory blocks
	 */
	for (track = 0; track < CBM_1541_TRACKS; ++track)
		if (track != CBM_1541_DIR_TRACK - 1)
			free_blocks += sec->bam[track].free_blocks;

	cbm_brelse(bh);

	return free_blocks;
}

/*
 * Count the number of free blocks on the 1571 disk, not including the
 * directory track
 */

inline long cbm_1571_free_blocks(struct super_block *sb)
{
	int track, free_blocks = 0;
	struct buffer_head *bh;
	struct cbm_1571_header_sector *sec;

	if (!(bh = cbm_bread(sb, CBM_SB(sb)->super_start))) {
		printk(KERN_WARNING "cbm_bread in cbm_1571_free_blocks failed\n");
		return -2;
	}
	sec = (struct cbm_1571_header_sector *) bh->b_data;

	/* Add up the free data blocks in each track, which means skip the free
	   directory blocks */
	for (track = 0; track < CBM_1571_TRACKS; ++track)
		if (track != CBM_1571_DIR_TRACK - 1)
			free_blocks += sec->bam[track].free_blocks;

	cbm_brelse(bh);

	return free_blocks;
}

/*
 * Count the number of free blocks on the 1581 disk, not including the
 * directory track
 *
 * NOTE: this doesn't count any free blocks in any partitions;
 *	 all blocks inside partitions are counted as used
 */

inline long cbm_1581_free_blocks(struct super_block *sb)
{
	int track, free_blocks = 0;
	struct buffer_head *bh;
	struct cbm_1581_bam_sector *sec;

	/*
	 * Total the first BAM sector's free blocks
	 */

	if (!(bh = cbm_bread(sb, CBM_SB(sb)->super_start + 1))) {
		printk(KERN_WARNING "cbm_bread in cbm_1581_free_blocks failed\n");
		return -2;
	}
	sec = (struct cbm_1581_bam_sector *) bh->b_data;

	/*
	 * Add up the free blocks in the first half of the disk track
	 */

	for (track = 0; track < CBM_1581_TRACKS / 2; ++track)
		if (track != CBM_1581_DIR_TRACK - 1)
			free_blocks += sec->bam[track].free_blocks;

	cbm_brelse(bh);

	/*
	 * Total the second BAM sector's free blocks
	 */

	if (!(bh = cbm_bread(sb, CBM_SB(sb)->super_start + 2))) {
		printk(KERN_WARNING "cbm_bread in cbm_1581_free_blocks failed\n");
		return -2;
	}
	sec = (struct cbm_1581_bam_sector *) bh->b_data;

	/*
	 * Add up the free blocks in the second half of the disk
	 */

	for (track = 0; track < CBM_1581_TRACKS / 2; ++track)
		free_blocks += sec->bam[track].free_blocks;

	cbm_brelse(bh);

	return free_blocks;
}

/*
 * Count the number of free data blocks on the disk
 */

long cbm_free_blocks(struct super_block *sb)
{
	switch (CBM_SB(sb)->fs_type) {
		case 1541: return cbm_1541_free_blocks(sb);
		case 1571: return cbm_1571_free_blocks(sb);
		case 1581:
		default: return cbm_1581_free_blocks(sb);
	}
}

/*
 * Returns nonzero if the given sector contains a valid 1541 header block
 */

int cbm_is_1541_header(struct cbm_1541_header_sector *header)
{
/* Old (non-working properly) version of detecting if header is valid
	return (header->disk_format == CBM_1541_DOS_FORMAT) &&
		(header->flag == 0) &&
		(header->space3[3] == CBM_END_NAME);	*/
	return (header->disk_format == CBM_1541_DOS_FORMAT) &&
	    (header->space3[3] == CBM_END_NAME) &&
	    (header->t == CBM_1541_DIR_TRACK);
}

/*
 * Returns nonzero if the given sector contains a valid 1571 header block
 */

int cbm_is_1571_header(struct cbm_1571_header_sector *header)
{
	return (header->disk_format == CBM_1571_DOS_FORMAT) &&
		(header->flag == 0x80) &&
		(header->space3[3] == CBM_END_NAME);
}

/*
 * Returns nonzero if the given sector contains a valid 1581 header block
 */

int cbm_is_1581_header(struct cbm_1581_header_sector *header)
{
	return (header->disk_format == CBM_1581_DOS_FORMAT) &&
		(header->flag == 0) &&
		(header->space3[0] == CBM_END_NAME) &&
		(header->space3[1] == CBM_END_NAME);
}

/*
 * File creation lock. This is system-wide to avoid deadlocks in rename.
 * (rename might deadlock before detecting cross-FS moves.)
 */

static struct wait_queue *creation_wait = NULL;
static creation_lock = 0;


void cbm_lock_creation(void)
{
	while (creation_lock)
		sleep_on(&creation_wait);
	creation_lock = 1;
}


void cbm_unlock_creation(void)
{
	creation_lock = 0;
	wake_up(&creation_wait);
}


void cbm_lock_bam(struct super_block *sb)
{
	while (CBM_SB(sb)->bam_lock)
		sleep_on(&CBM_SB(sb)->bam_wait);
	CBM_SB(sb)->bam_lock = 1;
}


void cbm_unlock_bam(struct super_block *sb)
{
	CBM_SB(sb)->bam_lock = 0;
	wake_up(&CBM_SB(sb)->bam_wait);
}


#if MAY_NEED_FOR_FUTURE_WRITE_SUPPORT
/*
 * cbm_add_cluster tries to allocate a new cluster and adds it to the file
 * represented by inode. The cluster is zero-initialized.
 */

int cbm_add_cluster(struct inode *inode)
{
	struct super_block *sb = inode->i_sb;
	int count, nr, limit, last, current, sector, last_sector;
	struct buffer_head *bh;
	int cluster_size = CBM_SB(inode->i_sb)->cluster_size;

	if (inode->i_ino == CBM_ROOT_INO)
		return -ENOSPC;
	if (!CBM_SB(inode->i_sb)->free_clusters)
		return -ENOSPC;
	cbm_lock_bam(inode->i_sb);
	limit = CBM_SB(inode->i_sb)->clusters;
	nr = limit; /* to keep GCC happy */
	for (count = 0; count < limit; count++) {
		nr = ((count + CBM_SB(inode->i_sb)->prev_free) % limit) + 2;
		if (cbm_bam_access(inode->i_sb, nr, -1) == 0)
			break;
	}

	printk(KERN_DEBUG "free cluster: %d\n", nr);
	CBM_SB(inode->i_sb)->prev_free = (count + CBM_SB(inode->i_sb)->
	    prev_free + 1) % limit;
	if (count >= limit) {
		CBM_SB(inode->i_sb)->free_clusters = 0;
		cbm_unlock_bam(inode->i_sb);
		return -ENOSPC;
	}
	cbm_bam_access(inode->i_sb, nr, CBM_SB(inode->i_sb)->bam_bits == 12 ?
	    0xff8 : 0xfff8);
	if (CBM_SB(inode->i_sb)->free_clusters != -1)
		CBM_SB(inode->i_sb)->free_clusters--;
	cbm_unlock_bam(inode->i_sb);
	printk(KERN_DEBUG "set to %x\n", cbm_bam_access(inode->i_sb, nr, -1));

	last = 0;
	if ((current = CBM_I(inode)->i_start) >= 0) {
		cbm_cache_lookup(inode, INT_MAX, &last, &current);
		while (current >= 0)
			if (!(current = cbm_bam_access(inode->i_sb,
			    last = current, -1))) {
				cbm_fs_panic(inode->i_sb, "File without EOF");
				return -ENOSPC;
			}
	}

	printk(KERN_DEBUG "last = %d\n", last);
	if (last) {
		cbm_bam_access(inode->i_sb, last, nr);
	} else {
		CBM_I(inode)->i_start = nr;
		inode->i_dirt = 1;
	}

	if (last)
		printk(KERN_DEBUG "next set to %d\n", cbm_bam_access(inode->i_sb, last, -1));

	sector = CBM_SB(inode->i_sb)->data_start + (nr - 2) * cluster_size;
	last_sector = sector + cluster_size;
	for (; sector < last_sector; sector++) {
		printk(KERN_DEBUG "zeroing sector %d\n", sector);
		if (!(bh = getblk(inode->i_dev, sector, SECTOR_SIZE))) {
			printk(KERN_WARNING "getblk failed\n");
		} else {
			memset(bh->b_data, 0, SECTOR_SIZE);
			cbm_set_uptodate(sb, bh, 1);
			mark_buffer_dirty(bh, 1);
			brelse(bh);
		}
	}
	inode->i_blocks += cluster_size;
	if (S_ISDIR(inode->i_mode)) {
		if (inode->i_size & (SECTOR_SIZE - 1)) {
			cbm_fs_panic(inode->i_sb, "Odd directory size");
			inode->i_size = (inode->i_size + SECTOR_SIZE) &
			    ~(SECTOR_SIZE - 1);
		}
		inode->i_size += SECTOR_SIZE * cluster_size;
		inode->i_dirt = 1;
		printk(KERN_DEBUG "size is %d now (%x)\n", inode->i_size, inode);
	}
	return 0;
}
#endif


/* Reads a block from the directory at offset *pos
 * Returns the inode number of the directory entry at offset *pos. If bh is
 * non-NULL, it is brelse'd before. Pos is incremented. The buffer header is
 * returned in *bh, and the pointer to the raw directory entry in *de
 *
 * This function appears to be like a normal `read' on a file
 *
 * Return: -1 == EOF or mapping error
 */

int cbm_get_entry(struct inode *dir, loff_t *pos, struct buffer_head **bh,
    struct cbm_dir_entry **de)
{
	struct super_block *sb = dir->i_sb;
	int sector, offset, entry;
	struct cbm_data_sector *sec;

	while (1) {
		offset = *pos;
		printk(KERN_DEBUG "cbm_get_entry offset %d\n", offset);

		/*
		 * Get the logical sector number of the directory block
		 */

		if ((sector = cbm_smap(dir, offset / CBM_DATA_SIZE)) == -2)
			return -1;	/* error */
		printk(KERN_DEBUG "cbm_get_entry sector %d %p\n", sector, *bh);
		if (sector == -1)
			return -1; /* beyond EOF */
		if (*bh)
			cbm_brelse(*bh);

		/*
		 * Increment pos to next dir entry.
		 * The first 7 entries in a block are the full size, but the
		 * last entry is 2 bytes shorter
		 */

		entry = CBM_OFS_DIRENT(offset);
		*pos += sizeof(struct cbm_dir_entry);
		if (entry == (CBM_DPS - 1))
			*pos -= 2;
		printk(KERN_DEBUG "cbm_get_entry entry #%d\n", entry);

		if (!(*bh = cbm_bread(sb, sector))) {
			printk(KERN_WARNING "Directory sread (sector %d) failed\n", sector);
			continue;
		}

		printk(KERN_DEBUG "cbm_get_entry apres sread\n");
		sec = (struct cbm_data_sector *) ((*bh)->b_data);
		*de = &((struct cbm_dir_entry *) sec->data)[entry];
		return CBM_MKINODE(sector, entry);
	}
}

/*
 * raw_scan_sector operates in four modes:
 *
 * name     number   ino      action
 * -------- -------- -------- -------------------------------------------------
 * !NULL    -        X        Find an entry with that name
 * NULL     !NULL    !NULL    Find an entry whose data starts at *number
 * NULL     !NULL    NULL     Count subdirectories in *number. (*)
 * NULL     NULL     !NULL    Find an empty entry
 *
 * (*) The return code should be ignored. It DOES NOT indicate success or
 *     failure. *number has to be initialized to zero.
 *
 * - = not used, X = a value is returned unless NULL
 *
 * If res_bh is non-NULL, the buffer is not deallocated but returned to the
 * caller on success. res_de is set accordingly.
 *
 * If cont is non-zero, raw_found continues with the entry after the one
 * res_bh/res_de point to.
 *
 * raw_scan_sector only looks for actual entries on disk, not pseudo "." and
 * ".." entries.
 */

static inline int raw_scan_sector(struct super_block *sb, int sector,
				  const char *name, int *number, int *ino,
				  struct buffer_head **res_bh,
				  struct cbm_dir_entry **res_de)
{
	struct buffer_head *bh;
	struct cbm_dir_entry *data;
	struct inode *inode;
	int entry, start, done = 0;

	if (!(bh = cbm_bread(sb, sector)))
		return -EIO;
	data = (struct cbm_dir_entry *)
			((struct cbm_data_sector *) bh->b_data)->data;

	/* loop through the sector, trying all directory entries if necessary */
	for (entry = 0; entry < CBM_DPS; entry++) {
		/* Search for name */

		if (name) {
		    done = !cbm_strcmp(data[entry].name, name) &&
				       !CBM_IS_FREE(data[entry].type);

		} else if (!ino) {

			/* Count subdirectories */

			if (IS_CBM(data[entry].type))
				(*number)++;

		} else if (number) {

			/* Search for start cluster */

			done = !CBM_IS_FREE(data[entry].type) &&
					    cbm_block(sb, data[entry].t,
						      data[entry].s) == *number;

		} else if ((done = CBM_IS_FREE(data[entry].type)) != 0) {
			/*
			 * Looks like we have a free entry -- get the inode
			 */
			inode = iget(sb, CBM_MKINODE(sector, entry));
			if (inode) {
				/*
				 * Directory slots of busy deleted files
				 * aren't available yet.
				 */
				done = !CBM_I(inode)->i_busy;
				iput(inode);
			}
		}

		/* Was the requested function completed? */

		if (done) {
			if (ino)
				*ino = CBM_MKINODE(sector, entry);
			start = cbm_block(sb, data[entry].t, data[entry].s);
			if (!res_bh) {
				cbm_brelse(bh);
			} else {
				*res_bh = bh;
				*res_de = &data[entry];
			}
			return start;	/* successful exit */
		}
	} /* for */

	/*
	 * Exit after counting entries or not finding matching entry
	 */

	cbm_brelse(bh);
	return -ENOENT;
}

/*
 * raw_scan performs raw_scan_sector on any (directory) sector until the
 * requested entry is found or the end of the directory is reached.
 *
 * NOTE: raw_scan must not be used on a directory that is is the process of
 *       being created. [relevant on 1581?-DF]
 *
 * start is first directory sector number to try, or 0 for the root dir
 */

static int raw_scan(struct super_block *sb, int start, const char *name,
		    int *number, int *ino, struct buffer_head **res_bh,
		    struct cbm_dir_entry **res_de)
{
	int sector;

	printk(KERN_DEBUG "raw_scan: start=%d\n", start);

	do {
		if ((sector = raw_scan_sector(sb, start, name, number, ino, res_bh, res_de)) >= 0)
			return sector;	/* Found the entry */

		/*
		 * Should probably look up cache here, but sector
		 * is practically guaranteed to be in memory,
		 * so this shouldn't be too slow
		 */
		if ((start = cbm_map_next(sb, start, NULL)) == -2) {
			cbm_fs_panic(sb, "cbm_map_next error");
			break;
		}
		printk(KERN_DEBUG "next start: %d\n", start);

	} while (start >= 0);
	return -ENOENT;
}

#if MAY_NEED_FOR_FUTURE_WRITE_SUPPORT
/*
 * cbm_parent_ino returns the inode number of the parent directory of dir.
 * File creation has to be deferred while cbm_parent_ino is running to
 * prevent renames.
 */

int cbm_parent_ino(struct inode *dir, int locked)
{
	static int zero = 0;
	int error, current, prev, nr;

	if (!S_ISDIR(dir->i_mode))
		panic("Non-directory fed to m_p_i");
	if (dir->i_ino == CBM_ROOT_INO)
		return dir->i_ino;
	if (!locked)
		cbm_lock_creation(); /* prevent renames */
	if ((current = raw_scan(dir->i_sb, CBM_I(dir)->i_start, CBM_DOTDOT,
	    &zero, NULL, NULL, NULL)) < 0) {
		if (!locked)
			cbm_unlock_creation();
		return current;
	}
	if (!current) {
		nr = CBM_ROOT_INO;
	} else {
		if ((prev = raw_scan(dir->i_sb, current, CBM_DOTDOT, &zero, NULL, NULL, NULL)) < 0) {
			if (!locked)
				cbm_unlock_creation();
			return prev;
		}
		if ((error = raw_scan(dir->i_sb, prev, NULL, &current, &nr, NULL, NULL)) < 0) {
			if (!locked)
				cbm_unlock_creation();
			return error;
		}
	}
	if (!locked)
		cbm_unlock_creation();
	return nr;
}
#endif

/*
 * cbm_subdirs counts the number of sub-directories of dir. It can be run
 * on directories being created [how does this fit in with 1581? -- DF].
 */

int cbm_subdirs(struct inode *dir)
{
	int count;

	count = 0;

	if (CBM_I(dir)->i_start < 0)
		return 0; /* in mkdir */

	(void) raw_scan(dir->i_sb, CBM_I(dir)->i_start,
		    NULL, &count, NULL, NULL, NULL);
	return count;
}


/*
 * Scans a directory for a given file (name points to its formatted name) or
 * for an empty directory slot (name is NULL). Returns an error code or zero.
 */

int cbm_scan(struct inode *dir, const char *name, struct buffer_head **res_bh,
	     struct cbm_dir_entry **res_de, int *ino)
{
	int res;

	res = raw_scan(dir->i_sb, CBM_I(dir)->i_start, name, NULL, ino,
		    res_bh, res_de);
	return res < 0 ? res : 0;
}


/*
 * Find the length of a file (or directory) by following the track & sector
 * chain starting at the beginning.
 *
 * Input: inode
 * Return: number of bytes, or -1 on error.
 */
long cbm_chain_len(struct inode *inode)
{
	int size, sector, block;

	printk(KERN_DEBUG "cbm_chain_len: inode %ld\n", inode->i_ino);
	block = 0;
	sector = CBM_I(inode)->i_start;		/* start at beginning of file */

	/* Try to find last sector of file */
	cbm_cache_lookup(inode, -1, &block, &sector);

	/* Read at least the last sector, maybe more before */
	while ((sector = cbm_map_next(inode->i_sb, sector, &size)) >= 0) {
		++block;
		/* should add to cache here */
		cbm_cache_add(inode, block, sector);
	}

	printk(KERN_DEBUG "cbm_chain_len: almost done block=%d size=%d sec=%d\n", block, size, sector);
	if (sector == -2) {
		cbm_fs_panic(inode->i_sb, "cbm_map_next error");
		return -1;
	}

	return block * CBM_DATA_SIZE + size;
}

/*
long chain_first(struct inode *inode)
*/
