#ifndef _LINUX_CBM_FS_H
#define _LINUX_CBM_FS_H

/*
 * Commodore 1541/1571/1581 filesystem constants/structures
 */
#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/types.h>

#include <asm/byteorder.h>

#define CBM_SUPER_MAGIC 0x1581	/* Linux filesystem superblock magic number */

#define CBM_BLOCK_SIZE 512	/* physical sector size on disk */
#define CBM_LBLOCK_SIZE 256	/* logical sector size */
#define CBM_DATA_SIZE 254	/* size in bytes of data portion of sector */
#define CBM_DPS	8		/* dir entries per CBM block */
#define CBM_DPB	(CBM_DPS)	/* dir entries per CBM block */
#define CBM_DPB_BITS 3		/* bits needed to hold CBM_DPB entries */
#define CBM_DPB_MASK 7		/* 2^CBM_DPB_BITS - 1 */

#define MAP_CACHE 26 /*259*/	/* entries in file location cache (20 bytes ea.)
				 * ought to be big enough to hold a 64k file
				 * for efficient random file access */

#define CBM_PREFETCH 32		/* size of prefetch buffer for file reading */
				/* must be an even number >= 2 */

/* C= 1541 specific constants */
#define CBM_1541_SUPER_LEN 1		/* Length of BAM in CBM blocks */
#define CBM_1541_DIR_LEN 18		/* Length of root directory
					 * in CBM blocks
					 */
#define CBM_1541_BLOCKS 683		/* Number of CBM blocks on standard
					 * 1541 disks (664 + dirtrack)
					 */
#define CBM_1541_TRACKS 35		/* Number of tracks on standard
					 * 1541 disks
					 */
#define CBM_1541_MAX_TRACKS 45		/* Number of tracks used when
					 * checking the bam
					 */
#define CBM_1541_DIR_TRACK 18		/* Directory track */
#define CBM_1541_DOS_FORMAT 'A'		/* Magic number */
#define CBM_1541_SUPER_LOC cbm_1541_block(CBM_1541_DIR_TRACK, 0)
					/* Start of superblock */

/* C= 1571 specific constants */
#define CBM_1571_SUPER_LEN 1		/* Length of BAM in CBM blocks */
#define CBM_1571_DIR_LEN 18		/* Length of root directory
					 * in CBM blocks */
#define CBM_1571_BLOCKS 1347		/* Number of CBM blocks on standard
					 * 1571 disks (1328 + dirtrack) */
#define CBM_1571_TRACKS 70		/* Number of tracks on standard
					 * 1571 disks */
#define CBM_1571_DIR_TRACK 18		/* Directory track */
#define CBM_1571_DOS_FORMAT 'A'		/* Magic number */
#define CBM_1571_SUPER_LOC cbm_1571_block(CBM_1571_DIR_TRACK, 0)
					/* Start of superblock */

/* C= 1581 specific constants */
#define CBM_1581_SUPER_LEN 3		/* Length of superblock in CBM blocks */
#define CBM_1581_DIR_LEN 37		/* Length of root directory
					 * in CBM blocks */
#define CBM_1581_BLOCKS 3200		/* Number of CBM blocks on standard
					 * 1581 disks */
#define CBM_1581_TRACKS 80		/* Number of tracks on standard
					 * 1581 disks */
#define CBM_1581_DIR_TRACK 40		/* Directory track */
#define CBM_1581_DOS_FORMAT 'D'		/* Magic number */
#define CBM_1581_SUPER_LOC cbm_1581_block(CBM_1581_DIR_TRACK, 0)
					/* Start of superblock */

/* File type mask bits (or'ed with cbm_file_types) */
enum cbm_file_attr {
	CBM_LOCKED = 0x40,		/* File locked */
	CBM_CLOSED = 0x80		/* File closed bit */
};

/* File types */
enum cbm_file_types {
	CBM_DEL = 0,
	CBM_SEQ = 1,
	CBM_PRG = 2,
	CBM_USR = 3,
	CBM_REL = 4,
	CBM_CBM = 5,			/* 1581/C65 only */
	CBM_DJJ = 6,			/* C65 only */
	CBM_FAB = 7,			/* C65 only */

	CBM_TYPE_MASK = 0x07		/* Mask to extract file type */
};

#define CBM_NAME 16			/* Maximum name length */
#define CBM_END_NAME 0xA0		/* Indicates end of file name */

/* ioctl commands */

#define CBM_IOC_GETFTYPE	_IOR(0x15, 1, long) /* Get file type */
#define CBM_IOC_SETFTYPE	_IOW(0x15, 2, long) /* Set file type */
#define CBM_IOC_GETMEDIA	_IOR(0x15, 3, long) /* Get disk type */
#define CBM_IOC_GETRECSIZE	_IOR(0x15, 4, long) /* Get REL record size */

/*
 * Conversion from and to little-endian byte order. (no-op on i386/i486)
 *
 * Naming: Ca_b_c, where a: F = from, T = to, b: LE = little-endian,
 * BE = big-endian, c: W = word (16 bits), L = longword (32 bits)
 */

#define CF_LE_W(v) le16_to_cpu(v)
#define CF_LE_L(v) le32_to_cpu(v)
#define CT_LE_W(v) cpu_to_le16(v)
#define CT_LE_L(v) cpu_to_le32(v)

/* Access macros */

#define CBM_IS_FREE(n) ((n) == 0)	/* cbm_dir_entry.type
					 * shows entry is free to use */
#define IS_REL(t) (((t) & CBM_TYPE_MASK) == CBM_REL)
#define IS_SEQ(t) (((t) & CBM_TYPE_MASK) == CBM_SEQ)
#define IS_CBM(t) (((t) & CBM_TYPE_MASK) == CBM_CBM)

#define CBM_VALID_MODE (S_IFREG | S_IFDIR | S_IRWXU | S_IRWXG | S_IRWXO)
	/* Valid file mode bits -- should S_IFDIR be included? */

#define CBM_SB(s) (&((s)->u.cbm_sb))
#define CBM_I(i) (&((i)->u.cbm_i))


/* File location cache; stores location on disk for blocks in files */
struct map_cache {
	int device;			/* Device number. 0 means unused. */
	ino_t ino;			/* Inode number. */
	int file_block;			/* Block number in the file. */
	int disk_sector;		/* Logical sector on device. */
	struct map_cache *next;		/* Next cache entry */
};

/*
 * The following structures must all be packed onto byte boundaries.
 * This is how they happen to end up in i386 GCC 2.5.8. Others
 * architectures may need to use __attribute__ ((packed)) on all entries
 */

/* Contents of one 1541 BAM entry */
struct cbm_1541_bam_entry {
	__u8 free_blocks;		/* Free blocks in this track */
	__u8 bit_map[3];		/* Free block bit map */
};

/* Contents of one 1571 BAM entry */
struct cbm_1571_bam_entry {
	__u8 free_blocks;		/* Free blocks in this track */
	__u8 bit_map[3];		/* Free block bit map */
};

/* Contents of one 1581 BAM entry */
struct cbm_1581_bam_entry {
	__u8 free_blocks;		/* Free blocks in this track */
	__u8 bit_map[5];		/* Free block bit map */
};

/* Contents of dir header sector */
struct cbm_1541_header_sector {
	__u8 t, s;			/* Track & Sector of directory */
	__u8 disk_format;		/* 'A' for 4040 disk format */
	__u8 flag;			/* Null flag for future use */
	struct cbm_1541_bam_entry bam[CBM_1541_TRACKS];	/* Block availability map */
	__u8 disk_name[16];		/* Disk name */
	__u8 space1[2];			/* Shifted spaces */
	__u8 disk_id[2];		/* Disk id code */
	__u8 space2;			/* Shifted space */
	__u8 dos_version[2];		/* '2A' for 1541 */
	__u8 space3[4];			/* Shifted spaces */
	__u8 filler[9];			/* Unused */
	__u8 string[12];		/* 1541 manual says this sometimes used */
};

/* Contents of dir header sector */
struct cbm_1571_header_sector {
	__u8 t, s;			/* Track & Sector of directory */
	__u8 disk_format;		/* 'A' for 4040 disk format */
	__u8 flag;			/* flag 0x80 - differs from 1541 format */
	struct cbm_1541_bam_entry bam[CBM_1541_TRACKS];	/* Block availability map */
	__u8 disk_name[16];		/* Disk name */
	__u8 space1[2];			/* Shifted spaces */
	__u8 disk_id[2];		/* Disk id code */
	__u8 space2;			/* Shifted space */
	__u8 dos_version[2];		/* '2A' for 1571 */
	__u8 space3[4];			/* Shifted spaces */
	__u8 filler[9];			/* Unused */
	__u8 string[12];		/* 1571 manual says this sometimes used */
};

/* Contents of each of the 2 1581 BAM sectors */
struct cbm_1581_bam_sector {
	__u8 t, s;			/* Track & Sector of next BAM */
	__u8 disk_format;		/* 'D' for 1581 disk format */
	__u8 disk_format_cpl;		/* Complemented disk_format value */
	__u8 disk_id[2];		/* Disk id code */
	__u8 io_flag;			/* Bit 7: verify;
					 * Bit 6: check header CRC */
	__u8 auto_loader_flag;		/* 0 or ??? */
	__u8 reserved[8];
	struct cbm_1581_bam_entry bam[CBM_1581_TRACKS];	/* 80 tracks worth of BAM entries */
};

struct cbm_1581_header_sector {
	__u8 t, s;			/* Track & Sector of directory */
	__u8 disk_format;		/* 'D' for 1581 disk format */
	__u8 flag;			/* Null flag for future use */
	__u8 disk_name[16];		/* Disk name */
	__u8 space1[2];			/* Shifted spaces */
	__u8 disk_id[2];		/* Disk id code */
	__u8 space2;			/* Shifted space */
	__u8 dos_version[2];		/* '3D' for 1581 */
	__u8 space3[2];			/* Shifted spaces */
};

/* CBM data sector */
struct cbm_data_sector {
	__u8 t, s;			/* Next Track & Sector of file */
	__u8 data[CBM_DATA_SIZE];	/* File data */
};


/* Contents of single directory entry */
struct cbm_dir_entry {
	__u8 type;			/* File type & attr */
	__u8 t, s;			/* First track & sector of file */
	__u8 name[16];			/* File name */
	__u8 ss_t, ss_s;		/* First side sector track & sector */
	__u8 rec_size;			/* Relative file record size */
	__u8 unused[4];
	__u8 open_t, open_s;		/* Track & Sector of replacement
					 * file for OPEN@ */
	__u16 blocks;			/* Size of file (in blocks) */
	__u8 padding[2];		/* Doesn't exist for final entry
					 * in CBM block */
};

/* Convert CBM attribute bits and a mask to the UNIX mode. */
#define CBM_MKMODE(a, m) ((m) & ((((a) & CBM_LOCKED) ? S_IXUGO : S_IWUGO | S_IXUGO)\
				 | (((a) & CBM_CLOSED) ? S_IRUGO : 0)))

/* Convert the UNIX mode to the CBM attribute bits. */
#define CBM_MKATTR(m) (((m) & S_IWUGO) ? CBM_CLOSED : CBM_LOCKED | CBM_CLOSED)

/* Convert a CBM block and directory entry number to inode number */
#define CBM_MKINODE(b, e) (((b) << CBM_DPB_BITS) | (e))

/* Convert an inode number to CBM block */
#define CBM_INOBLOCK(i) ((i) >> CBM_DPB_BITS)

/* Convert an inode number to directory entry number */
#define CBM_INODIRENT(i) ((i) & CBM_DPB_MASK)

/* Convert a directory file offset into a directory entry number */
#define CBM_OFS_DIRENT(o) (((o) % CBM_DATA_SIZE) / (CBM_LBLOCK_SIZE / CBM_DPS))

/* Get the root inode number for a superblock */
#define CBM_ROOT_INODE(sb) (CBM_MKINODE(CBM_SB(sb)->super_start, 0))

#ifdef __KERNEL__

/* misc.c */

extern void cbm_fs_panic(struct super_block *sb, char *msg);
extern unsigned long cbm_1541_block(int track, int sector);
extern unsigned long cbm_1571_block(int track, int sector);
extern unsigned long cbm_1581_block(int track, int sector);
extern unsigned long cbm_block(struct super_block *sb, int track, int sector);
extern void cbm_1541_ts(unsigned long block, int *track, int *sector);
extern void cbm_1571_ts(unsigned long block, int *track, int *sector);
extern void cbm_1581_ts(unsigned long block, int *track, int *sector);
extern long cbm_free_blocks(struct super_block *sb);
extern int cbm_is_1541_header(struct cbm_1541_header_sector *header);
extern int cbm_is_1571_header(struct cbm_1571_header_sector *header);
extern int cbm_is_1581_header(struct cbm_1581_header_sector *header);
extern void cbm_lock_creation(void);
extern void cbm_unlock_creation(void);
extern void cbm_lock_bam(struct super_block *sb);
extern void cbm_unlock_bam(struct super_block *sb);
extern int cbm_add_cluster(struct inode *inode);
extern int cbm_get_entry(struct inode *dir, loff_t *pos,
			 struct buffer_head **bh, struct cbm_dir_entry **de);
extern int cbm_parent_ino(struct inode *dir, int locked);
extern int cbm_subdirs(struct inode *dir);
extern int cbm_scan(struct inode *dir, const char *name,
		    struct buffer_head **res_bh,
		    struct cbm_dir_entry **res_de, int *ino);
extern long cbm_chain_len(struct inode *inode);

/* map.c */

extern int cbm_bam_access(struct super_block *sb, int nr, int new_value);
extern int cbm_map_next(struct super_block *sb, int nr, int *size);
extern void cbm_cache_init(void);
extern void cbm_cache_lookup(struct inode *inode, int block, int *file_blk,
			     int *disk_sec);
extern void cbm_cache_add(struct inode *inode, int f_clu, int d_clu);
extern void cbm_cache_inval_inode(struct inode *inode);
extern void cbm_cache_inval_dev(int device);
extern int cbm_get_sector(struct inode *inode, int sector);
extern int cbm_smap(struct inode *inode, int sector);
extern int cbm_bam_free(struct inode *inode, int skip);

/* namei.c */

extern struct dentry * cbm_lookup(struct inode *dir, struct dentry *dentry);
extern int cbm_create(struct inode *dir, const char *name, int len, int mode,
		      struct inode **result);
extern int cbm_mkdir(struct inode *dir, const char *name, int len, int mode);
extern int cbm_rmdir(struct inode *dir, const char *name, int len);
extern int cbm_unlink(struct inode *dir, const char *name, int len);
extern int cbm_rename(struct inode *old_dir, const char *old_name, int old_len,
		      struct inode *new_dir, const char *new_name, int new_len);

/* inode.c */

extern void cbm_put_inode(struct inode *inode);
extern void cbm_put_super(struct super_block *sb);
extern struct super_block *cbm_read_super(struct super_block *sb,
					  void *data, int silent);
extern int cbm_statfs(struct super_block *sb, struct statfs *buf, int unknown);
extern int cbm_bmap(struct inode *inode, int block);
extern void cbm_read_inode(struct inode *inode);
extern void cbm_write_inode(struct inode *inode);
extern int cbm_notify_change(struct dentry *, struct iattr *attr);
extern int init_cbm_fs(void);

/* dir.c */

extern struct inode_operations cbm_dir_inode_operations;

extern int cbm_readdir(struct file *filp, void *unknown, filldir_t filldir);

/* file.c */

extern struct inode_operations cbm_file_inode_operations;

extern ssize_t cbm_file_read(struct file *filp, char *buf, size_t count, loff_t *ppos);
extern ssize_t cbm_file_write(struct inode *filp, const char *buf, size_t count, loff_t *ppos);
extern void cbm_truncate(struct inode *inode);

/* mmap.c -- not existing! */
extern int cbm_mmap(struct inode *, struct file *, struct vm_area_struct *);

/* ioctl.c */
extern int cbm_ioctl(struct inode *inode, struct file *filp, unsigned int cmd,
		      unsigned long arg);

#endif /* __KERNEL__ */

#endif
