function v = npv (r, p, i)
  
# usage:  npv (r, p [, i])
#
# Returns the net present value of a series of irregular (i.e., not
# necessarily identical) payments p which occur at the ends of n
# consecutive periods.  r specifies the one-period interest rates
# and can either be a scalar (constant rates) or a vector of the
# same length as p. 
#
# With the optional scalar argument i, one can specify an initial
# investment.  
#
# Note that rates are not specified in percent, i.e., one has to 
# write 0.05 rather than 5 %.
#
# See also:  irr;  pv.
  
# Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1995/01/06
# Copyright Dept of Statistics and Probability Theory TU Wien

  if ((nargin < 2) || (nargin > 3))
    usage ("npv (r, p [, i]");
  endif
  
  if !(is_scalar (p) || is_vector (p))
    error ("npv:  p has to be a vector");
  else
    n = length (p);
    p = reshape (p, 1, n);
  endif
  
  if any (any (r <= -1))
    error ("npv:  all interest rates must be > -1");
  endif
  if is_scalar (r)
    d = 1 ./ (1 + r) .^ (0 : n);
  elseif (is_vector (r) && (length (r) == n))
    d = [1, 1 ./ cumprod (reshape (1 + r, 1, n))];
  else
    error (["npv:  r has be a scalar ", ...
	"or a vector of the same length as p"]);
  endif

  if (nargin == 3)
    if !is_scalar (i)
      error ("npv:  I_0 must be a scalar");
    endif
  else
    i = 0;
  endif
  
  p = [i, p];
  v = sum (d .* p);
  
endfunction