function v = pv (r, n, p, l, m)
  
# usage:  pv (r, n, p [, l] [, method])
#
# Returns the present value of an investment that will pay off p for n
# consecutive periods, assuming an interest r. 
#
# With the optional scalar argument l, one can specify an additional
# lump-sum payment made at the end of n periods.
#
# With the optional string argument `method', one can specify whether
# payments are made at the end ("e", default) or at the beginning
# ("b") of each period. 
#
# Note that the rate r is not specified in percent, i.e., one has to
# write 0.05 rather than 5 %.
#
# See also:  pmt, nper, rate;  npv.
  
# Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1995/01/06
# Copyright Dept of Statistics and Probability Theory TU Wien

  if ((nargin < 3) || (nargin > 5))
    usage ("pv (r, n, p [, l] [, method])");
  endif
  
  if !(is_scalar (r) && (r > -1))
    error ("pv:  r must be a scalar > -1");
  elseif !(is_scalar (n) && (n > 0))
    error ("pv:  n must be a positive scalar");
  elseif !is_scalar (p)
    error ("pv:  p must be a scalar.");
  endif
  
  if (r <> 0)
    v = p * (1 - (1 + r)^(-n)) / r;
  else
    v = p * n;
  endif
  
  if (nargin > 3)
    if (nargin == 5)
      if !isstr (m)
	error ("pv:  `method' must be a string");
      endif
    elseif isstr (l)
      m = l;
      l = 0;
    else
      m = "e";
    endif
    if strcmp (m, "b")
      v = v * (1 + r);
    endif
    if is_scalar (l)
      v = v + pvl (r, n, l);
    else
      error ("pv:  l must be a scalar");
    endif
  endif
  
endfunction
      
