function [FGN, Hist] = RMDtraffic (M, a, H, n, T)

# Generate a random column vector of numbers of length 2^n representing a
# trace of the instantaneous traffic generated by a self similar source with
# average traffic rate M, peakedness a and Hurst parameter H during a time
# interval T. 
#
# M, a and T should be scalars: M,a,T>0, 0<H<1, n > 0.  T is optional,
# defaulting to 2^n.  M, a and H are only target values for the generation.
# The trace will have generally different statistics, in particular the
# average will generally be higher than M due to the truncation of the
# negative samples.
#
# The whole output vector is considered to be of time length T, so the
#   values contained are the traffic intensities in time intervals of
#   length T*2^-n. 
# M is the value of the traffic rate.
# a is the peakedness factor, i.e. the ratio of the variance to the average
# of the traffic per unit time interval. 
#
# Lau, Erramilli, Wang, Willinger, "Self-similar Traffic Generation: The
# random Midpoint Displacement Algorithm and Its Properties", ICC '95,
# pp. 466-472.

# Put in the public domain by Francesco Potorti` <pot@cnuce.cnr.it>
# Mon Aug 28 16:57:10 MET 1995

  savedist = rand ("dist");

  if (nargin < 4 || nargin > 5)
    usage ("RMDtraffic (M, a, H, n, T)");
  endif
  if (!is_scalar (M) || M <= 0)
    error ("M should be a scalar greater than 0\n");
  endif
  if (!is_scalar (a) || a < 0)
    error ("a should be a scalar be not less than 0\n");
  endif
  if (!is_scalar (H) || H < 0 || H > 1)  
    error ("H should be a scalar in the interval [0;1]\n");
  endif
  if (!is_scalar (n) || n <= 0 || round (n) != n)
    error ("n should be an integer greater than 0\n");
  endif
  if (nargin < 5)
    T = 2^n;
  else
    if (!is_scalar (T) || T <= 0)
      error ("T should be a scalar greater than 0\n");
    endif
  endif
  hist = (nargout > 1);			# history enabled

  N = 2^n;				# length of the result vector
  rand ("normal");			# create gaussian random numbers

  # Create a fractional brownian motion process approximation using the
  # random midpoint displacement algorithm.
  FBM(1,1) = 0;				# init the FBM trace: first point is 0
  FBM(N+1,1) = 0;			# last point is 0
  FBM(2:N) = rand(N-1, 1);		# middle points: unscaled displacements
  scale = sqrt (1 - 2^(2*H-2)) * T^H;	# scaling factor for step 0
  if (hist) Hist(:,1) = [2^n;scale;FBM]; endif

  for i = 1:n				# step n times
    d = 2^(n-i);			# distance between successive points
    scale = scale * 2^-H;		# scaling factor for step i
    range = [d+1:2*d:N-d+1];		# indices of points to change in step i
    FBM(range) = 0.5 * (FBM(range-d) + FBM(range+d)) + FBM(range)*scale;
    if (hist) Hist(:,i+1) = [d;scale;FBM]; endif
  endfor

  # Now differentiate to get the fractional gaussian noise process, scale it
  # to get the requested mean and peakedness, and truncate it to eliminate
  # the negative traffic spikes. 
  FGN = FBM(2:N+1) - FBM(1:N);
  if (hist) Hist(:,n+2) = [0;0;0;FGN]; endif
  FGN = M*T/N + sqrt(a*M) * FGN;
  FGN = max (0, FGN);
  
  rand (savedist);
endfunction








