function y = betai(a, b, x)
  
  # usage:  betai(a, b, x)
  #
  # Returns the incomplete beta function
  #   betai (a, b, x) = BETA(a,b)^(-1) INT_0^x t^(a-1) (1-t)^(b-1) dt.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on Aug 2, 1994.
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on Oct 22, 1994.
  # Copyright Dept of Probability Theory and Statistics TU Wien.

  # Computation is based on the series expansion
  #   betai(a, b, x) 
  #     = \frac{x^a}{B(a, b)}
  #         \sum_{l=0}^\infty \frac{(1-b)\cdots(l-b)}{a+l} \frac{x^l}{l!}
  # for x <= 1/2.  For x > 1/2, betai(a, b, x) = 1 - betai(b, a, 1-x).

  if (nargin <> 3)
    error("usage:  betai (a, b, x)");
  endif

  [retval, x, a, b] = common_size (x, a, b);
  if (retval > 0)
    error ("betai:  a, b and x must be of common size or scalar");
  endif
  
  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  a = reshape (a, 1, s);
  b = reshape (b, 1, s);
  y = zeros (1, s);

  k = find ((x < 0) | (x > 1) | !(a > 0) | !(b > 0) | isnan (x));
  if any (k)
    y(k) = NaN * ones (1, length (k));
  endif

  k = find ((x == 1) & (a > 0) & (b > 0));
  if any (k)
    y(k) = ones (1, length (k));
  endif

  # now do the series computations.  
  # The error when truncating at term L is always less than 2^(-L),
  # hence the following choice of L. 

  L = ceil (-log (eps) / log (2));
  h = ones (L, 1);
  
  k = find ((x >= 0) & (x <= 1/2) & (a > 0) & (b > 0));
  if any (k)
    l   = (1 : L)' * ones (1, length (k));
    tmp = cumprod ((1 - (h * b(k)) ./ l) .* (h * x(k))) ...
	./ ((h * a(k)) + l);
    y(k) = exp (a(k) .* log (x(k))) .* (1 ./ a(k) + sum (tmp)) ...
	./ beta (a(k), b(k));
  endif
  
  k = find ((x > 1/2) & (x < 1) & (a > 0) & (b > 0));
  if any (k)
    l   = (1 : L)' * ones (1, length (k));
    tmp = cumprod ((1 - (h * a(k)) ./ l) .* (h * (1 - x(k)))) ...
	./ (h * b(k) + l);
    y(k) = 1 - exp (b(k) .* log (1 - x(k))) ...
	.* (1 ./ b(k) + sum (tmp)) ./ beta (a(k), b(k));
  endif

  y = reshape (y, r, c);
  
endfunction





