function psi = digamma(z,method,debug)
%
%  psi = digamma(z)   ... Digamma-Function for real argument z.
%
%  digamma(z) = d/dz log(gamma(z)) = gamma'(z)/gamma(z)
%
%  if 'z' is a matrix, then the digamma-function is evaluated for 
%  each element. Results may be inaccurate for real arguments < 10 
%  which are neither integers nor half-integers.
%
%  psi = digamma(z,method)
%
%  possible values for optional argument 'method': 
%    method = 1           : quick asymptotic series expansion (approximate)
%    method = 2           : finite recursion for integer values (exact)
%    method = 3           : finite recursion for half-integer values (exact)
%    method = 4 (default) : automatic selection of 1,2 or 3 for individual
%                           elements in z whichever is appropriate.
%
%  see also:  trigamma, gamma, gammaln, gammainc, specfun

%  reference: Abramowitz & Stegun, "Handbook of Mathematical Functions"
%             Chapter "Gamma Function and Related Functions" :
%  implemented by: Christoph Mecklenbraeuker
%             (email: cfm@sth.ruhr-uni-bochum.de), July 1, 1995.


dim = size(z); 				% save original matrix dimension
z = reshape(z,dim(1)*dim(2),1); 	% make a column vector
I1 = ones(length(z),1); 		% auxiliary vector of ones

if(nargin==1) 
  method=4; debug=0; 
elseif(nargin==2) 
  debug=0; 
end;

if(debug == 1) 				% if debug==1: track recursion
  [m,n] = size(z);
  fprintf(1,'digamma: method = %d, size(z)=[%d %d],\t min(z)=%f, max(z)=%f\n',...
      method,m,n,min(min(z)),max(max(z)));
end;


if(method==1) 				% use 8th order asymptotic expansion 
  if(any(z<1))
    fprintf(1,'Warning: some elements in argument of "digamma(z,1)" are < 1\n');
    fprintf(1,'minimal argument = %g: digamma-result is inaccurate!\n',min(min(z)));
  end
  % calculate powers of 1/z :
  w1 = 1./z; w2 = w1.*w1; w4 = w2.*w2; w6 = w2.*w4; w8 = w4.*w4;
  % generate coefficients of expansion: matrix with constant columns
  a = [ -I1/2   -I1/12   I1/120  -I1/252  I1/240 ];
  % make vector of powers of 1/z:
  w = [  w1      w2       w4       w6       w8   ];
  % calculate expansion by summing the ROWS of (a .* w) :
  psi = log(z) + sum((a.*w).').'; 	
elseif(method==2)
  zmax = max(max(floor(z)));
  psitab = zeros(zmax,1);
  psitab(1) = -0.5772156649015328606;
  for n=1:zmax-1;
    psitab(n+1) = psitab(n) + 1/n; 	% generate lookup table
  end;
  psi = psitab(z);
elseif(method==3)
  zmax = max(max(floor(z)));
  psitab = zeros(zmax+1,1);
  psitab(1) = -0.5772156649015328606 - 2*log(2);  % = psi(1/2)
  for n=1:zmax;
    psitab(n+1) = psitab(n) + 2/(2*n-1); % generate lookup table
  end;
  psi = psitab(z+0.5);
elseif(method==4) 			% decide here which method to use
  Less0 = find(z<0); 			% negative arguments evaluated by reflexion formula
  Less1 = find(z>0 & z<1); 		% values between 0 and 1.
  fraction = rem(z,1); 			% fractional part of arguments
  f2 = rem(2*fraction,1);
  Integers = find(fraction==0 & z>0); 	% Index set of positive integer arguments
  NegInts  = find(fraction==0 & z<=0); 	% Index set of positive integer arguments
  HalfInts = find(abs(fraction-0.5)<1e-7 & z>0); % Index set of positive half-integers
  Reals    = find(f2>1e-7 & z>1); 	% Index set of all other arguments > 1
  if(~isempty(Reals)) psi(Reals)    = digamma(z(Reals),1,debug);    end;
  if(~isempty(Less1)) psi(Less1)    = digamma(z(Less1)+2,1,debug) - ...
	1./z(Less1)-1./(z(Less1)+1);end;
  % reflexion formula:
  if(~isempty(Less0)) psi(Less0) = digamma(1-z(Less0),1,debug) - pi./tan(pi*z(Less0)); end;
  if(~isempty(Integers)) psi(Integers) = digamma(z(Integers),2,debug); end;
  if(~isempty(HalfInts)) psi(HalfInts) = digamma(z(HalfInts),3,debug); end;
  if(~isempty(NegInts))  psi(NegInts)  = Inf * NegInts; end;
end

psi = reshape(psi,dim(1),dim(2));

return;
