function [jn,yn] = sphbess(n,X,ToDo);
% [jn,yn]=SPHBESS(n,X,ToDo):
% Returns the spherical Bessel and Newmann functions for complex arguments.
%
% n    - Maximal order, integer scalar.
% x    - Real or complex argument vector.
% ToDo - Optional string of command characters (case not important):
%        'S' - return orders -n:n rather than 0:n.
%        'Y' - if nargout<2, return y_n(x) instead of j_n(x).
%        'L' - return only the highest computed order.
% -----------------------------------------
% jn,yn - Spherical Bessel functions of the 1st and (optionally) 2nd kind,
%         length(X) x (n+1) or length(X) x (2n+1) matrices containing orders
%         0:n for n>=0, n:0 for n<0, or -|n|:|n| for 'S'.

% Algorithm:
% ----------
% sphbess calls jybess(n+1/2,X), and multiplies by sqrt(pi/(2X)). The point
% X=0 is taken care of separately.

if nargin<1
  help sphbess
  return
end
if nargin<2, error('Not enough input arguments!'), end
if max(size(n))>1 | any(ceil(n)~=n),
  error('"n" must be an integer scalar!')
end
if nargin<3, ToDo=''; end
if ~isstr(ToDo), error('"ToDo" must be a string!'), end

NewToDo=''; do_J=1; do_Y=(nargout>1); sym=0;
if any(ToDo=='Y') | any(ToDo=='y')
  do_J=0; do_Y=1; NewToDo=[NewToDo, 'Y'];
end
OnlyLast=any(ToDo=='L') | any(ToDo=='l');
if any(ToDo=='S') | any(ToDo=='s')
  if n~=0
    sym=1; NewToDo=[NewToDo, 'S']; n=abs(n);
  end
  Orders=-n:n;
else
  Orders=0:n;
end
if n<0
  n_negative=1; NewToDo=[NewToDo, 'S']; n=-n;
end

X=X(:);

%           Take care of X=0
z=(X==0); X=X+z; z=find(z);

%                  Calculate Bessel functions
if do_J & do_Y
  [Jn,Yn]=jybess(n+1/2,X,NewToDo);
elseif do_J
  Jn=jybess(n+1/2,X,NewToDo);
else
  Yn=jybess(n+1/2,X,NewToDo);
end

fac=sqrt(pi./(2*X));
ZeroOrder=find(Orders==0);
IntOrder=find(Orders~=0);
if do_J
  if n_negative, Jn=Jn(:,2:2+n); end
  if sym, Jn=Jn(:,2:2*n+2); end
  [r,c]=size(Jn);
  jn=Jn.*fac(:,ones(1,c));
  if ~isempty(z)
    jn(z,ZeroOrder)=ones(size(jn(z,ZeroOrder)));
    if ~isempty(IntOrder), jn(z,IntOrder)=zeros(size(jn(z,IntOrder))); end
  end
  if OnlyLast
    if n_negative
      jn=jn(:,1);
    else
      jn=jn(:,c);
    end
  end
end
if do_Y
  if n_negative, Yn=Yn(:,2:2+n); end
  if sym, Yn=Yn(:,2:2*n+2); end
  [r,c]=size(Jn);
  yn=Yn.*fac(:,ones(1,c));
  if ~isempty(z), yn(z,:)=yn(z,:)-Inf; end
  if OnlyLast
    if n_negative
      yn=yn(:,1);
    else
      yn=yn(:,c);
    end
  end
  if nargout<2, jn=yn; end
end

  
