function [pval, chisq, df] = bartlett_test (x1, ...)

  # usage:  [pval, chisq, df] = bartlett_test (x1, ...)
  #
  # Performs a Bartlett test for the homogeneity of variances in the
  # data vectors x1, x2, ..., xk, where k > 1.  
  # Under the null of equal variances, the test statistic chisq
  # approximately ollows a chi-square distribution with df degrees of
  # freedom;  pval is the p-value (1 minus the CDF of this
  # distribution at chisq) of the test. 
  #
  # If no output argument is given, the p-value is displayed.

  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on Jul 31, 1994
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on Sep 9, 1994 
  # Copyright Dept of Probability Theory and Statistics TU Wien
  
  k = nargin;
  if (k < 2)
    error ("usage:  [pval, chisq, df] = bartlett_test (x1, ...)");
  endif

  f    = zeros (k, 1);
  v    = zeros (k, 1);
  if !(is_vector (x1))
    error ("bartlett_test:  all arguments must be vectors");
  endif
  f(1) = length (x1) - 1;
  v(1) = var (x1);
  va_start;
  for i = 2 : k;
    x = va_arg ();
    if !(is_vector (x))
      error ("bartlett_test:  all arguments must be vectors");
    endif
    f(i) = length (x) - 1;
    v(i) = var (x);
  endfor

  f_tot = sum (f);
  v_tot = sum (f .* v) / f_tot;
  c     = 1 + (sum (1 ./ f) - 1 / f_tot) / (3 * (k - 1));
  chisq = (f_tot * log (v_tot) - sum (f .* log (v))) / c;
  df    = k;
  pval  = 1 - chisquare_cdf (chisq, df);
  
  if (nargout == 0)
    printf("  pval:  %g\n", pval);
  endif

endfunction


