function rnd = beta_rnd (a, b, r, c)
  #
  # usage:  beta_rnd (a, b [, r, c])
  #
  # beta_rnd (a, b) returns a matrix of random samples from the Beta
  # distribution with parameters a and b.  The size of the matrix is
  # the common size of a and b.
  #
  # beta_rnd (a, b, r, c) returns an r by c matrix of random samples
  # from the Beta distribution with parameters a and b.  Both a and b 
  # must be scalar or of size r by c.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/11/01
  # Copyright Dept of Probability Theory and Statistics TU Wien
  
  if (nargin == 4)
    if !(is_scalar (r) && (r > 0) && (r == round (r)))
      error ("beta_rnd:  r must be a positive integer");
    endif
    if !(is_scalar (c) && (c > 0) && (c == round (c)))
      error ("beta_rnd:  c must be a positive integer");
    endif
    [retval, a, b] = common_size (a, b, zeros (r, c));
    if (retval > 0)
      error (["beta_rnd:  ", ...
	  "a and b must be scalar or of size ", ...
	  sprintf ("%d by %d", r, c)]);
    endif
  elseif (nargin == 2)
    [retval, a, b] = common_size (a, b);
    if (retval > 0)
      error (["beta_rnd:  ", ...
	  "a and b must be of common size or scalar"]);
    endif
  else
    error ("usage:  beta_rnd (a, b [, r, c])");
  endif

  [r, c] = size (a);
  s = r * c;
  a = reshape (a, 1, s);
  b = reshape (b, 1, s);
  rnd = zeros (1, s);
  
  k = find (!(a > 0) | !(a < Inf) | !(b > 0) | !(b < Inf));
  if any (k)
    rnd(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((a > 0) & (a < Inf) & (b > 0) & (b < Inf));
  if any (k)
    current_distribution = rand ("dist");
    rand ("uniform");
    rnd(k) = beta_inv (rand (1, length (k)), a(k), b(k));
    rand (current_distribution);
  endif
  
  rnd = reshape (rnd, r, c);
  
endfunction