function cdf = emp_cdf (X, a)

  # usage:  emp_cdf (X, a)
  #
  # If X is a vector, emp_cdf returns the empirical distribution of X 
  # evaluated at every single point of a. 
  #
  # If X is a matrix, the columns of X are taken as samples and the
  # empirical distribution of the i-th column of X is evaluated at
  # every point of the i-th column of a. 

  # Written by FL (Friedrich.Leisch@ci.tuwien.ac.at) on Mar 20, 1993  
  # Updated by FL on Sep 6, 1994
  # Updated by FL on Sep 27, 1994
  # Copyright Dept of Probability Theory and Statistics, TU Wien

  if (nargin != 2)
    error ("usage:  emp_cdf (X, a)");
  endif

  [xr, xc] = size (X);
  [ar, ac] = size (a);
  
  if (is_vector (X))

    n = max (xr, xc);
    X = reshape (X, n, 1);
    am = ones (n, 1) * reshape (a, 1, ar*ac);
    
    cdf = sum (X * ones (1, ar*ac) <= am) / n;
    cdf = reshape (cdf, ar, ac);
    
  elseif (is_matrix (X))
    
    if ( xc != ac )
      error ("emp_cdf:  X and a must have the same number of columns!");
    endif
    
    cdf = zeros (ar,ac);
    
    for k = 1 : xc
      am = ones (xr, 1) * a(:, k)';
      cdf(:, k) = sum (X(:, k) * ones (1, ar) <= am)' / xr;
    endfor
    
  else
    error ("emp_cdf:  X must be a vector or matrix");
  
  endif
  
endfunction


















