function inv = emp_inv (x, p, opt)
  
  # usage:  emp_inv (x, p [, opt])
  #
  # If x is a vector, return the p-quantiles of x.
  # If x is a matrix, return the p-quantiles of each column of x.
  # p may be a vector.
  #
  # With the optional argument opt, one can specify the kind of
  # inverse computed.
  # For opt equal to "s", the "symmetric" inverse (so that the
  # 1/2-quantile always equals the median) is returned.
  # For opt equal to "r", the right-continuous inverse is used.  
  # If opt is not specified, the symmetric inverse is used.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/07/31
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/10/15
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if ((nargin < 2) || (nargin > 3))
    error ("usage:  emp_inv (x, p [, opt])");
  endif
  
  if (min (size (p)) != 1 )
    error ("emp_inv:  p has to be a scalar or a vector");
  endif
  if (any ((p < 0) | (p > 1)) )
    error ("emp_inv:  all entries of p have to be in [0, 1]");
  endif
  p   = reshape (p, length (p), 1);
  
  if (max (size (x)) <= 1)
    error ("emp_inv:  x has to be a vector or a matrix");
  endif
  
  if (is_vector (x))
    x = reshape (x, length (x), 1);
  endif
  [nr, nc] = size (x);
  x   = sort (x);
  
  if (nargin == 2)
    opt = "s";
  endif
  
  if (strcmp (opt, "s"))
    k   = p * nr + 1/2;
    f   = floor (k);
    t   = (k - f) * ones (1, nc);
    x   = [x(1, :); x; x(nr, :)];
    inv = (1 - t) .* x(f + 1, :) + t .* x(f + 2, :);
  elseif (strcmp (opt, "r"))
    p   = max (p, ones (length (p), 1) / (2 * nr));
    inv = x (ceil (p * nr), :);
  else
    error (sprintf ("emp_inv:  option %s not recognized", opt));
  endif

endfunction


