function cdf = f_cdf (x, m, n)
  
  # usage:   f_cdf (x, m, n)
  #
  # For each element of x, compute the CDF at x of the F distribution
  # with m and n degrees of freedom, i.e., PROB( F(m,n) <= x ).

  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on Jun 10, 1993
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on Aug 13, 1994
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on Oct 26, 1994
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if (nargin <> 3)
    error ("usage:  f_cdf (x, m, n)");
  endif
  
  [retval, x, m, n] = common_size (x, m, n);
  if (retval > 0)
    error ("f_cdf:  x, m and n must be of common size or scalar");
  endif
  
  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  m = reshape (m, 1, s);
  n = reshape (n, 1, s);
  cdf = zeros (1, s);

  k = find (!(m > 0) | !(n > 0) | isnan (x));
  if any (k)
    cdf(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((x == Inf) & (m > 0) & (n > 0));
  if any (k)
    cdf(k) = ones (1, length (k));
  endif
  
  k = find ((x > 0) & (x < Inf) & (m > 0) & (n > 0));
  if any (k)
    cdf(k) = 1 - betai (n(k) / 2, m(k) / 2, ...
	1 ./ (1 + m(k) .* x(k) ./ n(k)));
  endif
  
  # should we really only allow for positive integer m, n?
  k = find ((m != round (m)) | (n != round (n)));
  if any (k)
    fprintf (stderr, ...
        "WARNING:  m and n should be positive integers\n");
    cdf(k) = NaN * ones (1, length (k));
  endif
  
  cdf = reshape (cdf, r, c);
  
endfunction
