function pdf = f_pdf (x, m, n)
  
  # usage:   f_pdf (x, m, n)
  #
  # For each element of x, compute the probability density function
  # (PDF) at x of the F distribution with m and n degrees of freedom.

  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on Jun 10, 1993
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on Aug 13, 1993
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on Oct 26, 1994
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if (nargin <> 3)
    error ("usage:  f_pdf (x, m, n).");
  endif
  
  [retval, x, m, n] = common_size (x, m, n);
  if (retval > 0)
    error ("f_pdf:  x, m and n must be of common size or scalar");
  endif
  
  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  m = reshape (m, 1, s);
  n = reshape (n, 1, s);
  pdf = zeros (1, s);
  
  k = find (isnan (x) | !(m > 0) | !(n > 0));
  if any (k)
    pdf(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((x > 0) & (x < Inf) & (m > 0) & (n > 0));
  if any (k)
    tmp = m(k) .* x(k) ./ n(k);
    pdf(k) = exp ((m(k) / 2 - 1) .* log (tmp) ...
	- ((m(k) + n(k)) / 2) .* log (1 + tmp)) ...
	.* (m(k) ./ n(k)) ./ beta (m(k) / 2, n(k) / 2);
  endif

  # should we really only allow for positive integer m, n?
  k = find ((m != round (m)) | (n != round (n)));
  if any (k)
    fprintf (stderr, ...
	"WARNING:  m and n should be positive integers\n");
    pdf(k) = NaN * ones (1, length (k));
  endif

  pdf = reshape (pdf, r, c);
  
endfunction
