function rnd = f_rnd (m, n, r, c)
  #
  # usage:  f_rnd (m, n [, r, c])
  #
  # f_rnd (m, n) returns a matrix of random samples from the F
  # distribution with m and n degrees of freedom.  The size of the
  # matrix is the common size of m and n.  
  #
  # f_rnd (m, n, r, c) returns an r by c matrix of random samples from
  # the F distribution with m and n degrees of freedom.  Both m and n
  # must be scalar or of size r by c.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/11/01
  # Copyright Dept of Probability Theory and Statistics TU Wien
  
  if (nargin == 4)
    if !(is_scalar (r) && (r > 0) && (r == round (r)))
      error ("f_rnd:  r must be a positive integer");
    endif
    if !(is_scalar (c) && (c > 0) && (c == round (c)))
      error ("f_rnd:  c must be a positive integer");
    endif
    [retval, m, n] = common_size (m, n, zeros (r, c));
    if (retval > 0)
      error (["f_rnd:  ", ...
	  "m and n must be scalar or of size ", ...
	  sprintf ("%d by %d", r, c)]);
    endif
  elseif (nargin == 2)
    [retval, m, n] = common_size (m, n);
    if (retval > 0)
      error (["f_rnd:  ", ...
	  "m and n must be of common size or scalar"]);
    endif
  else
    error ("usage:  f_rnd (m, n [, r, c])");
  endif

  [r, c] = size (m);
  s = r * c;
  m = reshape (m, 1, s);
  n = reshape (n, 1, s);
  rnd = zeros (1, s);
  
  k = find (!(m > 0) | !(m < Inf) | !(m == round (m)) ...
      | !(n > 0) | !(n < Inf) | !(n == round (n)));
  if any (k)
    rnd(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((m > 0) & (m < Inf) & (m == round (m)) ...
      & (n > 0) & (n < Inf) & (n == round (n)));
  if any (k)
    current_distribution = rand ("dist");
    rand ("uniform");
    rnd(k) = f_inv (rand (1, length (k)), m(k), n(k));
    rand (current_distribution);
  endif
  
  rnd = reshape (rnd, r, c);
  
endfunction