function [pval, f, df_num, df_den] = f_test (x, y, alt)
  
  # usage:  [pval, f, df_num, df_den] = f_test (x, y [, alt])
  #
  # For two samples x and y from normal distributions with unknown 
  # means and unknown equal variances, perform an F-test of the null
  # hypothesis of equal variances.
  # Under the null, the test statistic f follows an F-distribution
  # with df_num and df_den degrees of freedom.
  #
  # With the optional argument string alt, the alternative of interest
  # can be selected.  
  # If alt is "<>" or "!=", the null is tested against the two-sided
  # alternative mean(x) != mean(y).
  # If alt is ">", the one-sided alternative mean(x) > mean(y) is used,
  # similarly for "<".  
  # The default is the two-sided case.
  #
  # pval is the p-value of the test.
  #  
  # If no output argument is given, the p-value of the test is
  # displayed.

  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 94/08/13
  # Last updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 94/09/14
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if ((nargin < 2) || (nargin > 3))
    error (["usage:  [pval, f, df_num, df_den] = ", ...
	"f_test (x, y [, alt])"]);
  endif
    
  if !(is_vector (x) && is_vector (y))
    error ("f_test:  both x and y must be vectors");
  endif

  df_num = length (x) - 1;
  df_den = length (y) - 1;
  f      = var (x) / var (y);
  cdf    = f_cdf (f, df_num, df_den);
  
  if (nargin == 2)
    alt  = "!=";
  endif
    
  if !isstr (alt)
    error ("f_test:  alt must be a string");
  endif
  if (strcmp (alt, "<>") || strcmp (alt, "!="))
    pval = 2 * min (cdf, 1 - cdf);
  elseif strcmp (alt, ">")
    pval = 1 - cdf;
  elseif strcmp (alt, "<")
    pval = cdf;
  else
    error (sprintf ("f_test:  option %s not recognized", alt));
  endif
  
  if (nargout == 0)
    printf ("pval:  %g\n", pval);
  endif

endfunction
