function inv = gamma_inv (x, a, b)
  
  # usage:  gamma_inv (x, a, b)
  #
  # For each component of x, compute the quantile (the inverse of the
  # CDF) at x of the Gamma distribution with parameters a and b.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/09/27
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/10/22
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if (nargin <> 3)
    error("usage:  gamma_inv (x, a, b)");
  endif
  
  [retval, x, a, b] = common_size (x, a, b);
  if (retval > 0)
    error ("gamma_inv:  x, a and b must be of common size or scalars");
  endif

  [r, c] = size (x);
  s   = r * c;
  x   = reshape (x, s, 1);
  a   = reshape (a, s, 1);
  b   = reshape (b, s, 1);
  inv = zeros (s, 1);
  
  k = find ((x < 0) | (x > 1) | isnan (x) | !(a > 0) | !(b > 0));
  if any (k)
    inv (k) = NaN * ones (length (k), 1);
  endif
  
  k = find ((x == 1) & (a > 0) & (b > 0));
  if any (k)
    inv (k) = Inf * ones (length (k), 1);
  endif
  
  k = find ((x > 0) & (x < 1) & (a > 0) & (b > 0));
  if any (k)
    a = a (k);
    b = b (k);
    x = x (k);
    y = a ./ b;
    l = find (x < eps);
    if any (l)
      y(l) = sqrt (eps) * ones (length (l), 1);
    endif

    y_old = y;
    for i = 1 : 100
      h     = (gamma_cdf (y_old, a, b) - x) ./ gamma_pdf (y_old, a, b);
      y_new = y_old - h;
      ind   = find (y_new <= eps);
      if (any (ind))
	y_new (ind) = y_old (ind) / 10;
	h = y_old - y_new;
      endif
      if (max (abs (h)) < sqrt (eps))
	break;
      endif
      y_old = y_new;
    endfor
    
    inv (k) = y_new;
  endif

  inv = reshape (inv, r, c);
  
endfunction
