function rnd = geometric_rnd (p, r, c)
  #
  # usage:  geometric_rnd (p [, r, c])
  #
  # geometric_rnd (p) returns a matrix of random samples from the
  # geometric distribution with parameter p.  The size of the matrix
  # is the size of p.
  #
  # geometric_rnd (p, r, c) returns an r by c matrix of random samples
  # from the geometric distribution with parameter p, which must be a
  # scalar or of size r by c.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/11/01
  # Copyright Dept of Probability Theory and Statistics TU Wien
  
  if (nargin == 3)
    if !(is_scalar (r) && (r > 0) && (r == round (r)))
      error ("geometric_rnd:  r must be a positive integer");
    endif
    if !(is_scalar (c) && (c > 0) && (c == round (c)))
      error ("geometric_rnd:  c must be a positive integer");
    endif
    [retval, p] = common_size (p, zeros (r, c));
    if (retval > 0)
      error (["geometric_rnd:  ", ...
	  "p must be scalar or of size ", ... 
	  sprintf ("%d by %d", r, c)]);
    endif
  else
    if !(nargin == 1)
      error ("usage:  geometric_rnd (p [, r, c])");
    endif
  endif
  
  [r, c] = size (p);
  s = r * c;
  p = reshape (p, 1, s);
  rnd = zeros (1, s);
  
  k = find (!(p >= 0) | !(p <= 1));
  if any (k)
    rnd(k) = NaN * ones (1, length (k));
  endif
  
  k = find (p == 0);
  if any (k)
    rnd(k) = Inf * ones (1, length (k));
  endif
  
  k = find ((p > 0) & (p < 1));
  if any (k)
    current_distribution = rand ("dist");
    rand ("uniform");
    rnd(k) = floor (log (rand (1, length (k))) ./ log (1 - p(k)));
    rand (current_distribution);
  endif
  
  rnd = reshape (rnd, r, c);
  
endfunction
