function tau = kendall (x, y)
  
# usage:  kendall (x [, y])
#
# Computes Kendall's tau for each of the variables specified by the
# input arguments. 
# For matrices, each row is an observation and each column a variable;
# vectors are always observations and may be row or column vectors.
# kendall (x) is equivalent to kendall (x, x).
#
# For two data vectors x, y of common length n, Kendall's tau is the
# correlation of the signs of all rank differences of x and y;  i.e.,
# if both x and y have distinct entries, then
#    \tau = \frac{1}{n(n-1)} \sum_{i,j} SIGN(q_i-q_j) SIGN(r_i-r_j),
# where the q_i and r_i are the ranks of x and y, respectively.  
# If x and y are drawn from independent distributions, Kendall's tau
# is asymptotically normal with mean 0 and variance (2 * (2n+5)) / 
# (9 * n * (n-1)).

# Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1995/01/25
# Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1995/02/05
# Copyright Dept of Probability Theory and Statistics TU Wien
  
  if ((nargin < 1) || (nargin > 2))
    error ("usage:  kendall (x [, y])");
  endif

  if (rows (x) == 1)
    x = x';
  endif
  [n, c] = size (x);

  if (nargin == 2)
    if (rows (y) == 1)
      y = y';
    endif
    if (rows (y) <> n)
      error (["kendall:  ", ...
	      "x and y must have the same number of observations"]);
    else
      x = [x, y];
    endif
  endif
  
  r   = ranks (x);
  m   = sign (kron (r, ones (n, 1)) - kron (ones (n, 1), r));
  tau = corrcoef (m);
  
  if (nargin == 2)
    tau = tau (1 : c, (c + 1) : columns (x));
  endif
  
endfunction