function cdf = kolmogorov_smirnov_cdf (x, tol)
  
  # usage:  kolmogorov_smirnov_cdf (x [, tol])
  #
  # Returns the CDF at x of the Kolmogorov-Smirnov distribution, i.e.
  # Q(x) = sum_{k=-\infty}^\infty (-1)^k exp(-2 k^2 x^2), x > 0.
  #
  # The optional tol specifies the precision up to which the series
  # should be evaluated;  the default is tol = eps.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on Jul 29, 1994
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if (nargin < 1 || nargin > 2)
    error("usage:  kolmogorov_smirnov_cdf (x [, tol])");
  endif

  if (nargin == 1)
    tol = eps;
  else 
    if (!is_scalar (tol) || !(tol > 0))
      error(["kolmogorov_smirnov_cdf:  ", ...
	  "tol has to be a positive scalar."]);
    endif
  endif

  [nr, nc] = size(x);
  if (min (nr, nc) == 0)
    error("kolmogorov_smirnov_cdf:  x must not be empty.");
  endif

  n   = nr * nc;
  x   = reshape (x, 1, n);
  cdf = zeros (1, n);
  ind = find (x > 0);
  if (length (ind) > 0)
    y   = x(ind);
    K   = ceil( sqrt( - log (tol) / 2 ) / min (y) );
    k   = (1:K)';
    A   = exp( - 2 * k.^2 * y.^2 );
    odd = find (rem (k, 2) == 1);
    A(odd, :) = -A(odd, :);
    cdf(ind) = 1 + 2 * sum (A);
  endif

  cdf = reshape (cdf, nr, nc);
  
endfunction
  
