function cdf = lognormal_cdf (x, a, v)

  # usage:   lognormal_cdf (x [, a, v])
  #
  # For each element of x, compute the cumulative distribution
  # function (CDF) at x of the lognormal distribution with parameters
  # a and v. 
  # If a random variable follows this distribution, its logarithm is
  # normally distributed with mean log (a) and variance v.
  # 
  # Default values are m = 1, v = 1.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on Oct 30, 1994
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if !((nargin == 1) || (nargin == 3))
    error ("usage:  lognormal_cdf (x [, a, v])");
  endif

  if (nargin == 1)
    a = 1;
    v = 1;
  endif

  # The following "straightforward" implementation unfortunately does
  # not work (because exp (Inf) -> NaN etc):
  # cdf = normal_cdf (log (x), log (a), v);
  # Hence ...

  [retval, x, a, v] = common_size (x, a, v);
  if (retval > 0)
    error (["lognormal_cdf:  ", ...
        "x, m and v must be of common size or scalars"]);
  endif

  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  a = reshape (a, 1, s);
  v = reshape (v, 1, s);
  cdf = zeros (1, s);
  
  k = find (isnan (x) | !(a > 0) | !(a < Inf) ...
      | !(v > 0) | !(v < Inf));
  if any (k)
    cdf(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((x == Inf) & (a > 0) & (a < Inf) & (v > 0) & (v < Inf));
  if any (k)
    cdf(k) = ones (1, length (k));
  endif
  
  k = find ((x > 0) & (x < Inf) & (a > 0) & (a < Inf) ...
      & (v > 0) & (v < Inf));
  if any (k)
    cdf(k) = stdnormal_cdf ((log (x(k)) - log (a(k))) ./ sqrt (v(k)));
  endif
  
  cdf = reshape (cdf, r, c);
  
endfunction
