function pdf = lognormal_pdf (x, a, v)

  # usage:   lognormal_pdf (x [, a, v])
  #
  # For each element of x, compute the probability density function
  # (PDF) at x of the lognormal distribution with parameters a and v.
  # If a random variable follows this distribution, its logarithm is
  # normally distributed with mean log (a) and variance v.
  # 
  # Default values are m = 1, v = 1.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on Oct 30, 1994
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if !((nargin == 1) || (nargin == 3))
    error ("usage:  lognormal_pdf (x [, a, v])");
  endif

  if (nargin == 1)
    a = 1;
    v = 1;
  endif

  # The following "straightforward" implementation unfortunately does
  # not work for the special cases (Inf, ...)
  # pdf = (x > 0) ./ x .* normal_pdf (log (x), log (a), v);
  # Hence ...
  
  [retval, x, a, v] = common_size (x, a, v);
  if (retval > 0)
    error (["lognormal_pdf:  ", ...
        "x, m and v must be of common size or scalars"]);
  endif
  
  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  a = reshape (a, 1, s);
  v = reshape (v, 1, s);
  pdf = zeros (1, s);

  k = find (isnan (x) | !(a > 0) | !(a < Inf) ...
      | !(v > 0) | !(v < Inf)); 
  if any (k)
    pdf(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((x > 0) & (x < Inf) & (a > 0) & (a < Inf) ...
      & (v > 0) & (v < Inf));
  if any (k)
    pdf(k) = normal_pdf (log (x(k)), log (a(k)), v(k)) ./ x(k);
  endif

  pdf = reshape (pdf, r, c);
  
endfunction
